"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    Object.defineProperty(o, k2, { enumerable: true, get: function() { return m[k]; } });
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.HugoMarkdownDocumenter = void 0;
const path = __importStar(require("path"));
const node_core_library_1 = require("@rushstack/node-core-library");
const tsdoc_1 = require("@microsoft/tsdoc");
const api_extractor_model_1 = require("@microsoft/api-extractor-model");
const CustomDocNodeKind_1 = require("../../nodes/CustomDocNodeKind");
const DocHeading_1 = require("../../nodes/DocHeading");
const DocTable_1 = require("../../nodes/DocTable");
const DocEmphasisSpan_1 = require("../../nodes/DocEmphasisSpan");
const DocTableRow_1 = require("../../nodes/DocTableRow");
const DocTableCell_1 = require("../../nodes/DocTableCell");
const Utilities_1 = require("../../utils/Utilities");
const GrafanaMarkdownEmitter_1 = require("../../markdown/GrafanaMarkdownEmitter");
const DocFrontMatter_1 = require("../../nodes/grafana/DocFrontMatter");
const PageTitleAppender_1 = require("./PageTitleAppender");
const HeadingAppender_1 = require("./HeadingAppender");
const WarningAppender_1 = require("./WarningAppender");
const ImportAppender_1 = require("./ImportAppender");
const SignatureAppender_1 = require("./SignatureAppender");
const SummaryAppender_1 = require("./SummaryAppender");
/**
 * Renders API documentation in the Markdown file format.
 * For more info:  https://en.wikipedia.org/wiki/Markdown
 */
class HugoMarkdownDocumenter {
    constructor(parameters) {
        const { configuration } = CustomDocNodeKind_1.CustomDocNodes;
        this._apiModel = parameters.model;
        this._outputFolder = parameters.output;
        this._generateDraft = parameters.draft;
        this._tsdocConfiguration = configuration;
        this._markdownEmitter = new GrafanaMarkdownEmitter_1.GrafanaMarkdownEmitter(this._apiModel);
        this._pageTitleAppender = new PageTitleAppender_1.PageTitleAppender(configuration);
        this._headingAppender = new HeadingAppender_1.HeadingAppender(configuration);
        this._warningAppender = new WarningAppender_1.WarningAppender(configuration);
        this._importAppender = new ImportAppender_1.ImportAppender(configuration);
        this._signatureAppender = new SignatureAppender_1.SignatureAppender(configuration);
        this._summaryAppender = new SummaryAppender_1.SummaryAppender();
    }
    generateFiles() {
        this._deleteOldOutputFiles(this._outputFolder);
        this._writeApiItemPage(this._apiModel);
        console.log('Successfully generated markdown files');
    }
    _writeApiItemPage(apiItem) {
        const configuration = this._tsdocConfiguration;
        const draft = this._generateDraft;
        const output = new tsdoc_1.DocSection({ configuration });
        output.appendNode(new DocFrontMatter_1.DocFrontMatter({ configuration, apiItem, draft }));
        this._pageTitleAppender.append(output, apiItem);
        this._writeApiItemContent(output, apiItem, true);
        const filename = path.join(this._outputFolder, this._getFilenameForApiItem(apiItem));
        const stringBuilder = new tsdoc_1.StringBuilder();
        this._markdownEmitter.emit(stringBuilder, output, {
            contextApiItem: apiItem,
            onGetFilenameForApiItem: (apiItemForFilename) => {
                return this._getLinkFilenameForApiItem(apiItemForFilename);
            }
        });
        node_core_library_1.FileSystem.writeFile(filename, stringBuilder.toString(), {
            ensureFolderExists: true,
            convertLineEndings: "\r\n" /* CrLf */
        });
    }
    _writeApiItemContent(output, apiItem, appendImport = false) {
        this._warningAppender.append(output, apiItem);
        this._headingAppender.append(output, apiItem);
        this._summaryAppender.append(output, apiItem);
        this._signatureAppender.append(output, apiItem);
        if (appendImport) {
            this._importAppender.append(output, apiItem);
        }
        let appendRemarks = true;
        switch (apiItem.kind) {
            case "Class" /* Class */:
            case "Interface" /* Interface */:
            case "Namespace" /* Namespace */:
            case "Package" /* Package */:
                this._writeRemarksSection(output, apiItem);
                appendRemarks = false;
                break;
        }
        switch (apiItem.kind) {
            case "Class" /* Class */:
                this._writeClassTables(output, apiItem);
                break;
            case "Enum" /* Enum */:
                this._writeEnumTables(output, apiItem);
                break;
            case "Interface" /* Interface */:
                this._writeInterfaceTables(output, apiItem);
                break;
            case "Constructor" /* Constructor */:
            case "ConstructSignature" /* ConstructSignature */:
            case "Method" /* Method */:
            case "MethodSignature" /* MethodSignature */:
            case "Function" /* Function */:
                this._writeParameterTables(output, apiItem);
                this._writeThrowsSection(output, apiItem);
                break;
            case "Namespace" /* Namespace */:
                this._writePackageOrNamespaceTables(output, apiItem);
                break;
            case "Model" /* Model */:
                this._writeModelTable(output, apiItem);
                break;
            case "Package" /* Package */:
                this._writePackageOrNamespaceTables(output, apiItem);
                break;
            case "Property" /* Property */:
            case "PropertySignature" /* PropertySignature */:
                break;
            case "TypeAlias" /* TypeAlias */:
                break;
            case "Variable" /* Variable */:
                break;
            default:
                throw new Error('Unsupported API item kind: ' + apiItem.kind);
        }
        if (appendRemarks) {
            this._writeRemarksSection(output, apiItem);
        }
    }
    _writeRemarksSection(output, apiItem) {
        if (apiItem instanceof api_extractor_model_1.ApiDocumentedItem) {
            const tsdocComment = apiItem.tsdocComment;
            if (tsdocComment) {
                // Write the @remarks block
                if (tsdocComment.remarksBlock) {
                    output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Remarks' }));
                    this._appendSection(output, tsdocComment.remarksBlock.content);
                }
                // Write the @example blocks
                const exampleBlocks = tsdocComment.customBlocks.filter((x) => x.blockTag.tagNameWithUpperCase === tsdoc_1.StandardTags.example.tagNameWithUpperCase);
                let exampleNumber = 1;
                for (const exampleBlock of exampleBlocks) {
                    const heading = exampleBlocks.length > 1 ? `Example ${exampleNumber}` : 'Example';
                    output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: heading }));
                    this._appendSection(output, exampleBlock.content);
                    ++exampleNumber;
                }
            }
        }
    }
    _writeThrowsSection(output, apiItem) {
        if (apiItem instanceof api_extractor_model_1.ApiDocumentedItem) {
            const tsdocComment = apiItem.tsdocComment;
            if (tsdocComment) {
                // Write the @throws blocks
                const throwsBlocks = tsdocComment.customBlocks.filter((x) => x.blockTag.tagNameWithUpperCase === tsdoc_1.StandardTags.throws.tagNameWithUpperCase);
                if (throwsBlocks.length > 0) {
                    const heading = 'Exceptions';
                    output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: heading }));
                    for (const throwsBlock of throwsBlocks) {
                        this._appendSection(output, throwsBlock.content);
                    }
                }
            }
        }
    }
    /**
     * GENERATE PAGE: MODEL
     */
    _writeModelTable(output, apiModel) {
        const configuration = this._tsdocConfiguration;
        const packagesTable = new DocTable_1.DocTable({ configuration, headerTitles: ['Package', 'Description'] });
        for (const apiMember of apiModel.members) {
            const row = new DocTableRow_1.DocTableRow({ configuration }, [
                this._createTitleCell(apiMember),
                this._createDescriptionCell(apiMember)
            ]);
            switch (apiMember.kind) {
                case "Package" /* Package */:
                    packagesTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
            }
        }
        if (packagesTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration, title: 'Packages', level: 2 }));
            output.appendNode(packagesTable);
        }
    }
    /**
     * GENERATE PAGE: PACKAGE or NAMESPACE
     */
    _writePackageOrNamespaceTables(output, apiContainer) {
        const configuration = this._tsdocConfiguration;
        const classesTable = new DocTable_1.DocTable({ configuration, headerTitles: ['Class', 'Description'] });
        const enumerationsTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Enumeration', 'Description']
        });
        const functionsTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Function', 'Description']
        });
        const interfacesTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Interface', 'Description']
        });
        const namespacesTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Namespace', 'Description']
        });
        const variablesTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Variable', 'Description']
        });
        const typeAliasesTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Type Alias', 'Description']
        });
        const apiMembers = apiContainer.kind === "Package" /* Package */
            ? apiContainer.entryPoints[0].members
            : apiContainer.members;
        for (const apiMember of apiMembers) {
            const row = new DocTableRow_1.DocTableRow({ configuration }, [
                this._createTitleCell(apiMember),
                this._createDescriptionCell(apiMember)
            ]);
            switch (apiMember.kind) {
                case "Class" /* Class */:
                    classesTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
                case "Enum" /* Enum */:
                    enumerationsTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
                case "Interface" /* Interface */:
                    interfacesTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
                case "Namespace" /* Namespace */:
                    namespacesTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
                case "Function" /* Function */:
                    functionsTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
                case "TypeAlias" /* TypeAlias */:
                    typeAliasesTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
                case "Variable" /* Variable */:
                    variablesTable.addRow(row);
                    this._writeApiItemPage(apiMember);
                    break;
            }
        }
        if (classesTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Classes' }));
            output.appendNode(classesTable);
        }
        if (enumerationsTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Enumerations' }));
            output.appendNode(enumerationsTable);
        }
        if (functionsTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Functions' }));
            output.appendNode(functionsTable);
        }
        if (interfacesTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Interfaces' }));
            output.appendNode(interfacesTable);
        }
        if (namespacesTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Namespaces' }));
            output.appendNode(namespacesTable);
        }
        if (variablesTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Variables' }));
            output.appendNode(variablesTable);
        }
        if (typeAliasesTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Type Aliases' }));
            output.appendNode(typeAliasesTable);
        }
    }
    /**
     * GENERATE PAGE: CLASS
     */
    _writeClassTables(output, apiClass) {
        const configuration = this._tsdocConfiguration;
        const eventsTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Property', 'Modifiers', 'Type', 'Description']
        });
        const constructorsTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Constructor', 'Modifiers', 'Description']
        });
        const constructorsSection = new tsdoc_1.DocSection({ configuration });
        const propertiesTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Property', 'Modifiers', 'Type', 'Description']
        });
        const propertiesSection = new tsdoc_1.DocSection({ configuration });
        const methodsTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Method', 'Modifiers', 'Description']
        });
        const methodsSection = new tsdoc_1.DocSection({ configuration });
        for (const apiMember of apiClass.members) {
            switch (apiMember.kind) {
                case "Constructor" /* Constructor */: {
                    constructorsTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                        this._createTitleCell(apiMember),
                        this._createModifiersCell(apiMember),
                        this._createDescriptionCell(apiMember)
                    ]));
                    this._writeApiItemContent(constructorsSection, apiMember);
                    break;
                }
                case "Method" /* Method */: {
                    methodsTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                        this._createTitleCell(apiMember),
                        this._createModifiersCell(apiMember),
                        this._createDescriptionCell(apiMember)
                    ]));
                    this._writeApiItemContent(methodsSection, apiMember);
                    break;
                }
                case "Property" /* Property */: {
                    if (apiMember.isEventProperty) {
                        eventsTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                            this._createTitleCell(apiMember),
                            this._createModifiersCell(apiMember),
                            this._createPropertyTypeCell(apiMember),
                            this._createDescriptionCell(apiMember)
                        ]));
                    }
                    else {
                        propertiesTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                            this._createTitleCell(apiMember),
                            this._createModifiersCell(apiMember),
                            this._createPropertyTypeCell(apiMember),
                            this._createDescriptionCell(apiMember)
                        ]));
                    }
                    this._writeApiItemContent(propertiesSection, apiMember);
                    break;
                }
            }
        }
        if (eventsTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Events'));
            output.appendNode(eventsTable);
        }
        if (constructorsTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Constructors'));
            output.appendNode(constructorsTable);
        }
        if (propertiesTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Properties'));
            output.appendNode(propertiesTable);
        }
        if (methodsTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Methods'));
            output.appendNode(methodsTable);
        }
        this._appendSection(output, constructorsSection);
        this._appendSection(output, propertiesSection);
        this._appendSection(output, methodsSection);
    }
    /**
     * GENERATE PAGE: ENUM
     */
    _writeEnumTables(output, apiEnum) {
        const configuration = this._tsdocConfiguration;
        const enumMembersTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Member', 'Value', 'Description']
        });
        for (const apiEnumMember of apiEnum.members) {
            enumMembersTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                new DocTableCell_1.DocTableCell({ configuration }, [
                    new tsdoc_1.DocParagraph({ configuration }, [
                        new tsdoc_1.DocPlainText({ configuration, text: Utilities_1.Utilities.getConciseSignature(apiEnumMember) })
                    ])
                ]),
                new DocTableCell_1.DocTableCell({ configuration }, [
                    new tsdoc_1.DocParagraph({ configuration }, [
                        new tsdoc_1.DocCodeSpan({ configuration, code: apiEnumMember.initializerExcerpt.text })
                    ])
                ]),
                this._createDescriptionCell(apiEnumMember)
            ]));
        }
        if (enumMembersTable.rows.length > 0) {
            output.appendNode(new DocHeading_1.DocHeading({ configuration: this._tsdocConfiguration, title: 'Enumeration Members' }));
            output.appendNode(enumMembersTable);
        }
    }
    /**
     * GENERATE PAGE: INTERFACE
     */
    _writeInterfaceTables(output, apiClass) {
        const configuration = this._tsdocConfiguration;
        const eventsTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Property', 'Type', 'Description']
        });
        const propertiesTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Property', 'Type', 'Description']
        });
        const properitesSection = new tsdoc_1.DocSection({ configuration });
        const methodsTable = new DocTable_1.DocTable({ configuration, headerTitles: ['Method', 'Description'] });
        const methodsSection = new tsdoc_1.DocSection({ configuration });
        for (const apiMember of apiClass.members) {
            switch (apiMember.kind) {
                case "ConstructSignature" /* ConstructSignature */:
                case "MethodSignature" /* MethodSignature */: {
                    methodsTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                        this._createTitleCell(apiMember),
                        this._createDescriptionCell(apiMember)
                    ]));
                    this._writeApiItemContent(methodsSection, apiMember);
                    break;
                }
                case "PropertySignature" /* PropertySignature */: {
                    if (apiMember.isEventProperty) {
                        eventsTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                            this._createTitleCell(apiMember),
                            this._createPropertyTypeCell(apiMember),
                            this._createDescriptionCell(apiMember)
                        ]));
                    }
                    else {
                        propertiesTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                            this._createTitleCell(apiMember),
                            this._createPropertyTypeCell(apiMember),
                            this._createDescriptionCell(apiMember)
                        ]));
                    }
                    this._writeApiItemContent(properitesSection, apiMember);
                    break;
                }
            }
        }
        if (eventsTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Events'));
            output.appendNode(eventsTable);
        }
        if (propertiesTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Properties'));
            output.appendNode(propertiesTable);
        }
        if (methodsTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Methods'));
            output.appendNode(methodsTable);
        }
        this._appendSection(output, properitesSection);
        this._appendSection(output, methodsSection);
    }
    /**
     * GENERATE PAGE: FUNCTION-LIKE
     */
    _writeParameterTables(output, apiParameterListMixin) {
        const configuration = this._tsdocConfiguration;
        const parametersTable = new DocTable_1.DocTable({
            configuration,
            headerTitles: ['Parameter', 'Type', 'Description']
        });
        for (const apiParameter of apiParameterListMixin.parameters) {
            const parameterDescription = new tsdoc_1.DocSection({ configuration });
            if (apiParameter.tsdocParamBlock) {
                this._appendSection(parameterDescription, apiParameter.tsdocParamBlock.content);
            }
            parametersTable.addRow(new DocTableRow_1.DocTableRow({ configuration }, [
                new DocTableCell_1.DocTableCell({ configuration }, [
                    new tsdoc_1.DocParagraph({ configuration }, [
                        new tsdoc_1.DocPlainText({ configuration, text: apiParameter.name })
                    ])
                ]),
                new DocTableCell_1.DocTableCell({ configuration }, [
                    new tsdoc_1.DocParagraph({ configuration }, [
                        new tsdoc_1.DocCodeSpan({ configuration, code: apiParameter.parameterTypeExcerpt.text })
                    ])
                ]),
                new DocTableCell_1.DocTableCell({ configuration }, parameterDescription.nodes)
            ]));
        }
        if (parametersTable.rows.length > 0) {
            output.appendNode(this._writeBoldText(configuration, 'Parameters'));
            output.appendNode(parametersTable);
        }
        if (api_extractor_model_1.ApiReturnTypeMixin.isBaseClassOf(apiParameterListMixin)) {
            const returnTypeExcerpt = apiParameterListMixin.returnTypeExcerpt;
            output.appendNode(new tsdoc_1.DocParagraph({ configuration }, [
                new DocEmphasisSpan_1.DocEmphasisSpan({ configuration, bold: true }, [
                    new tsdoc_1.DocPlainText({ configuration, text: 'Returns:' })
                ])
            ]));
            output.appendNode(new tsdoc_1.DocParagraph({ configuration }, [
                new tsdoc_1.DocCodeSpan({ configuration, code: returnTypeExcerpt.text.trim() || '(not declared)' })
            ]));
            if (apiParameterListMixin instanceof api_extractor_model_1.ApiDocumentedItem) {
                if (apiParameterListMixin.tsdocComment && apiParameterListMixin.tsdocComment.returnsBlock) {
                    this._appendSection(output, apiParameterListMixin.tsdocComment.returnsBlock.content);
                }
            }
        }
    }
    _createTitleCell(apiItem) {
        const configuration = this._tsdocConfiguration;
        return new DocTableCell_1.DocTableCell({ configuration }, [
            new tsdoc_1.DocParagraph({ configuration }, [
                new tsdoc_1.DocLinkTag({
                    configuration,
                    tagName: '@link',
                    linkText: Utilities_1.Utilities.getGrafanaConciseSignature(apiItem),
                    urlDestination: this._getUrlDestination(apiItem)
                })
            ])
        ]);
    }
    _getUrlDestination(apiItem) {
        switch (apiItem.kind) {
            case "Method" /* Method */:
            case "MethodSignature" /* MethodSignature */: {
                const link = Utilities_1.Utilities.getSafeFilenameForName(apiItem.displayName);
                return `#${link}-method`;
            }
            case "Package" /* Package */: {
                const signature = node_core_library_1.PackageName.getUnscopedName(apiItem.displayName);
                const link = Utilities_1.Utilities.getSafeFilenameForName(signature);
                return `./${link}/`;
            }
            case "Property" /* Property */:
            case "PropertySignature" /* PropertySignature */: {
                const link = Utilities_1.Utilities.getSafeFilenameForName(apiItem.displayName);
                return `#${link}-property`;
            }
            case "ConstructSignature" /* ConstructSignature */:
            case "Constructor" /* Constructor */: {
                const signature = Utilities_1.Utilities.getGrafanaConciseSignature(apiItem);
                const link = Utilities_1.Utilities.getHeaderLinkForName(signature);
                return `#${link}`;
            }
            case "TypeAlias" /* TypeAlias */:
            case "Enum" /* Enum */:
            case "Variable" /* Variable */:
            case "Function" /* Function */:
            case "Interface" /* Interface */:
            case "Namespace" /* Namespace */:
            case "Class" /* Class */: {
                const link = Utilities_1.Utilities.getSafeFilenameForName(apiItem.displayName);
                return `./${link}/`;
            }
            default:
                return `${this._getLinkFilenameForApiItem(apiItem)}`;
        }
    }
    /**
     * This generates a DocTableCell for an ApiItem including the summary section and "(BETA)" annotation.
     *
     * @remarks
     * We mostly assume that the input is an ApiDocumentedItem, but it's easier to perform this as a runtime
     * check than to have each caller perform a type cast.
     */
    _createDescriptionCell(apiItem) {
        const configuration = this._tsdocConfiguration;
        const section = new tsdoc_1.DocSection({ configuration });
        if (api_extractor_model_1.ApiReleaseTagMixin.isBaseClassOf(apiItem)) {
            if (apiItem.releaseTag === api_extractor_model_1.ReleaseTag.Beta) {
                section.appendNodesInParagraph([
                    new DocEmphasisSpan_1.DocEmphasisSpan({ configuration, bold: true, italic: true }, [
                        new tsdoc_1.DocPlainText({ configuration, text: '(BETA)' })
                    ]),
                    new tsdoc_1.DocPlainText({ configuration, text: ' ' })
                ]);
            }
        }
        if (apiItem instanceof api_extractor_model_1.ApiDocumentedItem) {
            if (apiItem.tsdocComment !== undefined) {
                this._appendAndMergeSection(section, apiItem.tsdocComment.summarySection);
            }
        }
        return new DocTableCell_1.DocTableCell({ configuration }, section.nodes);
    }
    _createModifiersCell(apiItem) {
        const configuration = this._tsdocConfiguration;
        const section = new tsdoc_1.DocSection({ configuration });
        if (api_extractor_model_1.ApiStaticMixin.isBaseClassOf(apiItem)) {
            if (apiItem.isStatic) {
                section.appendNodeInParagraph(new tsdoc_1.DocCodeSpan({ configuration, code: 'static' }));
            }
        }
        return new DocTableCell_1.DocTableCell({ configuration }, section.nodes);
    }
    _createPropertyTypeCell(apiItem) {
        const configuration = this._tsdocConfiguration;
        const section = new tsdoc_1.DocSection({ configuration });
        if (apiItem instanceof api_extractor_model_1.ApiPropertyItem) {
            section.appendNodeInParagraph(new tsdoc_1.DocCodeSpan({ configuration, code: apiItem.propertyTypeExcerpt.text }));
        }
        return new DocTableCell_1.DocTableCell({ configuration }, section.nodes);
    }
    _appendSection(output, docSection) {
        for (const node of docSection.nodes) {
            output.appendNode(node);
        }
    }
    _appendAndMergeSection(output, docSection) {
        let firstNode = true;
        for (const node of docSection.nodes) {
            if (firstNode) {
                if (node.kind === "Paragraph" /* Paragraph */) {
                    output.appendNodesInParagraph(node.getChildNodes());
                    firstNode = false;
                    continue;
                }
            }
            firstNode = false;
            output.appendNode(node);
        }
    }
    _writeBoldText(configuration, text) {
        return new tsdoc_1.DocParagraph({ configuration }, [
            new DocEmphasisSpan_1.DocEmphasisSpan({ configuration, bold: true }, [new tsdoc_1.DocPlainText({ configuration, text })])
        ]);
    }
    _getFilenameForApiItem(apiItem, relative = false) {
        if (apiItem.kind === "Model" /* Model */) {
            return '_index.md';
        }
        if (apiItem.kind === "Package" /* Package */) {
            const unscopedName = node_core_library_1.PackageName.getUnscopedName(apiItem.displayName);
            const baseName = Utilities_1.Utilities.getSafeFilenameForName(unscopedName);
            return `${baseName}/_index.md`;
        }
        let baseName = '';
        let suffix = '.md';
        let relativeToRoot = '';
        for (const hierarchyItem of apiItem.getHierarchy()) {
            // For overloaded methods, add a suffix such as "MyClass.myMethod_2".
            let qualifiedName = Utilities_1.Utilities.getSafeFilenameForName(hierarchyItem.displayName);
            if (api_extractor_model_1.ApiParameterListMixin.isBaseClassOf(hierarchyItem)) {
                if (hierarchyItem.overloadIndex > 1) {
                    // Subtract one for compatibility with earlier releases of API Documenter.
                    // (This will get revamped when we fix GitHub issue #1308)
                    qualifiedName += `_${hierarchyItem.overloadIndex - 1}`;
                }
            }
            switch (hierarchyItem.kind) {
                case "Model" /* Model */:
                case "EntryPoint" /* EntryPoint */:
                    break;
                case "Package" /* Package */:
                    baseName = Utilities_1.Utilities.getSafeFilenameForName(node_core_library_1.PackageName.getUnscopedName(hierarchyItem.displayName));
                    break;
                case "Method" /* Method */:
                    baseName += `/#${qualifiedName}-method`;
                    suffix = '';
                    break;
                default:
                    relativeToRoot = `../${relativeToRoot}`;
                    baseName += '/' + qualifiedName;
                    break;
            }
        }
        if (relative) {
            return `${relativeToRoot}${baseName}${suffix}`;
        }
        return `${baseName}${suffix}`;
    }
    _getLinkFilenameForApiItem(apiItem) {
        const fileName = this._getFilenameForApiItem(apiItem, true);
        if (fileName.startsWith('../')) {
            return fileName;
        }
        return './' + fileName;
    }
    _deleteOldOutputFiles(outputFolder) {
        console.log('Deleting old output from ' + outputFolder);
        node_core_library_1.FileSystem.ensureEmptyFolder(outputFolder);
    }
}
exports.HugoMarkdownDocumenter = HugoMarkdownDocumenter;
//# sourceMappingURL=HugoMarkdownDocumenter.js.map