"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiTypeAlias = void 0;
const DeclarationReference_1 = require("@microsoft/tsdoc/lib/beta/DeclarationReference");
const ApiDeclaredItem_1 = require("../items/ApiDeclaredItem");
const ApiReleaseTagMixin_1 = require("../mixins/ApiReleaseTagMixin");
const ApiNameMixin_1 = require("../mixins/ApiNameMixin");
const ApiTypeParameterListMixin_1 = require("../mixins/ApiTypeParameterListMixin");
/**
 * Represents a TypeScript type alias declaration.
 *
 * @remarks
 *
 * This is part of the {@link ApiModel} hierarchy of classes, which are serializable representations of
 * API declarations.
 *
 * `ApiTypeAlias` represents a definition such as one of these examples:
 *
 * ```ts
 * // A union type:
 * export type Shape = Square | Triangle | Circle;
 *
 * // A generic type alias:
 * export type BoxedValue<T> = { value: T };
 *
 * export type BoxedArray<T> = { array: T[] };
 *
 * // A conditional type alias:
 * export type Boxed<T> = T extends any[] ? BoxedArray<T[number]> : BoxedValue<T>;
 *
 * ```
 *
 * @public
 */
class ApiTypeAlias extends ApiTypeParameterListMixin_1.ApiTypeParameterListMixin(ApiNameMixin_1.ApiNameMixin(ApiReleaseTagMixin_1.ApiReleaseTagMixin(ApiDeclaredItem_1.ApiDeclaredItem))) {
    constructor(options) {
        super(options);
        this.typeExcerpt = this.buildExcerpt(options.typeTokenRange);
    }
    /** @override */
    static onDeserializeInto(options, context, jsonObject) {
        super.onDeserializeInto(options, context, jsonObject);
        options.typeTokenRange = jsonObject.typeTokenRange;
    }
    static getContainerKey(name) {
        return `${name}|${"TypeAlias" /* TypeAlias */}`;
    }
    /** @override */
    get kind() {
        return "TypeAlias" /* TypeAlias */;
    }
    /** @override */
    get containerKey() {
        return ApiTypeAlias.getContainerKey(this.name);
    }
    /** @override */
    serializeInto(jsonObject) {
        super.serializeInto(jsonObject);
        jsonObject.typeTokenRange = this.typeExcerpt.tokenRange;
    }
    /** @beta @override */
    buildCanonicalReference() {
        const nameComponent = DeclarationReference_1.DeclarationReference.parseComponent(this.name);
        return (this.parent ? this.parent.canonicalReference : DeclarationReference_1.DeclarationReference.empty())
            .addNavigationStep("." /* Exports */, nameComponent)
            .withMeaning("type" /* TypeAlias */);
    }
}
exports.ApiTypeAlias = ApiTypeAlias;
//# sourceMappingURL=ApiTypeAlias.js.map