"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = syncProcess;

var _fs = _interopRequireDefault(require("fs"));

var _path = require("path");

var _pofile = _interopRequireDefault(require("pofile"));

var _https = _interopRequireDefault(require("https"));

var _glob = _interopRequireDefault(require("glob"));

var _dateFns = require("date-fns");

var getCreateHeaders = function getCreateHeaders(language) {
  return {
    "POT-Creation-Date": (0, _dateFns.format)(new Date(), "yyyy-MM-dd HH:mmxxxx"),
    "MIME-Version": "1.0",
    "Content-Type": "text/plain; charset=utf-8",
    "Content-Transfer-Encoding": "8bit",
    "X-Generator": "@lingui/cli",
    Language: language
  };
}; // Main sync method, call "Init" or "Sync" depending on the project context


function syncProcess(config, options) {
  if (config.format != 'po') {
    console.error("\n----------\nTranslation.io service is only compatible with the \"po\" format. Please update your Lingui configuration accordingly.\n----------");
    process.exit(1);
  }

  var successCallback = function successCallback(project) {
    console.log("\n----------\nProject successfully synchronized. Please use this URL to translate: ".concat(project.url, "\n----------"));
  };

  var failCallback = function failCallback(errors) {
    console.error("\n----------\nSynchronization with Translation.io failed: ".concat(errors.join(', '), "\n----------"));
  };

  init(config, options, successCallback, function (errors) {
    if (errors.length && errors[0] === 'This project has already been initialized.') {
      sync(config, options, successCallback, failCallback);
    } else {
      failCallback(errors);
    }
  });
} // Initialize project with source and existing translations (only first time!)
// Cf. https://translation.io/docs/create-library#initialization


function init(config, options, successCallback, failCallback) {
  var sourceLocale = config.sourceLocale || 'en';
  var pseudoLocale = config.pseudoLocale || 'pseudo';
  var targetLocales = config.locales.filter(function (value) {
    return value != sourceLocale && value != pseudoLocale;
  });
  var paths = poPathsPerLocale(config);
  var segments = {};
  targetLocales.forEach(function (targetLocale) {
    segments[targetLocale] = [];
  }); // Create segments from source locale PO items

  paths[sourceLocale].forEach(function (path) {
    var raw = _fs.default.readFileSync(path).toString();

    var po = _pofile.default.parse(raw);

    po.items.filter(function (item) {
      return !item['obsolete'];
    }).forEach(function (item) {
      targetLocales.forEach(function (targetLocale) {
        var newSegment = createSegmentFromPoItem(item);
        segments[targetLocale].push(newSegment);
      });
    });
  }); // Add translations to segments from target locale PO items

  targetLocales.forEach(function (targetLocale) {
    paths[targetLocale].forEach(function (path) {
      var raw = _fs.default.readFileSync(path).toString();

      var po = _pofile.default.parse(raw);

      po.items.filter(function (item) {
        return !item['obsolete'];
      }).forEach(function (item, index) {
        segments[targetLocale][index].target = item.msgstr[0];
      });
    });
  });
  var request = {
    "client": "lingui",
    "version": require('@lingui/core/package.json').version,
    "source_language": sourceLocale,
    "target_languages": targetLocales,
    "segments": segments
  };
  postTio("init", request, config.service.apiKey, function (response) {
    if (response.errors) {
      failCallback(response.errors);
    } else {
      saveSegmentsToTargetPos(config, paths, response.segments);
      successCallback(response.project);
    }
  }, function (error) {
    console.error("\n----------\nSynchronization with Translation.io failed: ".concat(error, "\n----------"));
  });
} // Send all source text from PO to Translation.io and create new PO based on received translations
// Cf. https://translation.io/docs/create-library#synchronization


function sync(config, options, successCallback, failCallback) {
  var sourceLocale = config.sourceLocale || 'en';
  var targetLocales = config.locales.filter(function (value) {
    return value != sourceLocale;
  });
  var paths = poPathsPerLocale(config);
  var segments = []; // Create segments with correct source

  paths[sourceLocale].forEach(function (path) {
    var raw = _fs.default.readFileSync(path).toString();

    var po = _pofile.default.parse(raw);

    po.items.filter(function (item) {
      return !item['obsolete'];
    }).forEach(function (item) {
      var newSegment = createSegmentFromPoItem(item);
      segments.push(newSegment);
    });
  });
  var request = {
    "client": "lingui",
    "version": require('@lingui/core/package.json').version,
    "source_language": sourceLocale,
    "target_languages": targetLocales,
    "segments": segments
  }; // Sync and then remove unused segments (not present in the local application) from Translation.io

  if (options.clean) {
    request['purge'] = true;
  }

  postTio("sync", request, config.service.apiKey, function (response) {
    if (response.errors) {
      failCallback(response.errors);
    } else {
      saveSegmentsToTargetPos(config, paths, response.segments);
      successCallback(response.project);
    }
  }, function (error) {
    console.error("\n----------\nSynchronization with Translation.io failed: ".concat(error, "\n----------"));
  });
}

function createSegmentFromPoItem(item) {
  var itemHasId = item.msgid != item.msgstr[0] && item.msgstr[0].length;
  var segment = {
    type: 'source',
    // No way to edit text for source language (inside code), so not using "key" here
    source: itemHasId ? item.msgstr[0] : item.msgid,
    // msgstr may be empty if --overwrite is used and no ID is used
    context: '',
    references: [],
    comment: ''
  };

  if (itemHasId) {
    segment.context = item.msgid;
  }

  if (item.references.length) {
    segment.references = item.references;
  }

  if (item.extractedComments.length) {
    segment.comment = item.extractedComments.join(' | ');
  }

  return segment;
}

function createPoItemFromSegment(segment) {
  var item = new _pofile.default.Item();
  item.msgid = segment.context ? segment.context : segment.source;
  item.msgstr = [segment.target];
  item.references = segment.references && segment.references.length ? segment.references : [];
  item.extractedComments = segment.comment ? segment.comment.split(' | ') : [];
  return item;
}

function saveSegmentsToTargetPos(config, paths, segmentsPerLocale) {
  var NAME = "{name}";
  var LOCALE = "{locale}";
  Object.keys(segmentsPerLocale).forEach(function (targetLocale) {
    // Remove existing target POs and JS for this target locale
    paths[targetLocale].forEach(function (path) {
      var jsPath = path.replace(/\.po?$/, "") + ".js";
      var dirPath = (0, _path.dirname)(path); // Remove PO, JS and empty dir

      if (_fs.default.existsSync(path)) {
        _fs.default.unlinkSync(path);
      }

      if (_fs.default.existsSync(jsPath)) {
        _fs.default.unlinkSync(jsPath);
      }

      if (_fs.default.existsSync(dirPath) && _fs.default.readdirSync(dirPath).length === 0) {
        _fs.default.rmdirSync(dirPath);
      }
    }); // Find target path (ignoring {name})

    var localePath = "".concat(config.catalogs[0].path.replace(LOCALE, targetLocale).replace(NAME, ''), ".po");
    var segments = segmentsPerLocale[targetLocale];
    var po = new _pofile.default();
    po.headers = getCreateHeaders(targetLocale);
    var items = [];
    segments.forEach(function (segment) {
      var item = createPoItemFromSegment(segment);
      items.push(item);
    }); // Sort items by messageId

    po.items = items.sort(function (a, b) {
      if (a.msgid < b.msgid) {
        return -1;
      }

      if (a.msgid > b.msgid) {
        return 1;
      }

      return 0;
    }); // Check that localePath directory exists and save PO file

    _fs.default.promises.mkdir((0, _path.dirname)(localePath), {
      recursive: true
    }).then(function () {
      po.save(localePath, function (err) {
        if (err) {
          console.error('Error while saving target PO files:');
          console.error(err);
          process.exit(1);
        }
      });
    });
  });
}

function poPathsPerLocale(config) {
  var NAME = "{name}";
  var LOCALE = "{locale}";
  var paths = [];
  config.locales.forEach(function (locale) {
    paths[locale] = [];
    config.catalogs.forEach(function (catalog) {
      var path = "".concat(catalog.path.replace(LOCALE, locale).replace(NAME, "*"), ".po"); // If {name} is present (replaced by *), list all the existing POs

      if (path.includes('*')) {
        paths[locale] = paths[locale].concat(_glob.default.sync(path));
      } else {
        paths[locale].push(path);
      }
    });
  });
  return paths;
}

function postTio(action, request, apiKey, successCallback, failCallback) {
  var jsonRequest = JSON.stringify(request);
  var options = {
    hostname: 'translation.io',
    path: '/api/v1/segments/' + action + '.json?api_key=' + apiKey,
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    }
  };

  var req = _https.default.request(options, function (res) {
    res.setEncoding('utf8');
    var body = "";
    res.on('data', function (chunk) {
      body = body.concat(chunk);
    });
    res.on('end', function () {
      var response = JSON.parse(body);
      successCallback(response);
    });
  });

  req.on('error', function (e) {
    failCallback(e);
  });
  req.write(jsonRequest);
  req.end();
}