"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.UngroupedProcessor = exports.Processor = void 0;
const aggregators = require("./aggregators");
const types_1 = require("./types");
/**
 * Base class for all processor types.
 *
 * The processor is responsible for storing the aggregators and aggregated
 * values received from updates from metrics in the meter. The stored values
 * will be sent to an exporter for exporting.
 */
class Processor {
    constructor() {
        this._batchMap = new Map();
    }
    checkPointSet() {
        return Array.from(this._batchMap.values());
    }
}
exports.Processor = Processor;
/**
 * Processor which retains all dimensions/labels. It accepts all records and
 * passes them for exporting.
 */
class UngroupedProcessor extends Processor {
    aggregatorFor(metricDescriptor) {
        switch (metricDescriptor.metricKind) {
            case types_1.MetricKind.COUNTER:
            case types_1.MetricKind.UP_DOWN_COUNTER:
                return new aggregators.SumAggregator();
            case types_1.MetricKind.SUM_OBSERVER:
            case types_1.MetricKind.UP_DOWN_SUM_OBSERVER:
            case types_1.MetricKind.VALUE_OBSERVER:
                return new aggregators.LastValueAggregator();
            case types_1.MetricKind.VALUE_RECORDER:
                return new aggregators.HistogramAggregator(metricDescriptor.boundaries || [Infinity]);
            default:
                return new aggregators.LastValueAggregator();
        }
    }
    process(record) {
        const labels = Object.keys(record.labels)
            .map(k => `${k}=${record.labels[k]}`)
            .join(',');
        this._batchMap.set(record.descriptor.name + labels, record);
    }
}
exports.UngroupedProcessor = UngroupedProcessor;
//# sourceMappingURL=Processor.js.map