"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createHttpAgent = exports.sendWithHttp = void 0;
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const url = require("url");
const http = require("http");
const https = require("https");
const zlib = require("zlib");
const stream_1 = require("stream");
const collectorTypes = require("../../types");
const api_1 = require("@opentelemetry/api");
const types_1 = require("./types");
const gzip = zlib.createGzip();
/**
 * Sends data using http
 * @param collector
 * @param data
 * @param contentType
 * @param onSuccess
 * @param onError
 */
function sendWithHttp(collector, data, contentType, onSuccess, onError) {
    const parsedUrl = new url.URL(collector.url);
    const options = {
        hostname: parsedUrl.hostname,
        port: parsedUrl.port,
        path: parsedUrl.pathname,
        method: 'POST',
        headers: Object.assign({ 'Content-Length': Buffer.byteLength(data), 'Content-Type': contentType }, collector.headers),
        agent: collector.agent,
    };
    const request = parsedUrl.protocol === 'http:' ? http.request : https.request;
    const req = request(options, (res) => {
        let responseData = '';
        res.on('data', chunk => (responseData += chunk));
        res.on('end', () => {
            if (res.statusCode && res.statusCode < 299) {
                api_1.diag.debug(`statusCode: ${res.statusCode}`, responseData);
                onSuccess();
            }
            else {
                const error = new collectorTypes.CollectorExporterError(res.statusMessage, res.statusCode, responseData);
                onError(error);
            }
        });
    });
    req.on('error', (error) => {
        onError(error);
    });
    switch (collector.compression) {
        case types_1.CompressionAlgorithm.GZIP: {
            req.setHeader('Content-Encoding', 'gzip');
            const dataStream = readableFromBuffer(data);
            dataStream.on('error', onError)
                .pipe(gzip).on('error', onError)
                .pipe(req);
            break;
        }
        default:
            req.write(data);
            req.end();
            break;
    }
}
exports.sendWithHttp = sendWithHttp;
function readableFromBuffer(buff) {
    const readable = new stream_1.Readable();
    readable.push(buff);
    readable.push(null);
    return readable;
}
function createHttpAgent(config) {
    if (config.httpAgentOptions && config.keepAlive === false) {
        api_1.diag.warn('httpAgentOptions is used only when keepAlive is true');
        return undefined;
    }
    if (config.keepAlive === false || !config.url)
        return undefined;
    try {
        const parsedUrl = new url.URL(config.url);
        const Agent = parsedUrl.protocol === 'http:' ? http.Agent : https.Agent;
        return new Agent(Object.assign({ keepAlive: true }, config.httpAgentOptions));
    }
    catch (err) {
        api_1.diag.error(`collector exporter failed to create http agent. err: ${err.message}`);
        return undefined;
    }
}
exports.createHttpAgent = createHttpAgent;
//# sourceMappingURL=util.js.map