"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererFileCacheΩ = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const path_1 = (0, tslib_1.__importDefault)(require("path"));
const utils_1 = require("../utils");
const reader_1 = require("./reader");
const writer_1 = require("./writer");
const BETTERER_CACHE_VERSION = 2;
// The BettererFileCacheΩ is a simple JSON cache of test + file data.
// Each test has it's own cache entry, which is a list of hashes:
//
// {
//   "version": 2,
//   "testCache": {
//     "test name": {
//        "relative file path": "hash",
//        "another relative file path": "another hash"
//        ...
//     }
//   }
// }
//
// Each stored hash is the concatenated hash of the Betterer config files
// and the file at that path at the point it is written. It is written each
// time a test runs on a given file and the file gets better, stays the same,
// or gets updated. If the hash hasn't changed, then the config file hasn't
// changed, and the file hasn't changed, so running the test on the file again
// *should* have the same result.
//
// Of course the actual test itself could have changed so ... 🤷‍♂️
class BettererFileCacheΩ {
    constructor(_configPaths) {
        this._configPaths = _configPaths;
        this._cachePath = null;
        this._fileHashMap = {};
        this._memoryCacheMap = {};
        this._reading = null;
    }
    clearCache(testName) {
        delete this._memoryCacheMap[testName];
    }
    async enableCache(cachePath) {
        this._cachePath = cachePath;
        this._memoryCacheMap = await this._readCache(this._cachePath);
    }
    async writeCache() {
        if (!this._cachePath) {
            return;
        }
        // Clean up any expired cache entries before writing to disk:
        Object.keys(this._memoryCacheMap).forEach((testName) => {
            const testCache = this._memoryCacheMap[testName];
            Object.keys(testCache).forEach((filePath) => {
                const hash = this._fileHashMap[filePath];
                if (hash === null) {
                    delete this._memoryCacheMap[filePath];
                }
            });
        });
        const relativeTestCache = {};
        Object.keys(this._memoryCacheMap).forEach((testName) => {
            const absoluteFileHashMap = this._memoryCacheMap[testName];
            const relativeFileHashMap = {};
            Object.keys(absoluteFileHashMap).forEach((absoluteFilePath) => {
                (0, assert_1.default)(this._cachePath);
                const relativePath = (0, utils_1.normalisedPath)(path_1.default.relative(path_1.default.dirname(this._cachePath), absoluteFilePath));
                relativeFileHashMap[relativePath] = absoluteFileHashMap[absoluteFilePath];
            });
            relativeTestCache[testName] = relativeFileHashMap;
        });
        const cache = { version: BETTERER_CACHE_VERSION, testCache: relativeTestCache };
        const cacheString = JSON.stringify(cache, null, '  ');
        await (0, writer_1.write)(cacheString, this._cachePath);
    }
    filterCached(testName, filePaths) {
        if (!this._cachePath) {
            return filePaths;
        }
        const testCache = this._memoryCacheMap[testName] || {};
        return filePaths.filter((filePath) => {
            const hash = this._fileHashMap[filePath];
            // If hash is null, then the file isn't tracked by version control *and* it can't be read,
            // so it probably doesn't exist
            if (hash === null) {
                return true;
            }
            const previousHash = testCache[filePath];
            return hash !== previousHash;
        });
    }
    updateCache(testName, filePaths) {
        if (!this._cachePath) {
            return;
        }
        this._memoryCacheMap[testName] = this._memoryCacheMap[testName] || {};
        const testCache = this._memoryCacheMap[testName];
        const existingFilePaths = Object.keys(testCache);
        const cacheFilePaths = Array.from(new Set([...existingFilePaths, ...filePaths])).sort();
        const updatedCache = {};
        cacheFilePaths.forEach((filePath) => {
            const hash = this._fileHashMap[filePath];
            // If hash is null, then the file isn't tracked by version control *and* it can't be read,
            // so it probably doesn't exist
            if (hash === null) {
                return;
            }
            updatedCache[filePath] = hash;
        });
        this._memoryCacheMap[testName] = updatedCache;
    }
    setHashes(newHashes) {
        if (!this._cachePath) {
            return;
        }
        const configHash = this._getConfigHash(newHashes);
        this._fileHashMap = {};
        Object.keys(newHashes).forEach((absolutePath) => {
            this._fileHashMap[absolutePath] = `${configHash}${newHashes[absolutePath]}`;
        });
    }
    async _readCache(cachePath) {
        if (!this._reading) {
            this._reading = (0, reader_1.read)(cachePath);
        }
        const cache = await this._reading;
        this._reading = null;
        if (!cache) {
            return {};
        }
        const parsedCache = JSON.parse(cache);
        const { version } = parsedCache;
        if (!version) {
            return {};
        }
        const relativeTestCacheMap = parsedCache.testCache;
        // Transform relative paths back into absolute paths:
        const absoluteTestCacheMap = {};
        Object.keys(relativeTestCacheMap).forEach((testName) => {
            const relativeFileHashMap = relativeTestCacheMap[testName];
            const absoluteFileHashMap = {};
            Object.keys(relativeFileHashMap).forEach((relativePath) => {
                (0, assert_1.default)(this._cachePath);
                const absolutePath = (0, utils_1.normalisedPath)(path_1.default.resolve(path_1.default.dirname(this._cachePath), relativePath));
                absoluteFileHashMap[absolutePath] = relativeFileHashMap[relativePath];
            });
            absoluteTestCacheMap[testName] = absoluteFileHashMap;
        });
        return absoluteTestCacheMap;
    }
    _getConfigHash(newFileHashMap) {
        return this._configPaths.map((configPath) => newFileHashMap[(0, utils_1.normalisedPath)(configPath)]).join('');
    }
}
exports.BettererFileCacheΩ = BettererFileCacheΩ;
//# sourceMappingURL=file-cache.js.map