"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererWorkerRunΩ = void 0;
const tslib_1 = require("tslib");
const constraints_1 = require("@betterer/constraints");
const errors_1 = require("@betterer/errors");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const config_1 = require("../config");
const fs_1 = require("../fs");
const results_1 = require("../results");
const test_1 = require("../test");
const test_2 = require("../test");
const run_summary_1 = require("./run-summary");
class BettererWorkerRunΩ {
    constructor(name, globals, testMeta, test) {
        this.name = name;
        this.globals = globals;
        this.testMeta = testMeta;
        this.test = test;
        this.filePaths = null;
        this._baseline = null;
        this._expected = null;
        this.config = globals.config;
        this.isNew = testMeta.isNew;
        this.isSkipped = testMeta.isSkipped;
    }
    get baseline() {
        (0, assert_1.default)(this._baseline != null);
        return this._baseline;
    }
    get expected() {
        (0, assert_1.default)(this._expected != null);
        return this._expected;
    }
    static async create(runConfig, name, versionControl) {
        const config = await (0, config_1.createWorkerConfig)(runConfig);
        const resultsFile = await results_1.BettererResultsFileΩ.create(config.resultsPath, versionControl);
        const globals = { config, resultsFile, versionControl };
        const isNew = !resultsFile.hasResult(name);
        const testFactories = (0, test_1.loadTestMeta)(config);
        const testFactoryMeta = testFactories[name];
        const test = await testFactoryMeta.factory();
        const isTest = (0, test_2.isBettererTest)(test);
        const isFileTest = (0, test_1.isBettererFileTest)(test);
        if (!(isTest || isFileTest)) {
            throw new errors_1.BettererError(`"${name}" must return a \`BettererTest\`.`);
        }
        test.config.configPath = testFactoryMeta.configPath;
        const baseTestMeta = {
            name,
            configPath: testFactoryMeta.configPath,
            isFileTest,
            isOnly: test.isOnly,
            isSkipped: test.isSkipped
        };
        let testMeta;
        if (isNew) {
            testMeta = Object.assign(Object.assign({}, baseTestMeta), { isNew: true, baselineJSON: null, expectedJSON: null });
        }
        else {
            const [baselineJSON, expectedJSON] = resultsFile.getExpected(name);
            testMeta = Object.assign(Object.assign({}, baseTestMeta), { isNew: false, baselineJSON,
                expectedJSON });
        }
        return new BettererWorkerRunΩ(name, globals, testMeta, test.config);
    }
    async run(filePaths, isSkipped, timestamp) {
        this.filePaths = filePaths;
        if (!this.testMeta.isNew) {
            this._baseline = this._deserialise(this.testMeta.baselineJSON);
            this._expected = this._deserialise(this.testMeta.expectedJSON);
        }
        const running = this._run(this.config, timestamp);
        if (isSkipped) {
            return await running.skipped();
        }
        try {
            const result = new results_1.BettererResultΩ(await this.test.test(this));
            return await running.done(result);
        }
        catch (error) {
            return await running.failed(error);
        }
    }
    _deserialise(resultJSON) {
        try {
            const serialised = JSON.parse(resultJSON);
            const { resultsPath } = this.config;
            const { deserialise } = this.test.serialiser;
            return new results_1.BettererResultΩ(deserialise(serialised, resultsPath));
        }
        catch (_a) {
            return null;
        }
    }
    _run(config, timestamp) {
        const end = async (status, result = null, diff = null, error = null) => {
            const isBetter = status === run_summary_1.BettererRunStatus.better;
            const isSame = status === run_summary_1.BettererRunStatus.same;
            const isUpdated = status === run_summary_1.BettererRunStatus.update;
            const isSkipped = status === run_summary_1.BettererRunStatus.skipped;
            const isFailed = status === run_summary_1.BettererRunStatus.failed;
            const isWorse = status === run_summary_1.BettererRunStatus.worse;
            const baselineValue = this.isNew ? null : this.baseline.value;
            const resultValue = !result ? null : result.value;
            const delta = await this.test.progress(baselineValue, resultValue);
            const { resultsPath, ci } = config;
            const { serialise } = this.test.serialiser;
            const resultSerialised = resultValue != null ? new results_1.BettererResultΩ(serialise(resultValue, resultsPath)) : null;
            const baselineSerialised = this.isNew ? null : new results_1.BettererResultΩ(serialise(this.baseline.value, resultsPath));
            const expectedSerialised = this.isNew ? null : new results_1.BettererResultΩ(serialise(this.expected.value, resultsPath));
            const isComplete = resultValue != null && (await this.test.goal(resultValue));
            const isExpired = timestamp >= this.test.deadline;
            let printed = null;
            const shouldPrint = !(isComplete || (this.isNew && (isFailed || isSkipped)));
            if (shouldPrint) {
                const toPrint = isFailed || isSkipped || isWorse ? this.expected : result;
                const toPrintSerialised = this.test.serialiser.serialise(toPrint.value, config.resultsPath);
                printed = (0, fs_1.forceRelativePaths)(await this.test.printer(toPrintSerialised), config.versionControlPath);
            }
            if (this.testMeta.isFileTest && !ci) {
                if (isComplete) {
                    await this.globals.versionControl.clearCache(this.name);
                }
                else if (isBetter || isSame || isUpdated || this.isNew) {
                    await this.globals.versionControl.updateCache(this.name, this.filePaths);
                }
            }
            return new run_summary_1.BettererRunSummaryΩ(this, resultSerialised, baselineSerialised, expectedSerialised, status, isComplete, isExpired, delta, diff, error, printed, timestamp);
        };
        return {
            done: async (result) => {
                if (this.isNew) {
                    return await end(run_summary_1.BettererRunStatus.new, result);
                }
                const comparison = await this.test.constraint(result.value, this.expected.value);
                if (comparison === constraints_1.BettererConstraintResult.same) {
                    return await end(run_summary_1.BettererRunStatus.same, result);
                }
                const diff = this.test.differ(this.expected.value, result.value);
                if (comparison === constraints_1.BettererConstraintResult.better) {
                    return await end(run_summary_1.BettererRunStatus.better, result, diff);
                }
                const status = config.update ? run_summary_1.BettererRunStatus.update : run_summary_1.BettererRunStatus.worse;
                return await end(status, result, diff);
            },
            failed: async (error) => {
                return await end(run_summary_1.BettererRunStatus.failed, null, null, error);
            },
            skipped: async () => {
                return await end(run_summary_1.BettererRunStatus.skipped);
            }
        };
    }
}
exports.BettererWorkerRunΩ = BettererWorkerRunΩ;
//# sourceMappingURL=worker-run.js.map