"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererRunnerΩ = void 0;
const errors_1 = require("@betterer/errors");
const context_1 = require("../context");
const globals_1 = require("../globals");
const utils_1 = require("../utils");
const watcher_1 = require("./watcher");
const DEBOUNCE_TIME = 200;
class BettererRunnerΩ {
    constructor(_context) {
        this._context = _context;
        this._jobs = [];
        this._running = null;
    }
    static async create(options) {
        const globals = await (0, globals_1.createGlobals)(options);
        const watcher = await (0, watcher_1.createWatcher)(globals);
        const context = new context_1.BettererContextΩ(globals, watcher);
        const runner = new BettererRunnerΩ(context);
        if (watcher) {
            watcher.on('all', (event, filePath) => {
                if (watcher_1.WATCHER_EVENTS.includes(event)) {
                    void runner.queue([filePath]);
                }
            });
        }
        return runner;
    }
    async options(optionsOverride) {
        await this._context.options(optionsOverride);
    }
    run() {
        return this._context.runOnce();
    }
    queue(filePathOrPaths = []) {
        const filePaths = Array.isArray(filePathOrPaths) ? filePathOrPaths : [filePathOrPaths];
        if (this._context.isDestroyed) {
            throw new errors_1.BettererError('You cannot queue a test run after the runner has been stopped! 💥');
        }
        this._addJob(filePaths);
        return new Promise((resolve) => {
            setTimeout(() => {
                void (async () => {
                    await this._processQueue();
                    resolve();
                })();
            }, DEBOUNCE_TIME);
        });
    }
    async stop(force) {
        try {
            if (!force) {
                await this._running;
            }
            return await this._context.stop();
        }
        catch (error) {
            if (force) {
                return null;
            }
            throw error;
        }
    }
    _addJob(filePaths = []) {
        const normalisedPaths = filePaths.map(utils_1.normalisedPath);
        this._jobs.push(normalisedPaths);
    }
    async _processQueue() {
        // It's possible for the queue debounce to trigger *after* `this.stop()` has been called:
        if (this._context.isDestroyed) {
            this._jobs = [];
            return;
        }
        if (this._jobs.length) {
            const filePaths = new Set();
            this._jobs.forEach((job) => {
                job.forEach((path) => {
                    filePaths.add(path);
                });
            });
            const runPaths = Array.from(filePaths).sort();
            this._jobs = [];
            this._running = this._context.run(runPaths);
            try {
                await this._running;
            }
            catch (_a) {
                // Errors will be handled by reporters
            }
        }
    }
}
exports.BettererRunnerΩ = BettererRunnerΩ;
//# sourceMappingURL=runner.js.map