"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.differ = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const FORMATTER = Intl.NumberFormat();
function differ(expected, result) {
    const diff = {};
    const expectedΩ = expected;
    const resultΩ = result;
    const unchangedResultFiles = resultΩ.files.filter((r) => expectedΩ.files.find((e) => e.absolutePath === r.absolutePath && e.hash === r.hash));
    const changedResultFiles = resultΩ.files.filter((r) => expectedΩ.files.find((e) => e.absolutePath === r.absolutePath && e.hash !== r.hash));
    const newOrMovedFiles = resultΩ.files.filter((r) => !expectedΩ.files.find((e) => e.absolutePath === r.absolutePath));
    const fixedOrMovedFiles = expectedΩ.files.filter((e) => !resultΩ.files.find((r) => r.absolutePath === e.absolutePath));
    const movedFiles = new Map();
    fixedOrMovedFiles.forEach((fixedOrMovedFile, index) => {
        // A file may have been moved it has the same hash in both result and expected
        const possibilities = newOrMovedFiles.filter((newOrMovedFile) => newOrMovedFile.hash === fixedOrMovedFile.hash);
        if (!possibilities.length) {
            return;
        }
        // Multiple possibilities means that the same content has been moved into multiple new files.
        // So just count the first one as a move, the rest will be new files:
        const [moved] = possibilities;
        movedFiles.set(moved, fixedOrMovedFile);
        // Remove the moved file from the fixedOrMovedFiles array:
        fixedOrMovedFiles.splice(index, 1);
        // And from the newOrMovedFiles array:
        newOrMovedFiles.splice(newOrMovedFiles.indexOf(moved), 1);
    });
    // All the moved files have been removed from fixedOrMovedFiles and newOrMovedFiles:
    const fixedFiles = fixedOrMovedFiles;
    const newFiles = newOrMovedFiles;
    fixedFiles.forEach((file) => {
        diff[file.absolutePath] = {
            fixed: file.issues.map(serialiseIssue)
        };
    });
    newFiles.forEach((file) => {
        diff[file.absolutePath] = {
            new: file.issues.map(serialiseIssue)
        };
    });
    const existingFiles = [...unchangedResultFiles, ...changedResultFiles, ...Array.from(movedFiles.keys())];
    existingFiles.forEach((resultFile) => {
        const expectedFile = movedFiles.get(resultFile) || expectedΩ.getFile(resultFile.absolutePath);
        // Convert all issues to their deserialised form for easier diffing:
        const resultIssues = [...resultFile.issues];
        const expectedIssues = expectedFile.issues;
        // Find all issues that exist in both result and expected:
        const unchangedExpectedIssues = expectedIssues.filter((r) => resultIssues.find((e) => {
            return e.line === r.line && e.column === r.column && e.length === r.length && e.hash === r.hash;
        }));
        const unchangedResultIssues = resultIssues.filter((r) => expectedIssues.find((e) => {
            return e.line === r.line && e.column === r.column && e.length === r.length && e.hash === r.hash;
        }));
        // Any result issues that aren't in expected are either new or have been moved:
        const newOrMovedIssues = resultIssues.filter((r) => !unchangedResultIssues.includes(r));
        // Any expected issues that aren't in result are either fixed or have been moved:
        const fixedOrMovedIssues = expectedIssues.filter((e) => !unchangedExpectedIssues.includes(e));
        // We can find the moved issues by matching the issue hashes:
        const movedIssues = [];
        const fixedIssues = [];
        fixedOrMovedIssues.forEach((fixedOrMovedIssue) => {
            const { hash, line, column } = fixedOrMovedIssue;
            // An issue may have been moved it has the same hash in both result and expected
            const possibilities = newOrMovedIssues.filter((newOrMovedIssue) => newOrMovedIssue.hash === hash);
            if (!possibilities.length) {
                // If there is no matching has the issue must have been fixed:
                fixedIssues.push(fixedOrMovedIssue);
                return;
            }
            // Start by marking the first possibility as best:
            let best = possibilities.shift();
            // And then search through all the possibilities to find the closest issue:
            possibilities.forEach((possibility) => {
                (0, assert_1.default)(best);
                if (Math.abs(line - possibility.line) >= Math.abs(line - best.line)) {
                    return;
                }
                if (Math.abs(line - possibility.line) < Math.abs(line - best.line)) {
                    best = possibility;
                }
                if (Math.abs(column - possibility.column) >= Math.abs(column - best.column)) {
                    return;
                }
                if (Math.abs(column - possibility.column) < Math.abs(column - best.column)) {
                    best = possibility;
                }
            });
            (0, assert_1.default)(best);
            // Remove the moved issue from the newOrMovedIssues array:
            newOrMovedIssues.splice(newOrMovedIssues.indexOf(best), 1);
            movedIssues.push(best);
        });
        // Find the raw issue data so that diffs can be logged:
        const newIssues = newOrMovedIssues.map((newIssue) => resultFile.issues[resultIssues.indexOf(newIssue)]);
        // If there's no change, move on:
        if (!newIssues.length && !fixedIssues.length) {
            return;
        }
        // Otherwise construct the diff:
        diff[resultFile.absolutePath] = {
            existing: [...unchangedExpectedIssues, ...movedIssues].map(serialiseIssue),
            fixed: fixedIssues.map(serialiseIssue),
            new: newIssues.map(serialiseIssue)
        };
    });
    const filePaths = Object.keys(diff);
    const logs = [];
    filePaths.forEach((filePath) => {
        const existing = diff[filePath].existing || [];
        const fixed = diff[filePath].fixed || [];
        if (fixed === null || fixed === void 0 ? void 0 : fixed.length) {
            logs.push({ success: `${fixed.length} fixed ${getIssues(fixed.length)} in "${filePath}".` });
        }
        if (existing === null || existing === void 0 ? void 0 : existing.length) {
            logs.push({ warn: `${existing.length} existing ${getIssues(existing.length)} in "${filePath}".` });
        }
        const newIssues = diff[filePath].new || [];
        const nIssues = newIssues.length;
        if (nIssues) {
            logs.push({ error: `New ${getIssues(nIssues)} in "${filePath}"!` });
            if (nIssues > 1) {
                logs.push({ error: `Showing first of ${FORMATTER.format(nIssues)} new issues:` });
            }
            const [firstIssue] = newIssues;
            const fileΩ = resultΩ.getFile(filePath);
            const { fileText } = fileΩ;
            const [line, column, length, message] = firstIssue;
            logs.push({ code: { message, filePath, fileText, line, column, length } });
        }
    });
    return {
        diff,
        logs
    };
}
exports.differ = differ;
function getIssues(count) {
    return count === 1 ? 'issue' : 'issues';
}
function serialiseIssue(issue) {
    return [issue.line, issue.column, issue.length, issue.message, issue.hash];
}
//# sourceMappingURL=differ.js.map