"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isBettererFileTest = exports.BettererFileTest = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const path_1 = (0, tslib_1.__importDefault)(require("path"));
const fs_1 = require("../../fs");
const config_1 = require("../config");
const constraint_1 = require("./constraint");
const differ_1 = require("./differ");
const file_test_result_1 = require("./file-test-result");
const goal_1 = require("./goal");
const printer_1 = require("./printer");
const progress_1 = require("./progress");
const serialiser_1 = require("./serialiser");
/**
 * @public A very common usecase for **Betterer** is to track issues across all the files in a
 * project. `BettererFileTest` provides a wrapper around {@link @betterer/betterer#BettererTest | `BettererTest` }
 * that makes it easier to implement such a test.
 *
 * @remarks `BettererFileTest` provides a useful example for the more complex possibilities of the {@link @betterer/betterer#BettererTestOptions | `BettererTestOptions` }
 * interface.
 *
 * @example
 * ```typescript
 * const fileTest = new BettererFileTest(async (filePaths, fileTestResult) => {
 *    await Promise.all(
 *      filePaths.map(async (filePath) => {
 *        const fileContents = await fs.readFile(filePath, 'utf8');
 *        const file = fileTestResult.addFile(filePath, fileContents);
 *        file.addIssue(0, 1, `There's something wrong with this file!`);
 *      })
 *    );
 *  });
 * ```
 *
 * @param fileTest - The test function that will detect issues in specific files.
 */
class BettererFileTest {
    constructor(fileTest) {
        this._isOnly = false;
        this._isSkipped = false;
        this._resolver = new fs_1.BettererFileResolverΩ();
        this.config = (0, config_1.createTestConfig)({
            test: createTest(this._resolver, fileTest),
            constraint: constraint_1.constraint,
            goal: goal_1.goal,
            serialiser: { deserialise: serialiser_1.deserialise, serialise: serialiser_1.serialise },
            differ: differ_1.differ,
            printer: printer_1.printer,
            progress: progress_1.progress
        });
    }
    /**
     * When `true`, all other tests will be skipped. Calling `only()` will set this to `true`.
     */
    get isOnly() {
        return this._isOnly;
    }
    /**
     * When `true`, this test will be skipped. Calling `skip()` will set this to `true`.
     */
    get isSkipped() {
        return this._isSkipped;
    }
    /**
     * Override the constraint in the test configuration.
     *
     * @param constraintOverride - The new constraint for the test.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    constraint(constraintOverride) {
        this.config.constraint = constraintOverride;
        return this;
    }
    /**
     * Override the deadline in the test configuration.
     *
     * @param deadlineOverride - The new deadline for the test.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    deadline(deadlineOverride) {
        this.config.deadline = (0, config_1.createDeadline)(Object.assign(Object.assign({}, this.config), { deadline: deadlineOverride }));
        return this;
    }
    /**
     * Add a list of {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions | Regular Expression } filters for files to exclude when running the test.
     *
     * @param excludePatterns - RegExp filters to match file paths that should be excluded.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    exclude(...excludePatterns) {
        this._resolver.exclude(...excludePatterns);
        return this;
    }
    /**
     * Override the goal in the test configuration.
     *
     * @param goalOverride - The new goal for the test.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    goal(goalOverride) {
        this.config.goal = (0, config_1.createGoal)(Object.assign(Object.assign({}, this.config), { goal: goalOverride }));
        return this;
    }
    /**
     * Add a list of {@link https://www.npmjs.com/package/glob#user-content-glob-primer | glob }
     * patterns for files to include when running the test.
     *
     * @param includePatterns - Glob patterns to match file paths that should be included. All
     * `includes` should be relative to the {@link https://phenomnomnominal.github.io/betterer/docs/test-definition-file | test definition file}.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    include(...includePatterns) {
        this._resolver.include(...includePatterns);
        return this;
    }
    /**
     * Run only this test. All other tests will be marked as skipped.
     *
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    only() {
        this._isOnly = true;
        return this;
    }
    /**
     * Skip this test.
     *
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    skip() {
        this._isSkipped = true;
        return this;
    }
}
exports.BettererFileTest = BettererFileTest;
function createTest(resolver, fileTest) {
    return async (run) => {
        const runΩ = run;
        (0, assert_1.default)(runΩ.filePaths);
        const baseDirectory = path_1.default.dirname(runΩ.test.configPath);
        const { config, versionControl } = runΩ.globals;
        resolver.init(baseDirectory, versionControl);
        const hasSpecifiedFiles = runΩ.filePaths.length > 0;
        // Get the maximal set of files that the test could run on:
        const testFiles = await resolver.files();
        // Get the set of files that the test will run on:
        let runFiles;
        // Specified files will include files from a global `includes`.
        if (hasSpecifiedFiles) {
            // Validate that they are relevant for this file test:
            runFiles = await resolver.validate(runΩ.filePaths);
        }
        else {
            runFiles = testFiles;
        }
        let isFullRun = runFiles === testFiles;
        if (!run.isNew) {
            const cacheMisses = await versionControl.filterCached(run.name, runFiles);
            isFullRun = isFullRun && cacheMisses.length === runFiles.length;
            runFiles = cacheMisses;
        }
        // Set the final files back on the `BettererRun`:
        runΩ.filePaths = runFiles;
        const result = new file_test_result_1.BettererFileTestResultΩ(resolver, config.resultsPath);
        await fileTest(runFiles, result, resolver);
        if (isFullRun) {
            return result;
        }
        // Get any filePaths that have expected issues but weren't included in this run:
        const expectedΩ = runΩ.expected.value;
        const excludedFilesWithIssues = expectedΩ.files
            .map((file) => file.absolutePath)
            .filter((filePath) => !runFiles.includes(filePath));
        // Filter them based on the resolver:
        const relevantExcludedFilePaths = await resolver.validate(excludedFilesWithIssues);
        // Add the existing issues to the new result:
        relevantExcludedFilePaths.forEach((filePath) => result.addExpected(expectedΩ.getFile(filePath)));
        return result;
    };
}
function isBettererFileTest(test) {
    return !!test && test.constructor.name === BettererFileTest.name;
}
exports.isBettererFileTest = isBettererFileTest;
//# sourceMappingURL=file-test.js.map