"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererFileΩ = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const lines_and_columns_1 = require("lines-and-columns");
const hasher_1 = require("../../hasher");
const utils_1 = require("../../utils");
const UNKNOWN_LOCATION = {
    line: 0,
    column: 0
};
class BettererFileΩ {
    constructor(absolutePath, _relativePath, fileText) {
        this.absolutePath = absolutePath;
        this._relativePath = _relativePath;
        this.fileText = fileText;
        this._issues = [];
        this.absolutePath = (0, utils_1.normalisedPath)(absolutePath);
        this.hash = (0, hasher_1.createHash)(this.fileText);
        this.key = `${(0, utils_1.normalisedPath)(this._relativePath)}:${this.hash}`;
    }
    get issues() {
        return this._issues;
    }
    addIssues(issues) {
        this._issues = [...this._issues, ...issues];
    }
    addIssue(...issueOverride) {
        this.addIssues([this._handleIssue(issueOverride, this.fileText)]);
    }
    _handleIssue(issueOverride, fileText) {
        const lc = new lines_and_columns_1.LinesAndColumns(fileText);
        const issue = getIssueFromStartEnd(lc, issueOverride) ||
            getIssueFromLineColLength(issueOverride) ||
            getIssueFromPositions(lc, issueOverride);
        (0, assert_1.default)(issue);
        const [line, column, length, message, overrideHash] = issue;
        const start = lc.indexForLocation({ line, column }) || 0;
        const issueText = fileText.substring(start, start + length);
        const normalisedText = (0, utils_1.normaliseNewlines)(issueText);
        const hash = overrideHash || (0, hasher_1.createHash)(normalisedText);
        return { line, column, length: normalisedText.length, message, hash };
    }
}
exports.BettererFileΩ = BettererFileΩ;
function getIssueFromLineColLength(issueOverride) {
    if (!isLineColLength(issueOverride)) {
        return null;
    }
    return issueOverride;
}
function isLineColLength(issueOverride) {
    const [, , , message] = issueOverride;
    return (0, utils_1.isString)(message);
}
function getIssueFromStartEnd(lc, issueOverride) {
    if (!isStartEnd(issueOverride)) {
        return null;
    }
    const [start, end, message, overrideHash] = issueOverride;
    const { line, column } = lc.locationForIndex(start) || UNKNOWN_LOCATION;
    const length = end - start;
    return [line, column, length, message, overrideHash];
}
function isStartEnd(issueOverride) {
    const [, , message] = issueOverride;
    return (0, utils_1.isString)(message);
}
function getIssueFromPositions(lc, issueOverride) {
    if (!isPositions(issueOverride)) {
        return null;
    }
    const [line, column, endLine, endColumn, message, overrideHash] = issueOverride;
    const start = lc.indexForLocation({ line, column }) || 0;
    const end = lc.indexForLocation({ line: endLine, column: endColumn }) || 0;
    const length = end - start;
    return [line, column, length, message, overrideHash];
}
function isPositions(issueOverride) {
    const [, , , , message] = issueOverride;
    return (0, utils_1.isString)(message);
}
//# sourceMappingURL=file.js.map