"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const loader_1 = __importDefault(require("../devtools/loader"));
const utils_1 = require("../devtools/utils");
const track_1 = require("../devtools/timelineModel/track");
const timelineUIUtils_1 = __importDefault(require("../devtools/timelineModel/timelineUIUtils"));
const timelineData_1 = __importDefault(require("../devtools/timelineModel/timelineData"));
const counterGraph_1 = __importDefault(require("../devtools/timelineModel/counterGraph"));
const utils_2 = __importDefault(require("./utils"));
class Tracelib {
    constructor(tracelog, range) {
        this.tracelog = tracelog;
        this._timelineLoader = new loader_1.default(this.tracelog);
        this._timelineLoader.init();
        this._performanceModel = this._timelineLoader.performanceModel;
    }
    _findMainTrack() {
        const threads = this._performanceModel
            .timelineModel()
            .tracks();
        const mainTrack = threads.find((track) => Boolean(track.type === track_1.TrackType.MainThread && track.forMainFrame && track.events.length));
        /**
         * If no main thread could be found, pick the thread with most events
         * captured in it and assume this is the main track.
         */
        if (!mainTrack) {
            return threads.slice(1).reduce((curr, com) => curr.events.length > com.events.length ? curr : com, threads[0]);
        }
        return mainTrack;
    }
    getMainTrackEvents() {
        const mainTrack = this._findMainTrack();
        return mainTrack.events;
    }
    getFPS() {
        const fpsData = {
            times: [],
            values: []
        };
        this._timelineLoader.performanceModel.frames().forEach(({ duration, startTime }) => {
            fpsData.values.push(utils_1.calcFPS(duration));
            fpsData.times.push(startTime);
        });
        return fpsData;
    }
    getSummary(from, to) {
        const timelineUtils = new timelineUIUtils_1.default();
        const startTime = from || this._performanceModel.timelineModel().minimumRecordTime();
        const endTime = to || this._performanceModel.timelineModel().maximumRecordTime();
        const mainTrack = this._findMainTrack();
        // We are facing data mutaion issue in devtools, to avoid it cloning syncEvents
        const syncEvents = mainTrack.syncEvents().slice();
        return Object.assign(Object.assign({}, timelineUtils.statsForTimeRange(syncEvents, startTime, endTime)), { startTime,
            endTime });
    }
    getWarningCounts() {
        const mainTrack = this._findMainTrack();
        return mainTrack.events.reduce((counter, event) => {
            const timelineData = timelineData_1.default.forEvent(event);
            const warning = timelineData.warning;
            if (warning) {
                counter[warning] = counter[warning] ? counter[warning] + 1 : 1;
            }
            return counter;
        }, {});
    }
    getMemoryCounters() {
        const counterGraph = new counterGraph_1.default();
        const counters = counterGraph.setModel(this._performanceModel, this._findMainTrack());
        return Object.keys(counters).reduce((acc, counter) => (Object.assign(Object.assign({}, acc), { [counter]: {
                times: counters[counter].times,
                values: counters[counter].values,
            } })), {});
    }
    getDetailStats(from, to) {
        const timelineUtils = new utils_2.default();
        const startTime = from || this._performanceModel.timelineModel().minimumRecordTime();
        const endTime = to || this._performanceModel.timelineModel().maximumRecordTime();
        const mainTrack = this._findMainTrack();
        // We are facing data mutaion issue in devtools, to avoid it cloning syncEvents
        const syncEvents = mainTrack.syncEvents().slice();
        return Object.assign(Object.assign({}, timelineUtils.detailStatsForTimeRange(syncEvents, startTime, endTime)), { range: {
                times: [startTime, endTime],
                values: [startTime, endTime]
            } });
    }
}
exports.default = Tracelib;
//# sourceMappingURL=index.js.map