"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.watch = exports.runner = exports.results = exports.merge = exports.betterer = void 0;
const debug_1 = require("@phenomnomnominal/debug");
const runner_1 = require("./runner");
const results_1 = require("./results");
/**
 * @public Run **Betterer** with the given options.
 *
 * @example
 * ```typescript
 * import { betterer } from '@betterer/betterer';
 *
 * const suiteSummary = await betterer(options);
 * ```
 *
 * @param options - Options for running **Betterer**.
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if something goes wrong while running **Betterer**.
 */
async function betterer(options = {}) {
    initDebug();
    const runner = await runner_1.BettererRunnerΩ.create(options);
    return await runner.run();
}
exports.betterer = betterer;
/**
 * @public Resolve any merge conflicts in the specified results file.
 *
 * @example
 * ```typescript
 * import { betterer } from '@betterer/betterer';
 *
 * await betterer.merge(options);
 * ```
 *
 * @param options - Options for merging conflicts in the results file.
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if something goes wrong while merging conflicts in the results file.
 */
async function merge(options = {}) {
    const merger = await results_1.BettererMergerΩ.create(options);
    return await merger.merge();
}
exports.merge = merge;
betterer.merge = merge;
/**
 * @public Get a summary of the results of the defined {@link @betterer/betterer#BettererTest | `BettererTest`s}.
 *
 * **Betterer** will read the {@link https://phenomnomnominal.github.io/betterer/docs/test-definition-file | test definition file }
 * and the {@link https://phenomnomnominal.github.io/betterer/docs/results-file | results file} and return a summary of the results.
 *
 * @example
 * ```typescript
 * import { betterer } from '@betterer/betterer';
 *
 * const resultsSummary = await betterer.results(options);
 * ```
 *
 * @param options - Options for getting the summary of the results.
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if something goes wrong while getting the summary of the results.
 */
function results(options = {}) {
    initDebug();
    return results_1.BettererResultsSummaryΩ.create(options);
}
exports.results = results;
betterer.results = results;
/**
 * @public Create a **BettererRunner** with the given options.
 *
 * @example
 * ```typescript
 * import { betterer } from '@betterer/betterer';
 *
 * const runner = await betterer.runner(options);
 * ```
 *
 * @param options - Options for creating the runner.
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if something goes wrong while creating the runner.
 */
function runner(options = {}) {
    initDebug();
    return runner_1.BettererRunnerΩ.create(options);
}
exports.runner = runner;
betterer.runner = runner;
/**
 * @public Create a **BettererRunner** with the given options. Also starts up a file watcher
 * for tracked files in the current working directory.
 *
 * @example
 * ```typescript
 * import { betterer } from '@betterer/betterer';
 *
 * const runner = await betterer.watch(options);
 * ```
 *
 * @param options - Options for creating the runner.
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if something goes wrong while creating the runner or watcher.
 */
function watch(options = {}) {
    initDebug();
    return runner_1.BettererRunnerΩ.create(Object.assign(Object.assign({}, options), { watch: true }));
}
exports.watch = watch;
betterer.watch = watch;
function initDebug() {
    const enabled = !!process.env.BETTERER_DEBUG;
    if (enabled) {
        (0, debug_1.debug)({
            header: 'betterer',
            include: [/@betterer\//],
            ignore: [require.resolve('./utils'), require.resolve('./register')],
            enabled,
            time: !!process.env.BETTERER_DEBUG_TIME,
            values: !!process.env.BETTERER_DEBUG_VALUES,
            logPath: process.env.BETTERER_DEBUG_LOG
        });
    }
}
//# sourceMappingURL=betterer.js.map