"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isBettererTest = exports.BettererTest = void 0;
const config_1 = require("./config");
/**
 * @public The main interface to the **Betterer** {@link https://phenomnomnominal.github.io/betterer/docs/tests | test system}.
 *
 * @example
 * ```typescript
 * import { BettererTest } from '@betterer/betterer';
 * import { smaller } from '@betterer/constraints';
 * import glob from 'glob';
 *
 * const test = new BettererTest({
 *   test: () => glob.sync('**\/*').length,
 *   constraint: smaller
 * });
 * ```
 *
 * @param options - The options that define the test.
 */
class BettererTest {
    constructor(options) {
        this._isOnly = false;
        this._isSkipped = false;
        this.config = (0, config_1.createTestConfig)(options);
    }
    /**
     * When `true`, all other tests will be skipped. Calling `only()` will set this to `true`.
     */
    get isOnly() {
        return this._isOnly;
    }
    /**
     * When `true`, this test will be skipped. Calling `skip()` will set this to `true`.
     */
    get isSkipped() {
        return this._isSkipped;
    }
    /**
     * Override the constraint in the test configuration.
     *
     * @param constraintOverride - The new constraint for the test.
     * @returns This {@link @betterer/betterer#BettererTest | `BettererTest`}, so it is chainable.
     */
    constraint(constraintOverride) {
        this.config.constraint = constraintOverride;
        return this;
    }
    /**
     * Override the deadline in the test configuration.
     *
     * @param deadlineOverride - The new deadline for the test.
     * @returns This {@link @betterer/betterer#BettererTest | `BettererTest`}, so it is chainable.
     */
    deadline(deadlineOverride) {
        this.config.deadline = (0, config_1.createDeadline)(Object.assign(Object.assign({}, this.config), { deadline: deadlineOverride }));
        return this;
    }
    /**
     * Override the goal in the test configuration.
     *
     * @param goalOverride - The new goal for the test.
     * @returns This {@link @betterer/betterer#BettererTest | `BettererTest`}, so it is chainable.
     */
    goal(goalOverride) {
        this.config.goal = (0, config_1.createGoal)(Object.assign(Object.assign({}, this.config), { goal: goalOverride }));
        return this;
    }
    /**
     * Run only this test. All other tests will be marked as skipped.
     *
     * @returns This {@link @betterer/betterer#BettererTest | `BettererTest`}, so it is chainable.
     */
    only() {
        this._isOnly = true;
        return this;
    }
    /**
     * Skip this test.
     *
     * @returns This {@link @betterer/betterer#BettererTest | `BettererTest`}, so it is chainable.
     */
    skip() {
        this._isSkipped = true;
        return this;
    }
}
exports.BettererTest = BettererTest;
function isBettererTest(test) {
    if (!test) {
        return false;
    }
    return getBaseName(test.constructor) === BettererTest.name;
}
exports.isBettererTest = isBettererTest;
function getBaseName(input) {
    var _a;
    const proto = Object.getPrototypeOf(input);
    if (proto === Function.prototype) {
        return ((_a = input) === null || _a === void 0 ? void 0 : _a.name) || null;
    }
    return getBaseName(proto);
}
//# sourceMappingURL=test.js.map