"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.regexp = void 0;
const betterer_1 = require("@betterer/betterer");
const errors_1 = require("@betterer/errors");
const fs_1 = require("fs");
/**
 * @public  Use this test to incrementally remove {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/RegExp | `RegExp` }
 * matches from your codebase.
 *
 * @remarks {@link regexp | `regexp`} is a {@link @betterer/betterer#BettererFileTest | `BettererFileTest`},
 * so you can use {@link @betterer/betterer#BettererFileTest.include | `include()`}, {@link @betterer/betterer#BettererFileTest.exclude | `exclude()`},
 * {@link @betterer/betterer#BettererFileTest.only | `only()`}, and {@link @betterer/betterer#BettererFileTest.skip | `skip()`}.
 *
 * @example
 * ```typescript
 * import { regexp } from '@betterer/regexp';
 *
 * export default {
 *   'no hack comments': () =>
 *     regexp(/(\/\/\s*HACK)/i)
 *     .include('./src/*.ts')
 * };
 * ```
 *
 * @param pattern - A {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/RegExp | `RegExp` }
 * pattern to match.
 *
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Will throw if the user doesn't pass `pattern`.
 */
function regexp(pattern) {
    if (!pattern) {
        throw new errors_1.BettererError('for `@betterer/regexp` to work, you need to provide a RegExp, e.g. `/^foo$/`. ❌');
    }
    return new betterer_1.BettererFileTest(async (filePaths, fileTestResult) => {
        if (!filePaths.length) {
            return;
        }
        pattern = new RegExp(pattern.source, pattern.flags.includes('g') ? pattern.flags : `${pattern.flags}g`);
        await Promise.all(filePaths.map(async (filePath) => {
            const fileText = await fs_1.promises.readFile(filePath, 'utf8');
            const matches = getFileMatches(pattern, fileText);
            if (matches.length === 0) {
                return;
            }
            const file = fileTestResult.addFile(filePath, fileText);
            matches.forEach((match) => {
                const [matchText] = match;
                const start = match.index;
                file.addIssue(start, start + matchText.length, 'RegExp match');
            });
        }));
    });
}
exports.regexp = regexp;
function getFileMatches(pattern, fileText) {
    const matches = [];
    let currentMatch;
    do {
        currentMatch = pattern.exec(fileText);
        if (currentMatch) {
            matches.push(currentMatch);
        }
    } while (currentMatch);
    return matches;
}
//# sourceMappingURL=regexp.js.map