"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.run = void 0;
const tslib_1 = require("tslib");
const errors_1 = require("@betterer/errors");
const fs_1 = require("fs");
const path = (0, tslib_1.__importStar)(require("path"));
const GITATTRIBUTES_COMMENT = '# Betterer merge';
const NEW_LINE = '\n';
const MERGE_CONFIG = '[merge "betterer"]';
const MERGE_DIRECTIVE = 'merge=betterer';
async function run(logger, cwd, resultsPath) {
    resultsPath = path.resolve(cwd, resultsPath);
    await logger.progress(`enabling Betterer merge for "${resultsPath}" file...`);
    const gitDir = await findGitRoot(cwd);
    const rootDir = path.dirname(gitDir);
    await Promise.all([gitconfig(logger, gitDir), gitattributes(logger, rootDir, resultsPath)]);
}
exports.run = run;
async function gitconfig(logger, gitDir) {
    const gitconfigPath = path.join(gitDir, 'config');
    let gitconfig = '';
    try {
        gitconfig = await fs_1.promises.readFile(gitconfigPath, 'utf-8');
    }
    catch (_a) {
        // Doesn't matter if it fails...
    }
    let lines = [];
    if (gitconfig) {
        lines = gitconfig.split(NEW_LINE);
        if (lines.includes(MERGE_CONFIG)) {
            await logger.warn(`git merge config already set, moving on...`);
            return;
        }
    }
    const cliPath = require.resolve('@betterer/cli');
    const mergePath = path.resolve(cliPath, '../../bin/betterer-merge');
    const mergeCommand = `\tdriver = ${mergePath} %A %B`;
    lines.push(MERGE_CONFIG, mergeCommand, '');
    try {
        await fs_1.promises.writeFile(gitconfigPath, lines.join(NEW_LINE), 'utf-8');
        await logger.success(`added Betterer merge config to "${gitconfigPath}"!`);
    }
    catch (_b) {
        throw new errors_1.BettererError(`could not write "${gitconfigPath}".`);
    }
}
async function gitattributes(logger, rootDir, resultsPath) {
    const gitattributesPath = path.join(rootDir, '.gitattributes');
    let gitattributes = '';
    try {
        gitattributes = await fs_1.promises.readFile(gitattributesPath, 'utf-8');
    }
    catch (_a) {
        // Doesn't matter if it fails...
    }
    const resultsPathRelative = path.relative(rootDir, resultsPath);
    const mergeAttribute = `${resultsPathRelative} ${MERGE_DIRECTIVE}`;
    let lines = [];
    if (gitattributes) {
        lines = gitattributes.split(NEW_LINE);
        if (lines.includes(mergeAttribute)) {
            await logger.warn(`Betterer merge already enabled for "${resultsPath}", moving on...`);
            return;
        }
        lines = lines.filter((line) => {
            return line !== GITATTRIBUTES_COMMENT && !line.includes(MERGE_DIRECTIVE);
        });
    }
    lines.push(GITATTRIBUTES_COMMENT, mergeAttribute, '');
    try {
        await fs_1.promises.writeFile(gitattributesPath, lines.join(NEW_LINE), 'utf-8');
        await logger.success(`added Betterer merge attribute to "${gitattributesPath}"!`);
    }
    catch (_b) {
        throw new errors_1.BettererError(`could not write "${gitattributesPath}".`);
    }
}
async function findGitRoot(cwd) {
    let dir = cwd;
    while (dir !== path.parse(dir).root) {
        try {
            const gitPath = path.join(dir, '.git');
            await fs_1.promises.access(gitPath);
            return gitPath;
        }
        catch (err) {
            dir = path.join(dir, '..');
        }
    }
    throw new errors_1.BettererError('.git directory not found. Betterer must be used within a git repository.');
}
//# sourceMappingURL=enable-automerge.js.map