"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererTasksLogger = void 0;
const render_1 = require("@betterer/render");
const status_1 = require("./status");
const useTasksState_1 = require("./useTasksState");
const useTimer_1 = require("./useTimer");
/**
 * @public Ink component for rendering the output of a set of {@link BettererTask | `BettererTask`s}.
 * The output will update based on the current status of the tasks.
 */
exports.BettererTasksLogger = (0, render_1.memo)(function BettererTasksLogger(props) {
    const { children, done = () => void 0, exit = true, name, update = defaultUpdate, timer = true } = props;
    const app = (0, render_1.useApp)();
    const [time, clear] = (0, useTimer_1.useTimer)(timer);
    const [state, tasks] = (0, useTasksState_1.useTasksState)();
    const { startTime, endTime, errors } = state;
    (0, render_1.useEffect)(() => {
        if (endTime != null) {
            clear();
        }
    }, [endTime, clear]);
    const result = `${update(state)}`;
    let status = ['🌟', 'whiteBright', result];
    if (errors > 0) {
        status = ['💥', 'redBright', result];
    }
    else if (endTime !== null) {
        status = ['🎉', 'greenBright', result];
    }
    const hasChildren = Array.isArray(children) ? children.length : !!children;
    if (!hasChildren || endTime != null) {
        if (exit) {
            setImmediate(() => app.exit());
        }
        done();
    }
    const label = timer ? ` (${getTime(startTime, endTime || time)}ms)` : '';
    return (render_1.React.createElement(useTasksState_1.BettererTasksContext.Provider, { value: [state, tasks] },
        render_1.React.createElement(render_1.Box, { flexDirection: "column" },
            render_1.React.createElement(status_1.BettererTaskStatus, { name: `${name}${label}`, status: status }),
            children)));
});
const FORMATTER = Intl.NumberFormat();
function getTime(startTime, time) {
    return FORMATTER.format(Math.floor(time - startTime));
}
function defaultUpdate(state) {
    const { done, errors, running } = state;
    const runningStatus = running ? `${tasks(running)} running... ` : '';
    const doneStatus = done ? `${tasks(done)} done! ` : '';
    const errorStatus = errors ? `${tasks(errors)} errored! ` : '';
    return `${runningStatus}${doneStatus}${errorStatus}`;
}
function tasks(n) {
    return `${n} ${n === 1 ? 'task' : 'tasks'}`;
}
//# sourceMappingURL=TasksLogger.js.map