"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererFileResolverΩ = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const minimatch_1 = (0, tslib_1.__importDefault)(require("minimatch"));
const path = (0, tslib_1.__importStar)(require("path"));
const utils_1 = require("../utils");
class BettererFileResolverΩ {
    constructor(_baseDirectory = null, _versionControl = null) {
        this._baseDirectory = _baseDirectory;
        this._versionControl = _versionControl;
        this._excluded = [];
        this._included = [];
        this._includedResolved = null;
        this._validatedFilePaths = [];
        this._validatedFilePathsMap = {};
    }
    get baseDirectory() {
        (0, assert_1.default)(this._baseDirectory);
        return this._baseDirectory;
    }
    get versionControl() {
        (0, assert_1.default)(this._versionControl);
        return this._versionControl;
    }
    init(directory, versionControl) {
        this._baseDirectory = directory;
        this._versionControl = versionControl;
    }
    async validate(filePaths) {
        // If `include()` was never called, just filter the given list:
        if (!this._included.length) {
            const validFilePaths = await this.versionControl.filterIgnored(filePaths);
            return validFilePaths.filter((filePath) => !this._isExcluded(filePath));
        }
        await this._update();
        return filePaths.filter((filePath) => this._validatedFilePathsMap[filePath]);
    }
    resolve(...pathSegments) {
        return (0, utils_1.normalisedPath)(path.resolve(this.baseDirectory, ...pathSegments));
    }
    include(...includePatterns) {
        this._included = [...this._included, ...(0, utils_1.flatten)(includePatterns)];
        return this;
    }
    exclude(...excludePatterns) {
        if (!this._included.length) {
            this.include('**/*');
        }
        this._excluded = [...this._excluded, ...(0, utils_1.flatten)(excludePatterns)];
        return this;
    }
    async files() {
        await this._update();
        return this._validatedFilePaths;
    }
    async _update() {
        this._validatedFilePathsMap = {};
        const filePaths = await this.versionControl.getFilePaths();
        filePaths.forEach((filePath) => {
            this._validatedFilePathsMap[filePath] = this._isIncluded(filePath);
        });
        this._validatedFilePaths = Object.keys(this._validatedFilePathsMap).filter((filePath) => {
            const included = this._validatedFilePathsMap[filePath] && !this._isExcluded(filePath);
            this._validatedFilePathsMap[filePath] = included;
            return included;
        });
    }
    _isIncluded(filePath) {
        if (!this._includedResolved) {
            this._includedResolved = this._included.map((pattern) => this.resolve(pattern));
        }
        return this._includedResolved.some((pattern) => (0, minimatch_1.default)(filePath, pattern));
    }
    _isExcluded(filePath) {
        return this._excluded.some((exclude) => exclude.test(filePath));
    }
}
exports.BettererFileResolverΩ = BettererFileResolverΩ;
//# sourceMappingURL=file-resolver.js.map