"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.workerRequire = exports.createWorkerRequire = void 0;
const tslib_1 = require("tslib");
const callsite_1 = tslib_1.__importDefault(require("callsite"));
const path = tslib_1.__importStar(require("path"));
const worker_threads_1 = require("worker_threads");
const comlink_1 = require("./vendor/comlink");
const node_adapter_1 = tslib_1.__importDefault(require("./vendor/node-adapter"));
const constants_1 = require("./constants");
const handlers_1 = require("./handlers");
handlers_1.setHandlers();
function createWorkerRequire(id, options) {
    const [, call] = callsite_1.default();
    const sourcePath = call.getFileName();
    return () => wrapModule(sourcePath, id, options);
}
exports.createWorkerRequire = createWorkerRequire;
function workerRequire(id, options = { cache: true }) {
    const [, call] = callsite_1.default();
    const sourcePath = call.getFileName();
    return wrapModule(sourcePath, id, options);
}
exports.workerRequire = workerRequire;
function wrapModule(sourcePath, id, options = { cache: true }) {
    const idPath = path.resolve(path.dirname(sourcePath), id);
    const requirePath = require.resolve(idPath);
    const isEnabled = process.env.WORKER_REQUIRE !== 'false';
    const handle = isEnabled ? getHandle(requirePath, options) : null;
    function destroy() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            if (handle) {
                yield destroyWorker(requirePath, handle);
            }
        });
    }
    const nonProxyResults = new Map([
        ['destroy', destroy],
        [constants_1.TO_CLONEABLE, null],
        ['then', null],
    ]);
    function createProxy(path, handle) {
        return new Proxy(() => void 0, {
            get(_, name) {
                if (nonProxyResults.has(name)) {
                    return nonProxyResults.get(name);
                }
                return createProxy([...path, name], handle);
            },
            apply(_, __, args) {
                // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment
                let obj = handle ? handle.remote : require(requirePath);
                while (path.length > 1) {
                    // eslint-disable-next-line @typescript-eslint/no-unsafe-assignment, @typescript-eslint/no-unsafe-member-access, @typescript-eslint/no-non-null-assertion
                    obj = obj[path.shift()];
                }
                const [prop] = path;
                try {
                    // eslint-disable-next-line @typescript-eslint/no-unsafe-member-access, @typescript-eslint/no-unsafe-call
                    return Promise.resolve(obj[prop](...args));
                }
                catch (error) {
                    return Promise.reject(error);
                }
            },
        });
    }
    return createProxy([], handle);
}
const HANDLES = new Map();
function getHandle(requirePath, options) {
    const cached = HANDLES.get(requirePath);
    if (options.cache && cached) {
        const [cachedItem] = cached;
        return cachedItem;
    }
    return createWorker(requirePath);
}
function createWorker(requirePath) {
    const cached = HANDLES.get(requirePath) || [];
    const worker = new worker_threads_1.Worker(constants_1.WORKER_PATH, { workerData: requirePath });
    const remote = comlink_1.wrap(node_adapter_1.default(worker));
    const handle = { remote, worker };
    cached.push(handle);
    HANDLES.set(requirePath, cached);
    return handle;
}
function destroyWorker(requirePath, handle) {
    return tslib_1.__awaiter(this, void 0, void 0, function* () {
        const cached = HANDLES.get(requirePath);
        if (!cached) {
            return;
        }
        cached.splice(cached.indexOf(handle), 1);
        if (cached.length === 0) {
            HANDLES.delete(requirePath);
        }
        handle.remote[comlink_1.releaseProxy]();
        yield handle.worker.terminate();
    });
}
//# sourceMappingURL=worker-require.js.map