#ifndef PQ_CLONE_ITEM_H
#define PQ_CLONE_ITEM_H

/* Copyright (c) 2013, 2020, Oracle and/or its affiliates. All rights reserved.
   Copyright (c) 2022, Huawei Technologies Co., Ltd.
   Copyright (c) 2023, 2025, GreatDB Software Co., Ltd.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License, version 2.0,
   as published by the Free Software Foundation.

   This program is also distributed with certain software (including
   but not limited to OpenSSL) that is licensed under separate terms,
   as designated in a particular file or component or in included license
   documentation.  The authors of MySQL hereby grant you an additional
   permission to link the program and your derivative works with the
   separately licensed software that they have included with MySQL.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License, version 2.0, for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301  USA */

#include "item_geofunc.h"
#include "item_inetfunc.h"
#include "item_pfs_func.h"
#include "mem_root_deque.h"
#include "sql/item.h"
#include "sql/item_cmpfunc.h"
#include "sql/item_regexp_func.h"
#include "sql/item_strfunc.h"
#include "sql/item_sum.h"
#include "sql/item_timefunc.h"
#include "sql/log.h"
#include "sql/parse_tree_items.h"
#include "sql/parse_tree_nodes.h"

#define CHECK_TYPE(T)                                                        \
  if (typeid(*this) != typeid(T)) {                                          \
    return nullptr;                                                          \
  } else if (DBUG_EVALUATE_IF("simulate_item_type_mismatch", true, false)) { \
    sql_print_warning("simulate_item_type_mismatch, type:%s, SQL=%s ",       \
                      typeid(T).name(), thd->query().str);                   \
    return nullptr;                                                          \
  }

#define COPY_FROM_SUPER(D, B)                                    \
  if (B::pq_copy_from(thd, select, item)) {                      \
    return true;                                                 \
  }                                                              \
  D *orig_item MY_ATTRIBUTE((unused)) = dynamic_cast<D *>(item); \
  assert(orig_item);

#define COPY_SELF_ATTR(OBJ)                           \
  if (!OBJ || OBJ->pq_copy_from(thd, select, this)) { \
    return nullptr;                                   \
  }

#define PQ_CLONE_DEF(T)                              \
  Item *T::pq_clone(THD *thd, Query_block *select) { \
    CHECK_TYPE(T)                                    \
    T *new_item = nullptr;

#define PQ_CLONE_RETURN    \
  COPY_SELF_ATTR(new_item) \
  return new_item;         \
  }

#define PQ_CLONE_ARGS                              \
  mem_root_deque<Item *> item_list(thd->mem_root); \
  for (uint i = 0; i < arg_count; i++) {           \
    Item *arg = args[i]->pq_clone(thd, select);    \
    if (arg == nullptr) return nullptr;            \
    item_list.push_back(arg);                      \
  }

#define PQ_COPY_FROM_DEF(D, B)                                      \
  bool D::pq_copy_from(THD *thd, Query_block *select, Item *item) { \
    COPY_FROM_SUPER(D, B)

#define PQ_COPY_FROM_RETURN \
  return false;             \
  }

#define PQ_REBUILD_SUM_DEF(T)                                     \
  Item_sum *T::pq_rebuild_sum_func(THD *thd, Query_block *select, \
                                   Item *item) {                  \
    CHECK_TYPE(T)                                                 \
    T *new_item = nullptr;

#define PQ_REBUILD_SUM_RETURN \
  COPY_SELF_ATTR(new_item)    \
  return new_item;            \
  }

#define ARG0 copy_args[0]
#define ARG1 copy_args[1]
#define ARG2 copy_args[2]
#define ARG3 copy_args[3]
#define ARG4 copy_args[4]
#define COPY_FUNC_ITEM(T, ...)                       \
  Item *T::pq_clone(THD *thd, Query_block *select) { \
    CHECK_TYPE(T)                                    \
    Item *copy_args[5];                              \
    assert(arg_count < 5);                           \
    for (uint i = 0; i < arg_count; i++) {           \
      copy_args[i] = args[i]->pq_clone(thd, select); \
      if (copy_args[i] == nullptr) {                 \
        return nullptr;                              \
      }                                              \
    }                                                \
    Item *new_item = nullptr;                        \
    new_item = new (thd->mem_root) T(__VA_ARGS__);   \
    COPY_SELF_ATTR(new_item)                         \
    return new_item;                                 \
  }

Item *Item::pq_clone(THD *thd MY_ATTRIBUTE((unused)),
                     Query_block *select MY_ATTRIBUTE((unused))) {
  // sql_print_warning(
  //     "Item type %s's deep copy method is not implemented, "
  //     "will not use parallel query, SQL= %s",
  //     typeid(*this).name(), thd->query().str);
  //  assert(DBUG_EVALUATE_IF("simulate_no_item_copy_function", true, false) ||
  //         false);
  return nullptr;
}

bool Item::pq_copy_from(THD *thd, Query_block *select MY_ATTRIBUTE((unused)),
                        Item *item) {
  cmp_context = item->cmp_context;
  marker = item->marker;

  collation = item->collation;
  item_name.copy(item->item_name.ptr(), item->item_name.length(),
                 system_charset_info, item->item_name.is_autogenerated());
  orig_name.copy(item->orig_name.ptr(), item->orig_name.length(),
                 system_charset_info, item->orig_name.is_autogenerated());
  decimals = item->decimals;
  m_accum_properties = item->m_accum_properties;
  m_data_type = item->m_data_type;
  m_is_window_function = item->m_is_window_function;
  max_length = item->max_length;
  m_nullable = item->is_nullable();
  null_value = item->null_value;
  str_value = item->str_value;
  fixed = item->fixed;
  hidden = item->hidden;
  unsigned_flag = item->unsigned_flag;
  outer_joined = item->outer_joined;
  if (item->type() == Item::FUNC_ITEM) {
    if (dynamic_cast<Item_func *>(this)->functype() !=
        Item_func::Functype::ROLLUP_GROUP_ITEM_FUNC) {
      dynamic_cast<Item_func *>(this)->resolve_type(thd);
    }
  }

#ifndef NDEBUG
  contextualized = item->contextualized;
#endif
  unsigned_flag = item->unsigned_flag;

  if (!pq_alloc_item && item->pq_alloc_item) thd->add_item(this);

  return false;
}

/* Item_basic_constant start */
PQ_COPY_FROM_DEF(Item_basic_constant, Item) {
  if (orig_item != nullptr) {
    used_table_map = orig_item->used_table_map;
  }
}
PQ_COPY_FROM_RETURN

/* Item_cache start */
PQ_COPY_FROM_DEF(Item_cache, Item_basic_constant) {
  if (orig_item != nullptr) {
    used_table_map = orig_item->used_table_map;
    cached_field = orig_item->cached_field;
  }
  if (orig_item != nullptr && orig_item->example != nullptr) {
    Item *example_arg = orig_item->example->pq_clone(thd, select);
    if (example_arg == nullptr) return true;
    if (!example_arg->fixed) {
      example_arg->fix_fields(thd, &example_arg);
    }
    setup(example_arg);
  }
  value_cached = down_cast<Item_cache *>(item)->value_cached;
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_cache_datetime) {
  new_item = new (thd->mem_root) Item_cache_datetime(data_type());
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_cache_decimal) {
  new_item = new (thd->mem_root) Item_cache_decimal();
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_cache_int) {
  new_item = new (thd->mem_root) Item_cache_int();
  if (origin_item) {
    new_item->example = origin_item->pq_clone(thd, select);
    if (new_item->example == nullptr) {
      return nullptr;
    }
  }
  new_item->set_value(value);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_cache_real) {
  new_item = new (thd->mem_root) Item_cache_real();
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_cache_row) {
  new_item = new (thd->mem_root) Item_cache_row();
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_cache_str) {
  const Item *item = static_cast<const Item *>(this);
  new_item = new (thd->mem_root) Item_cache_str(item);
  if (value_cached) {
    new_item->set_value(value);
  }
}
PQ_CLONE_RETURN
/* Item_cache end */

/* Item_hex_string start */
PQ_CLONE_DEF(Item_hex_string) {
  new_item = new (thd->mem_root) Item_hex_string(POS());
}
PQ_CLONE_RETURN

// TOOD str_value copyed twice
PQ_CLONE_DEF(Item_bin_string) {
  new_item =
      new (thd->mem_root) Item_bin_string(str_value.ptr(), str_value.length());
}
PQ_CLONE_RETURN
/* Item_hex_string end */

/* Item_null start */
PQ_CLONE_DEF(Item_null) { new_item = new (thd->mem_root) Item_null(POS()); }
PQ_CLONE_RETURN
/* Item_null end */

/* Item_num start */
PQ_CLONE_DEF(Item_int_with_ref) {
  Item *pq_ref = ref->pq_clone(thd, select);
  if (!pq_ref) return nullptr;
  new_item = new (thd->mem_root)
      Item_int_with_ref(pq_ref->data_type(), value, pq_ref, unsigned_flag);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_datetime_with_ref) {
  if (origin_item) {
    return origin_item->pq_clone(thd, select);
  }
  Item *pq_ref = ref->pq_clone(thd, select);
  if (!pq_ref) return nullptr;

  new_item = new (thd->mem_root)
      Item_datetime_with_ref(pq_ref->data_type(), decimals, value, pq_ref);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_time_with_ref) {
  if (origin_item) {
    return origin_item->pq_clone(thd, select);
  }
  Item *pq_ref = ref->pq_clone(thd, select);
  if (!pq_ref) return nullptr;

  new_item = new (thd->mem_root) Item_time_with_ref(decimals, value, pq_ref);
}
PQ_CLONE_RETURN
/* Item_num end */

/* Item_string start */
PQ_CLONE_DEF(Item_string) {
  if (origin_item) return origin_item->pq_clone(thd, select);

  new_item = new (thd->mem_root) Item_string(
      static_cast<Name_string>(item_name), str_value.ptr(), str_value.length(),
      collation.collation, collation.derivation, collation.repertoire);
  if (new_item) {
    new_item->set_cs_specified(m_cs_specified);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_static_string_func) {
  if (origin_item) return origin_item->pq_clone(thd, select);

  new_item = new (thd->mem_root)
      Item_static_string_func(func_name, str_value.ptr(), str_value.length(),
                              collation.collation, collation.derivation);
}
PQ_CLONE_RETURN
/* Item_string end */
/* Item_basic_constant end */

/* Item_ident start */
PQ_COPY_FROM_DEF(Item_ident, Item) {
  DBUG_EXECUTE_IF("simulate_item_clone_attr_copy_error", return true;);

  if (select == nullptr) {
    context = down_cast<Item_ident *>(item)->context;
  } else {
    context = &select->context;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_field) {
  DBUG_EXECUTE_IF("simulate_item_clone_error", return nullptr;);
  DBUG_EXECUTE_IF("simulate_no_item_copy_function",
                  return Item::pq_clone(thd, select););

  new_item =
      new (thd->mem_root) Item_field(POS(), db_name, table_name, field_name);
  new_item->original_field = field;
  new_item->field = field;
  new_item->field_index = field_index;
  new_item->table_ref = table_ref;
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_default_value) {
  Item *new_arg = nullptr;
  if (arg) {
    new_arg = arg->pq_clone(thd, select);
    if (new_arg == nullptr) return nullptr;
  }
  new_item = new (thd->mem_root) Item_default_value(POS(), new_arg);
  new_item->set_field(field);
}
PQ_CLONE_RETURN

int find_ref_in_table(Table_ref *tl, Item **ref) {
  int count = tl->field_translation_end - tl->field_translation;
  if (count <= 0) {
    return -1;
  }
  for (int i = 0; i < count; i++) {
    if (*ref == tl->field_translation[i].item) {
      return i;
    }
  }
  return -1;
}

/**
   Item_aggregate_ref wil be created from ref in setup_fields() afterwards,
   so clone ref for the item.
*/
Item *Item_aggregate_ref::pq_clone(class THD *thd, class Query_block *select) {
  Item *item_ref = (*m_ref_item)->pq_clone(thd, select);
  if (item_ref == nullptr) {
    return nullptr;
  }

  return item_ref;
}

Item *Item_ref::pq_clone(class THD *thd, class Query_block *select) {
  /*
   * c1: (Name_resolution_context, db_name, table_name, field_name)
   * c2: (pos, db_name, table_name, field_name)
   * c3: (context, ref, db_name, table_name, field_name)
   * c4: (thd, ref_item)
   */
  Item_ref *new_item = nullptr;
  Name_resolution_context *new_context = &select->context;
  if (select == nullptr) {
    new_context = context;
  }

  if (copy_type == WITH_CONTEXT)
    new_item = new (thd->mem_root)
        Item_ref(new_context, db_name, table_name, field_name);
  else if (copy_type == WITHOUT_CONTEXT)
    new_item =
        new (thd->mem_root) Item_ref(POS(), db_name, table_name, field_name);
  else if (copy_type == WITH_CONTEXT_REF) {
    return (*m_ref_item)->pq_clone(thd, select);
  } else {
    assert(copy_type == WITH_REF_ONLY);
    new_item = new (thd->mem_root) Item_ref(thd, this);
  }
  if (new_item == nullptr || new_item->pq_copy_from(thd, select, this))
    return nullptr;

  new_item->context = &select->context;
  return new_item;
}

Item *Item_view_ref::pq_clone(class THD *thd, class Query_block *select) {
  Item_ref *new_item = nullptr;
  Item **new_ref = (Item **)thd->mem_root->Alloc(sizeof(Item *));
  *new_ref = nullptr;
  if (m_ref_item != nullptr && ref_item() != nullptr) {
    *new_ref = ref_item()->pq_clone(thd, select);
    if (!*new_ref) return nullptr;
  }

  new_item = new (thd->mem_root)
      Item_view_ref(context, new_ref, db_name, table_name, m_orig_table_name,
                    field_name, cached_table);
  return new_item;
}

Item *Item_param::pq_clone(class THD *thd, class Query_block *) {
  Item_param *new_item = nullptr;
  new_item = new (thd->mem_root) Item_param(POS(), thd->mem_root, pos_in_query);
  new_item->copy_param_actual_type(this);
  new_item->set_result_type(m_result_type);
  new_item->str_value_ptr = str_value_ptr;
  new_item->decimal_value = decimal_value;
  new_item->value = value;
  return new_item;
}

PQ_CLONE_DEF(Item_name_const) {
  Item *name_arg, *val_arg;
  if (name_item == nullptr) {
    name_arg = nullptr;
  } else {
    name_arg = name_item->pq_clone(thd, select);
    if (name_arg == nullptr) return nullptr;
  }
  if (value_item == nullptr) {
    val_arg = nullptr;
  } else {
    val_arg = value_item->pq_clone(thd, select);
    if (val_arg == nullptr) return nullptr;
  }
  new_item = new (thd->mem_root) Item_name_const(POS(), name_arg, val_arg);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_name_const, Item) {
  if (orig_item != nullptr) {
    valid_args = orig_item->valid_args;
  }
}
PQ_COPY_FROM_RETURN

/* Item_result_field start */
/* Item_func start */
PQ_COPY_FROM_DEF(Item_func, Item_result_field) {
  if (orig_item != nullptr) {
    null_on_null = orig_item->null_on_null;
    used_tables_cache = orig_item->used_tables_cache;
    not_null_tables_cache = orig_item->not_null_tables_cache;
  }
}
PQ_COPY_FROM_RETURN

/* Item_func_bit start */
PQ_COPY_FROM_DEF(Item_func_bit, Item_func) {
  if (orig_item != nullptr) {
    hybrid_type = orig_item->hybrid_type;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(PTI_literal_underscore_charset_hex_num) {
  LEX_STRING str = {const_cast<char *>(str_value.ptr()), str_value.length()};
  new_item = new (thd->mem_root)
      PTI_literal_underscore_charset_hex_num(POS(), collation.collation, str);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_bit_neg, POS(), ARG0)

COPY_FUNC_ITEM(Item_func_bit_and, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_bit_or, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_bit_xor, POS(), ARG0, ARG1)

COPY_FUNC_ITEM(Item_func_shift_left, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_shift_right, POS(), ARG0, ARG1)
/* Item_func_bit end */

PQ_CLONE_DEF(Item_func_case) {
  PQ_CLONE_ARGS
  new_item =
      new (thd->mem_root) Item_func_case(POS(), &item_list, nullptr, nullptr);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_case, Item_func) {
  if (orig_item != nullptr) {
    first_expr_num = orig_item->first_expr_num;
    else_expr_num = orig_item->else_expr_num;
    cached_result_type = orig_item->cached_result_type;
    left_result_type = orig_item->left_result_type;
    ncases = orig_item->ncases;
    cmp_type = orig_item->cmp_type;
    orafun = orig_item->orafun;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_if, ARG0, ARG1, ARG2)
COPY_FUNC_ITEM(Item_func_month, POS(), ARG0)

/* Item_func_coalesce start */
PQ_CLONE_DEF(Item_func_coalesce) {
  assert(arg_count < 3);
  Item *new_args[2] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }
  if (arg_count == 1) {
    new_item = new (thd->mem_root) Item_func_coalesce(POS(), new_args[0]);
  } else if (arg_count == 2) {
    new_item =
        new (thd->mem_root) Item_func_coalesce(POS(), new_args[0], new_args[1]);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_ora_nvl2) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_ora_nvl2(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_any_value, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_ifnull, POS(), ARG0, ARG1)
/* Item_func_coalesce end */

/* Item_func_min_max start */
PQ_CLONE_DEF(Item_func_max) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_max(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_min) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_min(POS(), &pt_item_list);
}
PQ_CLONE_RETURN
/* Item_func_min_max end */

/* Item_func_num1 start */
COPY_FUNC_ITEM(Item_func_abs, POS(), ARG0)

COPY_FUNC_ITEM(Item_func_ceiling, ARG0)
COPY_FUNC_ITEM(Item_func_floor, ARG0)

COPY_FUNC_ITEM(Item_func_neg, ARG0)
COPY_FUNC_ITEM(Item_func_round, ARG0, ARG1, truncate)
COPY_FUNC_ITEM(Item_func_ora_round, ARG0, ARG1, truncate)
COPY_FUNC_ITEM(Item_func_trunc, POS(), ARG0, ARG1, default_arg)
/* Item_func_num1 end */

/* Item_num_op start */
COPY_FUNC_ITEM(Item_func_plus, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_minus, ARG0, ARG1)

COPY_FUNC_ITEM(Item_func_div, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mod, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mul, ARG0, ARG1)
/* Item_num_op end */

/* Item_func_regexp start */
PQ_CLONE_DEF(Item_func_regexp_instr) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_regexp_instr(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_regexp_like) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_regexp_like(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_regexp_substr) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_regexp_substr(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_regexp_replace) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_regexp_replace(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(item_func_regexp_count) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) item_func_regexp_count(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

/* Item_func_regexp end */

PQ_CLONE_DEF(Item_typeto_number) {
  PQ_CLONE_ARGS
  new_item =
      new (thd->mem_root) Item_typeto_number(POS(), args[0], pq_len, pq_dec);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_connect_by_func_level) {
  new_item = new (thd->mem_root) Item_connect_by_func_level(POS());
}
PQ_CLONE_RETURN

/* Item_func_weekday start */
PQ_CLONE_DEF(Item_func_weekday) {
  PQ_CLONE_ARGS
  new_item = new (thd->mem_root)
      Item_func_weekday(POS(), item_list[0], this->odbc_type);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_dayname, POS(), ARG0)
/* Item_func_weekday end */

/* Item_int_func start */
/* Item_bool_func2 start */
COPY_FUNC_ITEM(Item_func_eq, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_equal, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_ge, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_gt, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_le, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_lt, ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_ne, ARG0, ARG1)

PQ_CLONE_DEF(Item_func_like) {
  Item *arg[3];
  for (uint i = 0; i < arg_count; i++) {
    arg[i] = args[i]->pq_clone(thd, select);
    if (arg[i] == nullptr) return nullptr;
  }

  if (arg_count == 2) {
    new_item = new (thd->mem_root) Item_func_like(arg[0], arg[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root) Item_func_like(arg[0], arg[1], arg[2]);
  } else {
    sql_print_warning("arg_count is wrong!");
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_nullif, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrcontains, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_strcmp, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_xor, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_contains, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrcoveredby, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrcovers, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_crosses, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_disjoint, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_equals, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_intersects, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_overlaps, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_touches, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_within, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrwithin, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrtouches, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbroverlaps, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrintersects, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrequals, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_mbrdisjoint, POS(), ARG0, ARG1)
/* Item_bool_func2 end */

/* geometry func start */
COPY_FUNC_ITEM(Item_func_point, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_spatial_decomp, POS(), ARG0, decomp_func)
COPY_FUNC_ITEM(Item_func_spatial_decomp_n, POS(), ARG0, ARG1, decomp_func_n)
COPY_FUNC_ITEM(Item_func_st_transform, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_latitude_mutator, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_latitude_observer, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_st_longitude_mutator, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_longitude_observer, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_st_x_mutator, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_x_observer, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_st_y_mutator, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_y_observer, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_swap_xy, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_numgeometries, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_numinteriorring, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_numpoints, POS(), ARG0)

COPY_FUNC_ITEM(Item_func_st_area, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_st_srid_mutator, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_st_srid_observer, POS(), ARG0)
PQ_CLONE_DEF(Item_func_st_buffer) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_st_buffer(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_st_length) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_st_length(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_geometry_from_text) {
  PQ_CLONE_ARGS
  if (arg_count == 1) {
    new_item = new (thd->mem_root)
        Item_func_geometry_from_text(POS(), item_list[0], m_functype);
  } else if (arg_count == 2) {
    new_item = new (thd->mem_root) Item_func_geometry_from_text(
        POS(), item_list[0], item_list[1], m_functype);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root) Item_func_geometry_from_text(
        POS(), item_list[0], item_list[1], item_list[2], m_functype);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_geometry_from_wkb) {
  PQ_CLONE_ARGS
  if (arg_count == 1) {
    new_item = new (thd->mem_root)
        Item_func_geometry_from_wkb(POS(), item_list[0], m_functype);
  } else if (arg_count == 2) {
    new_item = new (thd->mem_root) Item_func_geometry_from_wkb(
        POS(), item_list[0], item_list[1], m_functype);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root) Item_func_geometry_from_wkb(
        POS(), item_list[0], item_list[1], item_list[2], m_functype);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_as_wkt) {
  PQ_CLONE_ARGS
  if (arg_count == 1) {
    new_item = new (thd->mem_root) Item_func_as_wkt(POS(), item_list[0]);
  } else if (arg_count == 2) {
    new_item =
        new (thd->mem_root) Item_func_as_wkt(POS(), item_list[0], item_list[1]);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_as_wkb) {
  PQ_CLONE_ARGS
  if (arg_count == 1) {
    new_item = new (thd->mem_root) Item_func_as_wkb(POS(), item_list[0]);
  } else if (arg_count == 2) {
    new_item =
        new (thd->mem_root) Item_func_as_wkb(POS(), item_list[0], item_list[1]);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_spatial_collection) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root)
      Item_func_spatial_collection(POS(), &pt_item_list, coll_type, item_type);
}
PQ_CLONE_RETURN

/* geometry func end */

/* Item_cond start */
PQ_COPY_FROM_DEF(Item_cond, Item_bool_func) {
  Item *list_item;
  List_iterator_fast<Item> list_it(orig_item->list);
  while ((list_item = list_it++)) {
    Item *arg = list_item->pq_clone(thd, select);
    if (arg == nullptr) return true;
    list.push_back(arg);
  }
  if (orig_item != nullptr) {
    abort_on_null = orig_item->abort_on_null;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_cond_and) { new_item = new (thd->mem_root) Item_cond_and(); }
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_cond_and, Item_cond) {
  if (orig_item != nullptr) {
    cond_equal.max_members = orig_item->cond_equal.max_members;
  }
  Item_equal *item_equal;
  List_iterator_fast<Item_equal> it(orig_item->cond_equal.current_level);
  for (size_t i = 0; (item_equal = it++); i++) {
    Item_equal *new_item_equal =
        dynamic_cast<Item_equal *>(item_equal->pq_clone(thd, select));
    if (new_item_equal == nullptr) return true;
    cond_equal.current_level.push_back(new_item_equal);
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_cond_or) { new_item = new (thd->mem_root) Item_cond_or(); }
PQ_CLONE_RETURN
/* Item_cond end */

PQ_CLONE_DEF(Item_equal) { new_item = new (thd->mem_root) Item_equal(); }
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_equal, Item_bool_func) {
  Item_field *item_field;
  List_iterator_fast<Item_field> it(orig_item->fields);
  for (size_t i = 0; (item_field = it++); i++) {
    Item_field *new_field =
        dynamic_cast<Item_field *>(item_field->pq_clone(thd, select));
    if (new_field == nullptr) return true;
    fields.push_back(new_field);
  }
  if (orig_item != nullptr && orig_item->m_const_arg != nullptr) {
    m_const_arg = orig_item->m_const_arg->pq_clone(thd, select);
    if (m_const_arg == nullptr) {
      return true;
    }
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_true, POS())
COPY_FUNC_ITEM(Item_func_false, POS())

COPY_FUNC_ITEM(Item_func_isnotnull, ARG0)

PQ_CLONE_DEF(Item_func_isnull) {
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;
  new_item = new (thd->mem_root) Item_func_isnull(POS(), arg);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_isnull, Item_bool_func) {
  if (orig_item != nullptr) {
    cached_value = orig_item->cached_value;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_json_schema_valid, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_not, ARG0)

PQ_CLONE_DEF(Item_func_truth) {
  PQ_CLONE_ARGS
  new_item =
      new (thd->mem_root) Item_func_truth(POS(), item_list[0], truth_test);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_extract) {
  PQ_CLONE_ARGS
  new_item =
      new (thd->mem_root) Item_extract(POS(), this->int_type, item_list[0]);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_extract, Item_int_func) {
  if (orig_item != nullptr) {
    date_value = orig_item->date_value;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_typecast_year) {
  PQ_CLONE_ARGS
  new_item = new (thd->mem_root) Item_typecast_year(POS(), item_list[0]);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_ascii, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_bit_count, POS(), ARG0)

PQ_CLONE_DEF(Item_func_char_length) {
  assert(arg_count == 1);
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;
  new_item = new (thd->mem_root) Item_func_char_length(POS(), arg);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_char_length, Item_int_func) {
  if (orig_item != nullptr) {
    value.copy(orig_item->value);
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_coercibility, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_crc32, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_dayofmonth, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_dayofyear, POS(), ARG0)

PQ_CLONE_DEF(Item_func_field) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_field(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_find_in_set, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_get_lock, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_hour, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_inet_aton, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_div_int, POS(), ARG0, ARG1)

PQ_CLONE_DEF(Item_func_interval) {
  assert(arg_count == 1 && args[0]->type() == Item::ROW_ITEM);
  Item_row *row = down_cast<Item_row *>(args[0]->pq_clone(thd, select));
  if (nullptr == row) return nullptr;
  new_item = new (thd->mem_root) Item_func_interval(POS(), row);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_interval, Item_int_func) {
  if (orig_item != nullptr) {
    use_decimal_comparison = orig_item->use_decimal_comparison;
    intervals = orig_item->intervals;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_func_json_contains) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_json_contains(thd, POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_json_depth, POS(), ARG0)

PQ_CLONE_DEF(Item_func_last_insert_id) {
  Item *item_arg = nullptr;
  if (arg_count == 1) {
    item_arg = args[0]->pq_clone(thd, select);
    if (item_arg == nullptr) {
      return nullptr;
    }
  }

  if (arg_count == 0) {
    new_item = new (thd->mem_root) Item_func_last_insert_id(POS());
  } else if (arg_count == 1) {
    new_item = new (thd->mem_root) Item_func_last_insert_id(POS(), item_arg);
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_length, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_lengthb, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_bit_length, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_minute, POS(), ARG0)

PQ_CLONE_DEF(Item_func_locate) {
  assert(arg_count < 4);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2) {
    new_item =
        new (thd->mem_root) Item_func_locate(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_locate(POS(), new_args[0], new_args[1], new_args[2]);
  }
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_locate, Item_int_func) {}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_instr, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_microsecond, POS(), ARG0)

PQ_CLONE_DEF(Item_func_ora_instr) {
  assert(arg_count < 5);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_ora_instr(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 4) {
    new_item = new (thd->mem_root) Item_func_ora_instr(
        POS(), new_args[0], new_args[1], new_args[2], new_args[3]);
  }
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_opt_neg, Item_int_func) {
  if (orig_item != nullptr) {
    negated = orig_item->negated;
    pred_level = orig_item->pred_level;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_between, POS(), ARG0, ARG1, ARG2, negated)

PQ_CLONE_DEF(Item_func_in) {
  PT_select_item_list pt_item{POS()};
  for (uint i = 0; i < arg_count; i++) {
    Item *arg = args[i]->pq_clone(thd, select);
    if (arg == nullptr) return nullptr;
    pt_item.value.push_back(arg);
  }
  new_item = new (thd->mem_root) Item_func_in(POS(), &pt_item, negated);
}
PQ_CLONE_RETURN

// COPY_FUNC_ITEM(Item_func_ord, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_period_add, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_period_diff, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_quarter, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_second, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_sleep, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_time_to_sec, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_timestamp_diff, POS(), ARG0, ARG1, int_type)
COPY_FUNC_ITEM(Item_func_to_days, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_to_seconds, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_uncompressed_length, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_week, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_year, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_yearweek, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_typecast_signed, POS(), ARG0)
COPY_FUNC_ITEM(Item_typecast_unsigned, POS(), ARG0)
/* Item_int_func end */

/* Item_real_func start */
/* Item_dec_func start*/
COPY_FUNC_ITEM(Item_func_sin, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_sqrt, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_cos, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_tan, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_cot, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_pow, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_ln, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_log2, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_log10, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_asin, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_acos, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_exp, POS(), ARG0)

PQ_CLONE_DEF(Item_func_atan) {
  Item *item_args[2];
  assert(arg_count < 3);
  for (uint i = 0; i < arg_count; i++) {
    item_args[i] = args[i]->pq_clone(thd, select);
    if (item_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 1)
    new_item = new (thd->mem_root) Item_func_atan(POS(), item_args[0]);
  else if (arg_count == 2)
    new_item =
        new (thd->mem_root) Item_func_atan(POS(), item_args[0], item_args[1]);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_log) {
  Item *item_args[2];
  assert(arg_count < 3);
  for (uint i = 0; i < arg_count; i++) {
    item_args[i] = args[i]->pq_clone(thd, select);
    if (item_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 1)
    new_item = new (thd->mem_root) Item_func_log(POS(), item_args[0]);
  else if (arg_count == 2)
    new_item =
        new (thd->mem_root) Item_func_log(POS(), item_args[0], item_args[1]);
}
PQ_CLONE_RETURN

/* Item_dec_func end*/

COPY_FUNC_ITEM(Item_func_longfromgeohash, POS(), ARG0)

PQ_CLONE_DEF(Item_func_rand) {
  new_item = new (thd->mem_root) Item_func_rand(POS());
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_latfromgeohash, POS(), ARG0)
/* Item_real_func end */

/* Item_str_func start */
PQ_CLONE_DEF(Item_func_aes_decrypt) {
  assert(arg_count < 4);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_aes_decrypt(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_aes_decrypt(POS(), new_args[0], new_args[1], new_args[2]);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_aes_encrypt) {
  assert(arg_count < 4);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_aes_encrypt(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_aes_encrypt(POS(), new_args[0], new_args[1], new_args[2]);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_char) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_char(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_charset, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_collation, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_compress, POS(), ARG0)

PQ_CLONE_DEF(Item_func_concat) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_concat(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_concat, Item_func) {
  if (orig_item != nullptr) {
    m_orafun = orig_item->m_orafun;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_func_concat_ws) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root) Item_func_concat_ws(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_conv, POS(), ARG0, ARG1, ARG2)
COPY_FUNC_ITEM(Item_func_conv_charset, POS(), ARG0, m_cast_cs)

PQ_CLONE_DEF(Item_func_date_format) {
  PQ_CLONE_ARGS
  new_item = new (thd->mem_root) Item_func_date_format(
      POS(), item_list[0], item_list[1], this->is_time_format);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_date_format, Item_str_func) {
  if (orig_item != nullptr) {
    value.copy(orig_item->value);
    fixed_length = orig_item->fixed_length;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_func_elt) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_elt(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_export_set) {
  PQ_CLONE_ARGS

  if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_export_set(POS(), item_list[0], item_list[1], item_list[2]);
  } else if (arg_count == 4) {
    new_item = new (thd->mem_root) Item_func_export_set(
        POS(), item_list[0], item_list[1], item_list[2], item_list[3]);
  } else if (arg_count == 5) {
    new_item = new (thd->mem_root)
        Item_func_export_set(POS(), item_list[0], item_list[1], item_list[2],
                             item_list[3], item_list[4]);
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_from_base64, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_inet_ntoa, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_insert, POS(), ARG0, ARG1, ARG2, ARG3)

PQ_CLONE_DEF(Item_func_json_quote) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_json_quote(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_left, POS(), ARG0, ARG1)

COPY_FUNC_ITEM(Item_func_lpad, POS(), ARG0, ARG1, ARG2)

PQ_COPY_FROM_DEF(Item_func_lpad, Item_str_func) {
  if (orig_item != nullptr) lpad_str.copy(orig_item->lpad_str);
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_func_oracle_lpad) {
  assert(arg_count < 4);
  Item *new_args[3] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }
  if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_oracle_lpad(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_oracle_lpad(POS(), new_args[0], new_args[1], new_args[2]);
  }
  COPY_SELF_ATTR(new_item);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_oracle_rpad) {
  assert(arg_count < 4);
  Item *new_args[3] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }
  if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_oracle_rpad(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_oracle_rpad(POS(), new_args[0], new_args[1], new_args[2]);
  }
  COPY_SELF_ATTR(new_item);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_make_set) {
  // MERGE_PS8404_CHECK_TAG: @tianjun, confirm commentd codes useless?
  // Item *arg_a = item->pq_clone(thd, select);
  // if (arg_a == nullptr) return nullptr;

  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_func_make_set(POS(),  // arg_a,
                                                    &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_table_rows) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_table_rows(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_avg_row_length) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root)
      Item_func_internal_avg_row_length(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_data_length) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_data_length(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_max_data_length) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root)
      Item_func_internal_max_data_length(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_index_length) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_index_length(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_data_free) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_data_free(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_auto_increment) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root)
      Item_func_internal_auto_increment(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_update_time) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_update_time(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_check_time) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_check_time(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_checksum) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item =
      new (thd->mem_root) Item_func_internal_checksum(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_internal_get_comment_or_error) {
  PQ_CLONE_ARGS
  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;
  new_item = new (thd->mem_root)
      Item_func_internal_get_comment_or_error(POS(), &pt_item_list);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_monthname, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_pfs_format_bytes, POS(), ARG0)

PQ_COPY_FROM_DEF(Item_func_pfs_format_bytes, Item_str_func) {
  if (orig_item != nullptr) {
    m_value = orig_item->m_value;
    memcpy(orig_item->m_value_buffer, m_value_buffer, 20);
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_pfs_format_pico_time, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_quote, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_repeat, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_replace, POS(), ARG0, ARG1, ARG2)
COPY_FUNC_ITEM(Item_func_reverse, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_random_bytes, POS(), ARG0)

PQ_CLONE_DEF(Item_func_right) {
  PQ_CLONE_ARGS
  new_item =
      new (thd->mem_root) Item_func_right(POS(), item_list[0], item_list[1]);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_rpad, POS(), ARG0, ARG1, ARG2)

PQ_COPY_FROM_DEF(Item_func_rpad, Item_str_func) {
  if (orig_item != nullptr) {
    rpad_str.copy(orig_item->rpad_str);
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_set_collation, POS(), ARG0, collation_string)

PQ_COPY_FROM_DEF(Item_func_set_collation, Item_str_func) {
  if (orig_item != nullptr && orig_item->args[1] != nullptr) {
    args[1] = orig_item->args[1]->pq_clone(thd, select);
    if (args[1] == nullptr) return true;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_soundex, ARG0)
COPY_FUNC_ITEM(Item_func_space, POS(), ARG0)

PQ_CLONE_DEF(Item_func_substr) {
  assert(arg_count < 4);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2) {
    new_item =
        new (thd->mem_root) Item_func_substr(POS(), new_args[0], new_args[1]);
  } else if (arg_count == 3) {
    new_item = new (thd->mem_root)
        Item_func_substr(POS(), new_args[0], new_args[1], new_args[2]);
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_substr_index, POS(), ARG0, ARG1, ARG2)
COPY_FUNC_ITEM(Item_func_database, POS())
COPY_FUNC_ITEM(Item_func_user, POS())

PQ_CLONE_DEF(Item_func_trim) {
  PQ_CLONE_ARGS

  if (arg_count > 1)
    new_item = new (thd->mem_root)
        Item_func_trim(POS(), item_list[0], item_list[1], m_trim_mode);
  else
    new_item =
        new (thd->mem_root) Item_func_trim(POS(), item_list[0], m_trim_mode);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_ltrim, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_rtrim, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_uncompress, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_unhex, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_uuid, POS())

PQ_CLONE_DEF(Item_func_uuid_to_bin) {
  assert(arg_count < 3);
  Item *new_args[4] = {nullptr};

  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 1) {
    new_item = new (thd->mem_root) Item_func_uuid_to_bin(POS(), new_args[0]);
  } else if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_uuid_to_bin(POS(), new_args[0], new_args[1]);
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_weight_string, POS(), ARG0, result_length,
               num_codepoints, flags, as_binary)

PQ_CLONE_DEF(Item_func_bin_to_uuid) {
  assert(arg_count < 3);
  Item *new_args[4] = {nullptr};

  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 1) {
    new_item = new (thd->mem_root) Item_func_bin_to_uuid(POS(), new_args[0]);
  } else if (arg_count == 2) {
    new_item = new (thd->mem_root)
        Item_func_bin_to_uuid(POS(), new_args[0], new_args[1]);
  }
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_format) {
  assert(arg_count < 4);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2)
    new_item =
        new (thd->mem_root) Item_func_format(POS(), new_args[0], new_args[1]);
  else if (arg_count == 3)
    new_item = new (thd->mem_root)
        Item_func_format(POS(), new_args[0], new_args[1], new_args[2]);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_get_format) {
  assert(arg_count == 1);
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;

  new_item = new (thd->mem_root) Item_func_get_format(POS(), type, arg);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_hex, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_inet6_aton, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_inet6_ntoa, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_md5, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_sha, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_sha2, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_to_base64, POS(), ARG0)

PQ_COPY_FROM_DEF(Item_str_conv, Item_str_func) {
  if (orig_item != nullptr) {
    multiply = orig_item->multiply;
    converter = orig_item->converter;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_func_upper, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_lower, POS(), ARG0)

PQ_COPY_FROM_DEF(Item_temporal_hybrid_func, Item_str_func) {
  if (orig_item != nullptr) {
    sql_mode = orig_item->sql_mode;
    ascii_buf.copy(orig_item->ascii_buf);
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_date_add_interval) {
  Item *arg_a = args[0]->pq_clone(thd, select);
  Item *arg_b = args[1]->pq_clone(thd, select);
  if (arg_a == nullptr || arg_b == nullptr) return nullptr;
  new_item = new (thd->mem_root)
      Item_date_add_interval(arg_a, arg_b, m_interval_type, m_subtract);
  if (new_item) {
    new_item->set_data_type(data_type());
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_add_time, POS(), ARG0, ARG1, m_datetime,
               m_sign == -1 ? true : false)

COPY_FUNC_ITEM(Item_func_str_to_date, POS(), ARG0, ARG1)

COPY_FUNC_ITEM(Item_func_add_months, POS(), ARG0, ARG1)

PQ_COPY_FROM_DEF(Item_func_str_to_date, Item_temporal_hybrid_func) {
  if (orig_item != nullptr) {
    cached_timestamp_type = orig_item->cached_timestamp_type;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_typecast_char, thd, ARG0, m_cast_length, m_cast_cs)

/* Item_to_clob start */
PQ_CLONE_DEF(Item_to_clob) {
  PQ_CLONE_ARGS

  PT_item_list pt_item_list{POS()};
  pt_item_list.value = item_list;

  new_item = new (thd->mem_root) Item_to_clob(thd, POS(), &pt_item_list);
}
PQ_CLONE_RETURN
/* Item_to_clob end */

PQ_CLONE_DEF(Item_date_literal) {
  MYSQL_TIME ltime;
  cached_time.get_time(&ltime);
  new_item = new (thd->mem_root) Item_date_literal(&ltime);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_to_char) {
  assert(arg_count < 4);
  Item *new_args[4] = {nullptr};
  for (uint i = 0; i < arg_count; i++) {
    new_args[i] = args[i]->pq_clone(thd, select);
    if (new_args[i] == nullptr) return nullptr;
  }

  if (arg_count == 2)
    new_item =
        new (thd->mem_root) Item_func_to_char(POS(), new_args[0], new_args[1]);
  else if (arg_count == 3)
    new_item = new (thd->mem_root)
        Item_func_to_char(POS(), new_args[0], new_args[1], new_args[2]);
}
PQ_CLONE_RETURN

/* Item_str_func end */

COPY_FUNC_ITEM(Item_func_curdate_utc, POS())
COPY_FUNC_ITEM(Item_func_curdate_local, POS())
COPY_FUNC_ITEM(Item_func_from_days, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_makedate, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_typecast_date, POS(), ARG0)

PQ_COPY_FROM_DEF(Item_typecast_date, Item_date_func) {
  if (orig_item != nullptr) {
    m_explicit_cast = orig_item->m_explicit_cast;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_datetime_literal) {
  MYSQL_TIME *ltime = new (thd->mem_root) MYSQL_TIME();
  if (ltime != nullptr) {
    this->get_date(ltime, 0);
    new_item = new (thd->mem_root) Item_datetime_literal(
        ltime, this->cached_time.decimals(), thd->variables.time_zone);
  }
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_convert_tz, POS(), ARG0, ARG1, ARG2)
COPY_FUNC_ITEM(Item_func_from_unixtime, POS(), ARG0)

PQ_CLONE_DEF(Item_func_sysdate_local) {
  new_item = new (thd->mem_root) Item_func_sysdate_local(decimals);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_typecast_datetime) {
  if (origin_item) {
    return origin_item->pq_clone(thd, select);
  }
  Item *arg_item = args[0]->pq_clone(thd, select);
  if (!arg_item) return nullptr;

  new_item = new (thd->mem_root) Item_typecast_datetime(POS(), arg_item);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_typecast_datetime, Item_datetime_func) {
  if (orig_item != nullptr) {
    detect_precision_from_arg = orig_item->detect_precision_from_arg;
    decimals = orig_item->decimals;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_func_at_time_zone) {
  Item *arg_item = args[0]->pq_clone(thd, select);
  if (arg_item == nullptr) return nullptr;

  new_item = new (thd->mem_root) Item_func_at_time_zone(
      POS(), arg_item, m_specifier_string, m_is_interval);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_curtime_local) {
  PQ_CLONE_ARGS
  new_item = new (thd->mem_root) Item_func_curtime_local(POS(), this->decimals);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_curtime_utc) {
  new_item = new (thd->mem_root) Item_func_curtime_utc(POS(), decimals);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_maketime, POS(), ARG0, ARG1, ARG2)
COPY_FUNC_ITEM(Item_func_sec_to_time, POS(), ARG0)
COPY_FUNC_ITEM(Item_func_timediff, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_typecast_time, POS(), ARG0)

PQ_COPY_FROM_DEF(Item_typecast_time, Item_time_func) {
  if (orig_item != nullptr) {
    detect_precision_from_arg = orig_item->detect_precision_from_arg;
    decimals = orig_item->decimals;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_time_literal) {
  MYSQL_TIME *ltime = new (thd->mem_root) MYSQL_TIME();
  if (ltime == nullptr) return nullptr;
  cached_time.get_time(ltime);
  new_item = new (thd->mem_root) Item_time_literal(ltime, pq_dec_arg);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_typecast_decimal) {
  Item *item_arg = args[0]->pq_clone(thd, select);
  if (item_arg == nullptr) return nullptr;

  new_item = new (thd->mem_root)
      Item_typecast_decimal(POS(), item_arg, pq_precision, decimals);
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_typecast_real, ARG0)

PQ_CLONE_DEF(Item_func_get_system_var) {
  new_item =
      new (thd->mem_root) Item_func_get_system_var(var_tracker, var_scope);
  new_item->item_name = item_name;
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_get_system_var, Item_var_func) {
  if (orig_item != nullptr) {
    cached_llval = orig_item->cached_llval;
    cached_dval = orig_item->cached_dval;
    cached_strval.copy(orig_item->cached_strval);
    cached_null_value = orig_item->cached_null_value;
    used_query_id = orig_item->used_query_id;
    cache_present = orig_item->cache_present;
  }
}
PQ_COPY_FROM_RETURN
/* Item_func end */

PQ_COPY_FROM_DEF(Item_sum, Item_result_field) {
  if (orig_item != nullptr) {
    force_copy_fields = orig_item->force_copy_fields;
    with_distinct = orig_item->with_distinct;
    max_aggr_level = orig_item->max_aggr_level;
    max_sum_func_level = orig_item->max_sum_func_level;
    allow_group_via_temp_table = orig_item->allow_group_via_temp_table;
    save_deny_window_func = orig_item->save_deny_window_func;
    used_tables_cache = orig_item->used_tables_cache;
    forced_const = orig_item->forced_const;
  }
  resolve_type(thd);
  Item_sum *is = dynamic_cast<Item_sum *>(item);
  aggr_query_block = is->aggr_query_block;
  base_query_block = is->base_query_block;
  used_tables_cache = is->used_tables_cache;
  force_copy_fields = is->has_force_copy_fields();
  with_distinct = is->has_with_distinct();
  max_aggr_level = is->max_aggr_level;
  max_sum_func_level = is->max_sum_func_level;
  allow_group_via_temp_table = is->allow_group_via_temp_table;
  forced_const = is->is_forced_const();
  null_value = item->null_value;
  set_nullable(item->is_nullable());
}
PQ_COPY_FROM_RETURN

PQ_COPY_FROM_DEF(Item_sum_bit, Item_sum) {
  if (orig_item != nullptr) {
    reset_bits = orig_item->reset_bits;
    bits = orig_item->bits;
    hybrid_type = orig_item->hybrid_type;
    m_count = orig_item->m_count;
    m_frame_null_count = orig_item->m_frame_null_count;
  }
  m_digit_cnt = nullptr;
  m_digit_cnt_card = 0;
  if (orig_item != nullptr) {
    m_is_xor = orig_item->m_is_xor;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_sum_and) {
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;

  new_item = new (thd->mem_root) Item_sum_and(POS(), arg, nullptr);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_sum_or) {
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;

  new_item = new (thd->mem_root) Item_sum_or(POS(), arg, nullptr);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_sum_xor) {
  Item *arg = args[0]->pq_clone(thd, select);
  if (nullptr == arg) return nullptr;

  new_item = new (thd->mem_root) Item_sum_xor(POS(), arg, nullptr);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_sum_hybrid, Item_sum) {
  if (orig_item == nullptr) {
    return true;
  }

  hybrid_type = orig_item->hybrid_type;
  m_has_values = orig_item->m_has_values;
  m_nulls_first = orig_item->m_nulls_first;
  m_optimize = orig_item->m_optimize;
  m_want_first = orig_item->m_want_first;
  m_cnt = orig_item->m_cnt;
  m_saved_last_value_at = orig_item->m_saved_last_value_at;
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_sum_max, ARG0)

COPY_FUNC_ITEM(Item_sum_min, ARG0)

PQ_COPY_FROM_DEF(Item_sum_num, Item_sum) {
  DBUG_EXECUTE_IF("simulate_item_rebuild_attr_copy_error", return true;);
  if (orig_item != nullptr) {
    is_evaluated = orig_item->is_evaluated;
  }
}
PQ_COPY_FROM_RETURN

COPY_FUNC_ITEM(Item_sum_count, POS(), ARG0, nullptr)

Item *PTI_count_sym::pq_clone(THD *thd, Query_block *select) {
  CHECK_TYPE(PTI_count_sym)
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;
  Item_sum_count *new_count =
      new (thd->mem_root) Item_sum_count(POS(), arg, nullptr);
  if (new_count == nullptr || new_count->pq_copy_from(thd, select, this))
    return nullptr;
  return new_count;
}

COPY_FUNC_ITEM(Item_sum_sum, POS(), ARG0, has_with_distinct(), nullptr)

PQ_CLONE_DEF(Item_sum_avg) {
  assert(arg_count == 1);
  Item *arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;
  new_item = new (thd->mem_root)
      Item_sum_avg(POS(), arg, has_with_distinct(), nullptr);
}
PQ_CLONE_RETURN

/* Item_result_field end */

/* Item_splocal start */
Item *Item_splocal::pq_clone(THD *thd, Query_block *select) {
  CHECK_TYPE(Item_splocal);
  Item *new_item = nullptr;

  Item *sp_this = this_item();
  if (sp_this->type() != Item::FIELD_ITEM) {
    return nullptr;
  }
  null_value = ((Item_field *)sp_this)->field->is_null();
  sp_this->null_value = null_value;

  if (null_value) {
    new_item = new (thd->mem_root) Item_null;
  } else {
    switch (type()) {
      case Item::INT_ITEM: {
        longlong val = val_int();
        new_item = new (thd->mem_root) Item_int(val);
        break;
      }
      case Item::DECIMAL_ITEM: {
        my_decimal dec_buf;
        my_decimal *decimal_value = val_decimal(&dec_buf);
        new_item = new (thd->mem_root) Item_decimal(decimal_value);
        break;
      }
      case Item::REAL_ITEM: {
        double val = val_real();
        new_item = new (thd->mem_root) Item_float(val, decimals);
        break;
      }
      case Item::STRING_ITEM: {
        String str_buf;
        const char *val = val_str(&str_buf)->ptr();
        size_t len = val_str(&str_buf)->length();
        new_item =
            new (thd->mem_root) Item_string(val, len, collation.collation);
        break;
      }
      default: {
        return nullptr;
      }
    }
  }
  if (!new_item || new_item->Item::pq_copy_from(thd, select, this)) {
    return nullptr;
  }
  return new_item;
}
/* Item_splocal end */

PQ_CLONE_DEF(Item_row) {
  assert(arg_count > 0);
  Item *arg_head = items[0]->pq_clone(thd, select);
  if (arg_head == nullptr) return nullptr;
  mem_root_deque<Item *> tail(thd->mem_root);
  for (uint i = 1; i < arg_count; i++) {
    Item *arg_tail = items[i]->pq_clone(thd, select);
    if (arg_tail == nullptr) return nullptr;
    tail.push_back(arg_tail);
  }
  new_item = new (thd->mem_root) Item_row(arg_head, tail);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_row, Item) {
  // generated a random item_name for item_row
  if (orig_item != nullptr) {
    if (orig_item->item_name.length() == 0) {
      assert(orig_item->item_name.ptr() == nullptr);
      uint32 addr_mid_8 = ((uint64)this >> 32) << 24;
      std::string std_addr = "ITEM_ROW" + std::to_string(addr_mid_8);
      item_name.copy(std_addr.c_str(), std_addr.length(), system_charset_info,
                     true);
    }
    used_tables_cache = orig_item->used_tables_cache;
    not_null_tables_cache = orig_item->not_null_tables_cache;
    with_null = orig_item->with_null;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_float) {
  new_item =
      new (thd->mem_root) Item_float(item_name, value, decimals, max_length);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_int) { new_item = new (thd->mem_root) Item_int(this); }
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_int, Item_num) {
  if (orig_item != nullptr) {
    value = orig_item->value;
  }
}
PQ_COPY_FROM_RETURN

PQ_CLONE_DEF(Item_uint) {
  new_item = new (thd->mem_root) Item_uint(item_name, value, max_length);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_decimal) {
  new_item = new (thd->mem_root)
      Item_decimal(item_name, &decimal_value, decimals, max_length);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_version) {
  new_item = new (thd->mem_root) Item_func_version(POS());
}
PQ_CLONE_RETURN

/* Item_func_lnnvl pq start */
PQ_CLONE_DEF(Item_func_lnnvl) {
  PQ_CLONE_ARGS
  new_item = new (thd->mem_root) Item_func_lnnvl(POS(), item_list[0]);
}
PQ_CLONE_RETURN
/* Item_func_lnnvl pq end */

PQ_CLONE_DEF(PTI_function_call_nonkeyword_now) {
  new_item =
      new (thd->mem_root) PTI_function_call_nonkeyword_now(POS(), decimals);
  new_item->is_current_timestmap = is_current_timestmap;
  new_item->is_dec_appoint = is_dec_appoint;
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(PTI_text_literal_text_string) {
  new_item =
      new (thd->mem_root) PTI_text_literal_text_string(POS(), is_7bit, literal);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(PTI_text_literal_nchar_string) {
  new_item = new (thd->mem_root)
      PTI_text_literal_nchar_string(POS(), is_7bit, literal);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(PTI_text_literal_underscore_charset) {
  new_item = new (thd->mem_root)
      PTI_text_literal_underscore_charset(POS(), is_7bit, cs, literal);
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(PTI_user_variable) {
  new_item = new (thd->mem_root)
      PTI_user_variable(POS(), {const_cast<char *>(name.m_str), name.m_length});
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_connection_id) {
  new_item = new (thd->mem_root) Item_func_connection_id(POS());
}
PQ_CLONE_RETURN

PQ_CLONE_DEF(Item_func_trig_cond) {
  Item *arg = nullptr;
  if (arg_count > 0) arg = args[0]->pq_clone(thd, select);
  if (arg == nullptr) return nullptr;
  new_item = new (thd->mem_root) Item_func_trig_cond(
      arg, trig_var, thd->lex->unit->first_query_block()->join, m_idx,
      trig_type);
}
PQ_CLONE_RETURN

PQ_COPY_FROM_DEF(Item_func_connection_id, Item_int_func) {}
PQ_COPY_FROM_RETURN

Item *Item_func_unix_timestamp::pq_clone(THD *thd, Query_block *select) {
  Item *arg_item = nullptr;
  if (arg_count > 0) {
    arg_item = args[0]->pq_clone(thd, select);
    if (!arg_item) return nullptr;
  }

  Item_func_unix_timestamp *new_item = nullptr;
  if (arg_count) {
    new_item = new (thd->mem_root) Item_func_unix_timestamp(POS(), arg_item);
  } else {
    new_item = new (thd->mem_root) Item_func_unix_timestamp(POS());
  }

  if (!new_item || new_item->pq_copy_from(thd, select, this)) return nullptr;

  return new_item;
}

Item *Item_func_current_user::pq_clone(THD *thd, Query_block *select) {
  Item_func_current_user *new_item =
      new (thd->mem_root) Item_func_current_user(POS());
  if (!new_item || new_item->pq_copy_from(thd, select, this)) return nullptr;

  new_item->context = &select->context;
  return new_item;
}

Item *Item_rollup_group_item::pq_clone(
    THD *thd, Query_block *select MY_ATTRIBUTE((unused))) {
  Item *arg_item = args[0]->pq_clone(thd, select);
  if (!arg_item) return nullptr;

  Item_rollup_group_item *new_item =
      new (thd->mem_root) Item_rollup_group_item(min_rollup_level(), arg_item);
  if (!new_item || new_item->pq_copy_from(thd, select, this)) return nullptr;

  return new_item;
}

Item *Item_rollup_sum_switcher::pq_clone(
    THD *thd, Query_block *select MY_ATTRIBUTE((unused))) {
  List<Item> alternatives;
  uint num_level = get_num_levels();
  for (uint i = 0; i < num_level; ++i) {
    alternatives.push_back(args[i]);
  }
  Item_rollup_sum_switcher *new_item =
      new (thd->mem_root) Item_rollup_sum_switcher(&alternatives);
  if (!new_item || new_item->pq_copy_from(thd, select, this)) return nullptr;

  return new_item;
}

COPY_FUNC_ITEM(Item_func_benchmark, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_found_rows, POS())

COPY_FUNC_ITEM(Item_func_to_date, POS(), ARG0, ARG1)
COPY_FUNC_ITEM(Item_func_last_day, POS(), ARG0)
PQ_CLONE_DEF(Item_func_now_local) {
  new_item = new (thd->mem_root) Item_func_now_local(decimals);
  new_item->is_current_timestmap = is_current_timestmap;
  new_item->is_dec_appoint = is_dec_appoint;
}
PQ_CLONE_RETURN

COPY_FUNC_ITEM(Item_func_rawtohex, POS(), ARG0)

COPY_FUNC_ITEM(Item_func_translate, POS(), ARG0, ARG1, ARG2);

Item *Item_func_group_concat::pq_clone(THD *thd, Query_block *select) {
  if (m_is_pivot_ref || m_window != nullptr || has_wf()) return nullptr;
  auto *new_item = new (thd->mem_root) Item_func_group_concat(thd, this);
  if (!new_item || new_item->pq_copy_from(thd, select, this)) return nullptr;
  return new_item;
}

PQ_COPY_FROM_DEF(Item_func_group_concat, Item_sum) {
  auto *gcf = down_cast<Item_func_group_concat *>(item);
  for (uint i = 0; i < m_order_arg_count; i++) {
    uint j = 0;
    for (j = 0; j < arg_count; j++) {
      if (*gcf->get_order_array()[i].item == gcf->args[j]) {
        order_array[i].item = args + j;
        break;
      }
    }
    if (j == arg_count) {
      // assert(0);
      return true;
    }
  }
}
PQ_COPY_FROM_RETURN

PT_order_list *PT_order_list::pq_clone(THD *thd, Query_block *select) {
  auto *ret = new (current_thd->mem_root) PT_order_list(POS());
  if (!ret) {
    return nullptr;
  }
#ifndef NDEBUG
  ret->set_contextualized(true);
#endif  // NDEBUG
  for (ORDER *ori = value.first; ori != nullptr; ori = ori->next) {
    ORDER *order = new (thd->mem_root) ORDER();
    if (!order) return nullptr;
    order->next = nullptr;
    order->item_initial = (*ori->item)->pq_clone(thd, select);
    if (!order->item_initial) return nullptr;
    order->direction = ori->direction;
    order->in_field_list = ori->in_field_list;
    order->used_alias = ori->used_alias;
    order->field_in_tmp_table = nullptr;
    order->buff = nullptr;
    order->used = 0;
    order->depend_map = 0;
    order->is_explicit = ori->is_explicit;

    if (!order) return nullptr;

    ret->value.link_in_list(order, &order->next);
  }
  return ret;
}

Item *Item_lead_lag::pq_clone(THD *thd, Query_block *select) {
  if (has_aggregation()) return nullptr;
  m_origin_args0 = args[0]->pq_clone(thd, select);
  if (!m_origin_args0) return nullptr;
  if (arg_count == 3) {
    m_origin_args2 = args[2]->pq_clone(thd, select);
    if (!m_origin_args2) return nullptr;
  }
  return this;
}

Window *Window::pq_clone(THD *thd, Query_block *select) {
  if (m_inherit_from) {
    return nullptr;
  }

  auto *ret = new (current_thd->mem_root)
      PT_window(POS(), m_frame, m_needs_frame_buffering, m_needs_peerset,
                m_needs_last_peer_in_frame, m_needs_partition_cardinality,
                m_row_optimizable, m_range_optimizable, m_static_aggregates,
                m_opt_first_row, m_opt_last_row, m_last, keep_dir);
  if (ret) {
#ifndef NDEBUG
    ret->set_contextualized(true);
#endif  // NDEBUG
    ret->set_name(m_name);
    ret->set_def_pos(m_def_pos);
    if (m_partition_by) {
      auto *ol1 = m_partition_by->pq_clone(thd, select);
      if (!ol1) return nullptr;
      ret->set_partition_list(ol1);
    }
    if (m_order_by) {
      auto *ol2 = m_order_by->pq_clone(thd, select);
      if (!ol2) return nullptr;
      ret->set_order_list(ol2);
    }

    for (auto *elem : m_partition_items) {
      auto *real_item = elem->get_item()->pq_clone(thd, select);
      if (!real_item) {
        return nullptr;
      }
      auto *ci = new_Cached_item(current_thd, real_item);
      if (!ci) {
        return nullptr;
      }
      ret->get_partition_items()->push_back(ci);
    }
  }
  return ret;
}

#endif
