//===-- CSKYSubtarget.h - Define Subtarget for the CSKY----------*- C++ -*-===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file declares the CSKY specific subclass of TargetSubtargetInfo.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIB_TARGET_CSKY_CSKYSUBTARGET_H
#define LLVM_LIB_TARGET_CSKY_CSKYSUBTARGET_H

#include "CSKYFrameLowering.h"
#include "CSKYISelLowering.h"
#include "CSKYInstrInfo.h"
#include "CSKYRegisterInfo.h"
#include "llvm/CodeGen/SelectionDAGTargetInfo.h"
#include "llvm/CodeGen/TargetSubtargetInfo.h"
#include "llvm/Target/TargetMachine.h"

#define GET_SUBTARGETINFO_HEADER
#include "CSKYGenSubtargetInfo.inc"

namespace llvm {
class StringRef;

class CSKYSubtarget : public CSKYGenSubtargetInfo {
  virtual void anchor();

  CSKYFrameLowering FrameLowering;
  CSKYInstrInfo InstrInfo;
  CSKYRegisterInfo RegInfo;
  CSKYTargetLowering TLInfo;
  SelectionDAGTargetInfo TSInfo;

  bool UseHardFloat;
  bool UseHardFloatABI;
  bool HasFPUv2SingleFloat;
  bool HasFPUv2DoubleFloat;
  bool HasFPUv3SingleFloat;
  bool HasFPUv3DoubleFloat;

  bool HasBTST16;
  bool HasJAVA;
  bool HasExtendLrw;
  bool HasDoloop;
  bool HasHighRegisters;

  bool HasE1;
  bool HasE2;
  bool Has2E3;
  bool HasMP;
  bool Has3E3r1;
  bool Has3r1E3r2;
  bool Has3r2E3r3;
  bool Has3E7;
  bool HasMP1E2;
  bool Has7E10;
  bool Has10E60;

public:
  CSKYSubtarget(const Triple &TT, StringRef CPU, StringRef TuneCPU,
                StringRef FS, const TargetMachine &TM);

  const CSKYFrameLowering *getFrameLowering() const override {
    return &FrameLowering;
  }
  const CSKYInstrInfo *getInstrInfo() const override { return &InstrInfo; }
  const CSKYRegisterInfo *getRegisterInfo() const override { return &RegInfo; }
  const CSKYTargetLowering *getTargetLowering() const override {
    return &TLInfo;
  }
  const SelectionDAGTargetInfo *getSelectionDAGInfo() const override {
    return &TSInfo;
  }

  /// Initializes using the passed in CPU and feature strings so that we can
  /// use initializer lists for subtarget initialization.
  CSKYSubtarget &initializeSubtargetDependencies(const Triple &TT,
                                                 StringRef CPU,
                                                 StringRef TuneCPU,
                                                 StringRef FS);

  // Generated by inc file
  void ParseSubtargetFeatures(StringRef CPU, StringRef TuneCPU, StringRef FS);

  bool useHardFloatABI() const;
  bool useHardFloat() const { return UseHardFloat; }
  bool hasFPUv2SingleFloat() const { return HasFPUv2SingleFloat; }
  bool hasFPUv2DoubleFloat() const { return HasFPUv2DoubleFloat; }
  bool hasFPUv2() const { return HasFPUv2SingleFloat || HasFPUv2DoubleFloat; }
  bool hasFPUv3SingleFloat() const { return HasFPUv3SingleFloat; }
  bool hasFPUv3DoubleFloat() const { return HasFPUv3DoubleFloat; }
  bool hasFPUv3() const { return HasFPUv3SingleFloat || HasFPUv3DoubleFloat; }
  bool hasAnyFloatExt() const { return hasFPUv2() || hasFPUv3(); };

  bool hasBTST16() const { return HasBTST16; }
  bool hasJAVA() const { return HasJAVA; }
  bool hasExtendLrw() const { return HasExtendLrw; }
  bool hasDoloop() const { return HasDoloop; }
  bool hasHighRegisters() const { return HasHighRegisters; }

  bool hasE1() const { return HasE1; }
  bool hasE2() const { return HasE2; }
  bool has2E3() const { return Has2E3; }
  bool has3r1E3r2() const { return Has3r1E3r2; }
  bool has3r2E3r3() const { return Has3r2E3r3; }
  bool has3E3r1() const { return Has3E3r1; }
  bool has3E7() const { return Has3E7; }
  bool hasMP() const { return HasMP; }
  bool hasMP1E2() const { return HasMP1E2; }
  bool has7E10() const { return Has7E10; }
  bool has10E60() const { return Has10E60; }
};
} // namespace llvm

#endif // LLVM_LIB_TARGET_CSKY_CSKYSUBTARGET_H
