/*
 * $Header: /cvshome/repository/org/osgi/service/upnp/UPnPEventListener.java,v 1.12 2002/10/08 06:43:04 pkriens Exp $
 *
 * Copyright (c) The Open Services Gateway Initiative (2002).
 * All Rights Reserved.
 *
 * Implementation of certain elements of the Open Services Gateway Initiative
 * (OSGI) Specification may be subject to third party intellectual property
 * rights, including without limitation, patent rights (such a third party may
 * or may not be a member of OSGi). OSGi is not responsible and shall not be
 * held responsible in any manner for identifying or failing to identify any or
 * all such third party intellectual property rights.
 *
 * This document and the information contained herein are provided on an "AS
 * IS" basis and OSGI DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION HEREIN WILL
 * NOT INFRINGE ANY RIGHTS AND ANY IMPLIED WARRANTIES OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT WILL OSGI BE LIABLE FOR ANY
 * LOSS OF PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTIAL,
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH THIS
 * DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH LOSS OR DAMAGE.
 *
 * All Company, brand and product names may be trademarks that are the sole
 * property of their respective owners. All rights reserved.
 */

package org.osgi.service.upnp;

import java.util.Dictionary;

/**
 * UPnP Events are mapped and delivered to applications according to the
 * OSGi whiteboard model. An application that wishes to be notified
 * of events generated by a particular UPnP Device registers a
 * service extending this interface.
 * <p>
 * The notification call from the UPnP Service to any <tt>UPnPEventListener</tt>
 * object must be done asynchronous with respect to the originator (in a separate thread).
 * <p>
 * Upon registration of the UPnP Event Listener service with the Framework, the service is notified
 * for each variable which it listens for with an initial event
 * containing the current value of the variable.
 * Subsequent notifications only happen on changes of the value of the
 * variable.
 * <p>
 * A UPnP Event Listener service filter the events it receives. This event set
 * is limited using a standard framework filter expression which is specified
 * when the listener service is registered.
 * <p>
 * The filter is specified in a property named "upnp.filter" and has as a
 * value an object of type <tt>org.osgi.framework.Filter</tt>.
 * <p>
 * When the Filter is evaluated, the folowing keywords are recognized as
 * defined as literal constants in the <tt>UPnPDevice</tt> class.
 * <p>
 * The valid subset of properties for the registration of UPnP Event Listener
 * services are:
 * <ul>
 *   <li><tt>UPnPDevice.TYPE</tt> -- Which type of device to listen for events.</li>
 *   <li><tt>UPnPDevice.ID</tt> -- The ID of a specific device to listen for events.</li>
 *   <li><tt>UPnPService.TYPE</tt> -- The type of a specific service to listen for events.</li>
 *   <li><tt>UPnPService.ID</tt> -- The ID of a specific service to listen for events.</li>
 * </ul>
 **/
public interface UPnPEventListener {

    /**
     * Key for a service property having a value that is an object of type
     * <tt>org.osgi.framework.Filter</tt> and that is used to limit
     * received events.
     **/
    static final String UPNP_FILTER="upnp.filter";

    /**
     * Callback method that is invoked for received events.
     *
     * The events are collected in a <tt>Dictionary</tt> object. Each entry has a
     * <tt>String</tt> key representing the event name (= state
     * variable name) and the new value of the state variable.
     * The class of the value object must match the class specified by the
     * UPnP State Variable associated with the event. This method must be called asynchronously
     *
     * @param deviceId ID of the device sending the events
     * @param serviceId ID of the service sending the events
     * @param events <tt>Dictionary</tt> object containing the new values for the state variables that have changed.
     *
     *
     **/

    void notifyUPnPEvent (String deviceId, String serviceId, Dictionary events);

}
