/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*-
 * vi:set noexpandtab tabstop=8 shiftwidth=8:
 *
 * Copyright (C) 2012-2018 Richard Hughes <richard@hughsie.com>
 *
 * SPDX-License-Identifier: GPL-2.0+
 */

#pragma once

#include <glib-object.h>

G_BEGIN_DECLS

/**
 * GsPluginStatus:
 * @GS_PLUGIN_STATUS_UNKNOWN:		Unknown status
 * @GS_PLUGIN_STATUS_WAITING:		Waiting
 * @GS_PLUGIN_STATUS_FINISHED:		Finished
 * @GS_PLUGIN_STATUS_SETUP:		Setup in progress
 * @GS_PLUGIN_STATUS_DOWNLOADING:	Downloading in progress
 * @GS_PLUGIN_STATUS_QUERYING:		Querying in progress
 * @GS_PLUGIN_STATUS_INSTALLING:	Installing in progress
 * @GS_PLUGIN_STATUS_REMOVING:		Removing in progress
 *
 * The status of the plugin.
 **/
typedef enum {
	GS_PLUGIN_STATUS_UNKNOWN,
	GS_PLUGIN_STATUS_WAITING,
	GS_PLUGIN_STATUS_FINISHED,
	GS_PLUGIN_STATUS_SETUP,
	GS_PLUGIN_STATUS_DOWNLOADING,
	GS_PLUGIN_STATUS_QUERYING,
	GS_PLUGIN_STATUS_INSTALLING,
	GS_PLUGIN_STATUS_REMOVING,
	GS_PLUGIN_STATUS_LAST  /*< skip >*/
} GsPluginStatus;

/**
 * GsPluginFlags:
 * @GS_PLUGIN_FLAGS_NONE:		No flags set
 * @GS_PLUGIN_FLAGS_INTERACTIVE:	User initiated the job
 *
 * The flags for the plugin at this point in time.
 **/
typedef enum {
	GS_PLUGIN_FLAGS_NONE = 0,
	GS_PLUGIN_FLAGS_INTERACTIVE = 1 << 4,
} GsPluginFlags;

/**
 * GsPluginError:
 * @GS_PLUGIN_ERROR_FAILED:			Generic failure
 * @GS_PLUGIN_ERROR_NOT_SUPPORTED:		Action not supported
 * @GS_PLUGIN_ERROR_CANCELLED:			Action was cancelled
 * @GS_PLUGIN_ERROR_NO_NETWORK:			No network connection available
 * @GS_PLUGIN_ERROR_NO_SECURITY:		Security policy forbid action
 * @GS_PLUGIN_ERROR_NO_SPACE:			No disk space to allow action
 * @GS_PLUGIN_ERROR_AUTH_REQUIRED:		Authentication was required
 * @GS_PLUGIN_ERROR_AUTH_INVALID:		Provided authentication was invalid
 * @GS_PLUGIN_ERROR_PLUGIN_DEPSOLVE_FAILED:	The plugins installed are incompatible
 * @GS_PLUGIN_ERROR_DOWNLOAD_FAILED:		The download action failed
 * @GS_PLUGIN_ERROR_WRITE_FAILED:		The save-to-disk failed
 * @GS_PLUGIN_ERROR_INVALID_FORMAT:		The data format is invalid
 * @GS_PLUGIN_ERROR_DELETE_FAILED:		The delete action failed
 * @GS_PLUGIN_ERROR_RESTART_REQUIRED:		A restart is required
 * @GS_PLUGIN_ERROR_AC_POWER_REQUIRED:		AC power is required
 * @GS_PLUGIN_ERROR_TIMED_OUT:			The job timed out
 * @GS_PLUGIN_ERROR_BATTERY_LEVEL_TOO_LOW:	The system battery level is too low
 *
 * The failure error types.
 **/
typedef enum {
	GS_PLUGIN_ERROR_FAILED,
	GS_PLUGIN_ERROR_NOT_SUPPORTED,
	GS_PLUGIN_ERROR_CANCELLED,
	GS_PLUGIN_ERROR_NO_NETWORK,
	GS_PLUGIN_ERROR_NO_SECURITY,
	GS_PLUGIN_ERROR_NO_SPACE,
	GS_PLUGIN_ERROR_AUTH_REQUIRED,
	GS_PLUGIN_ERROR_AUTH_INVALID,
	GS_PLUGIN_ERROR_PLUGIN_DEPSOLVE_FAILED,
	GS_PLUGIN_ERROR_DOWNLOAD_FAILED,
	GS_PLUGIN_ERROR_WRITE_FAILED,
	GS_PLUGIN_ERROR_INVALID_FORMAT,
	GS_PLUGIN_ERROR_DELETE_FAILED,
	GS_PLUGIN_ERROR_RESTART_REQUIRED,
	GS_PLUGIN_ERROR_AC_POWER_REQUIRED,
	GS_PLUGIN_ERROR_TIMED_OUT,
	GS_PLUGIN_ERROR_BATTERY_LEVEL_TOO_LOW,
	GS_PLUGIN_ERROR_LAST  /*< skip >*/
} GsPluginError;

/**
 * GsPluginRefineFlags:
 * @GS_PLUGIN_REFINE_FLAGS_DEFAULT:			No explicit flags set
 * @GS_PLUGIN_REFINE_FLAGS_USE_HISTORY:			Get the historical view (unused)
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_LICENSE:		Require the license
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_URL:			Require the URL
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_DESCRIPTION:		Require the long description
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_SIZE:		Require the installed and download sizes
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_RATING:		Require the rating
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_VERSION:		Require the version
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_HISTORY:		Require the history
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_SETUP_ACTION:	Require enough to install or remove the package
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_UPDATE_DETAILS:	Require update details
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_ORIGIN:		Require the origin
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_RELATED:		Require related packages
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_ADDONS:		Require available addons
 * @GS_PLUGIN_REFINE_FLAGS_ALLOW_PACKAGES:		Allow packages to be returned
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_UPDATE_SEVERITY:	Require update severity
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_UPGRADE_REMOVED:	Require distro upgrades
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_PROVENANCE:		Require the provenance
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_REVIEWS:		Require user-reviews
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_REVIEW_RATINGS:	Require user-ratings
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_ICON:		Require the icon to be loaded
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_PERMISSIONS:		Require the needed permissions
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_ORIGIN_HOSTNAME:	Require the origin hostname
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_ORIGIN_UI:		Require the origin for UI
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_RUNTIME:		Require the runtime
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_SCREENSHOTS:		Require screenshot information
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_CATEGORIES:		Require categories
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_PROJECT_GROUP:	Require project group
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_DEVELOPER_NAME:	Require developer name
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_KUDOS:		Require kudos
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_CONTENT_RATING:	Require content rating
 * @GS_PLUGIN_REFINE_FLAGS_REQUIRE_SIZE_DATA:		Require user and cache data sizes (Since: 41)
 * @GS_PLUGIN_REFINE_FLAGS_MASK:			All flags (Since: 40)
 *
 * The refine flags.
 **/
typedef enum {
	GS_PLUGIN_REFINE_FLAGS_DEFAULT			= 0,
	GS_PLUGIN_REFINE_FLAGS_USE_HISTORY		= 1 << 0, /* unused, TODO: perhaps ->STATE */
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_LICENSE		= 1 << 1,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_URL		= 1 << 2,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_DESCRIPTION	= 1 << 3,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_SIZE		= 1 << 4,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_RATING		= 1 << 5,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_VERSION		= 1 << 6,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_HISTORY		= 1 << 7,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_SETUP_ACTION	= 1 << 8,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_UPDATE_DETAILS	= 1 << 9,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_ORIGIN		= 1 << 10,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_RELATED		= 1 << 11,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_SIZE_DATA	= 1 << 12,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_ADDONS		= 1 << 13,
	GS_PLUGIN_REFINE_FLAGS_ALLOW_PACKAGES		= 1 << 14, /* TODO: move to request */
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_UPDATE_SEVERITY	= 1 << 15,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_UPGRADE_REMOVED	= 1 << 16,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_PROVENANCE	= 1 << 17,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_REVIEWS		= 1 << 18,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_REVIEW_RATINGS	= 1 << 19,
	/* 1 << 20 is currently unused; was previously REQUIRE_KEY_COLORS */
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_ICON		= 1 << 21,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_PERMISSIONS	= 1 << 22,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_ORIGIN_HOSTNAME	= 1 << 23,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_ORIGIN_UI	= 1 << 24,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_RUNTIME		= 1 << 25,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_SCREENSHOTS	= 1 << 26,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_CATEGORIES	= 1 << 27,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_PROJECT_GROUP	= 1 << 28,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_DEVELOPER_NAME	= 1 << 29,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_KUDOS		= 1 << 30,
	GS_PLUGIN_REFINE_FLAGS_REQUIRE_CONTENT_RATING	= 1 << 31,
	GS_PLUGIN_REFINE_FLAGS_MASK			= ~0,
} GsPluginRefineFlags;

/**
 * GsPluginRule:
 * @GS_PLUGIN_RULE_CONFLICTS:		The plugin conflicts with another
 * @GS_PLUGIN_RULE_RUN_AFTER:		Order the plugin after another
 * @GS_PLUGIN_RULE_RUN_BEFORE:		Order the plugin before another
 * @GS_PLUGIN_RULE_BETTER_THAN:		Results are better than another
 *
 * The rules used for ordering plugins.
 * Plugins are expected to add rules in gs_plugin_initialize().
 **/
typedef enum {
	GS_PLUGIN_RULE_CONFLICTS,
	GS_PLUGIN_RULE_RUN_AFTER,
	GS_PLUGIN_RULE_RUN_BEFORE,
	GS_PLUGIN_RULE_BETTER_THAN,
	GS_PLUGIN_RULE_LAST  /*< skip >*/
} GsPluginRule;

/**
 * GsPluginAction:
 * @GS_PLUGIN_ACTION_UNKNOWN:			Action is unknown
 * @GS_PLUGIN_ACTION_SETUP:			Plugin setup (internal)
 * @GS_PLUGIN_ACTION_INSTALL:			Install an application
 * @GS_PLUGIN_ACTION_REMOVE:			Remove an application
 * @GS_PLUGIN_ACTION_UPDATE:			Update an application
 * @GS_PLUGIN_ACTION_SET_RATING:		Set rating on an application
 * @GS_PLUGIN_ACTION_UPGRADE_DOWNLOAD:		Download a distro upgrade
 * @GS_PLUGIN_ACTION_UPGRADE_TRIGGER:		Trigger a distro upgrade
 * @GS_PLUGIN_ACTION_LAUNCH:			Launch an application
 * @GS_PLUGIN_ACTION_UPDATE_CANCEL:		Cancel the update
 * @GS_PLUGIN_ACTION_ADD_SHORTCUT:		Add a shortcut to an application
 * @GS_PLUGIN_ACTION_REMOVE_SHORTCUT:		Remove a shortcut to an application
 * @GS_PLUGIN_ACTION_GET_UPDATES:		Get the list of updates
 * @GS_PLUGIN_ACTION_GET_DISTRO_UPDATES:	Get the list of distro updates
 * @GS_PLUGIN_ACTION_GET_SOURCES:		Get the list of sources
 * @GS_PLUGIN_ACTION_GET_INSTALLED:		Get the list of installed applications
 * @GS_PLUGIN_ACTION_GET_POPULAR:		Get the list of popular applications
 * @GS_PLUGIN_ACTION_GET_FEATURED:		Get the list of featured applications
 * @GS_PLUGIN_ACTION_SEARCH:			Get the search results for a query
 * @GS_PLUGIN_ACTION_SEARCH_FILES:		Get the search results for a file query
 * @GS_PLUGIN_ACTION_SEARCH_PROVIDES:		Get the search results for a provide query
 * @GS_PLUGIN_ACTION_GET_CATEGORIES:		Get the list of categories
 * @GS_PLUGIN_ACTION_GET_CATEGORY_APPS:		Get the apps for a specific category
 * @GS_PLUGIN_ACTION_REFINE:			Refine the application
 * @GS_PLUGIN_ACTION_REFRESH:			Refresh all the sources
 * @GS_PLUGIN_ACTION_FILE_TO_APP:		Convert the file to an application
 * @GS_PLUGIN_ACTION_URL_TO_APP:		Convert the URI to an application
 * @GS_PLUGIN_ACTION_GET_RECENT:		Get the apps recently released
 * @GS_PLUGIN_ACTION_GET_UPDATES_HISTORICAL:    Get the list of historical updates
 * @GS_PLUGIN_ACTION_INITIALIZE:		Initialize the plugin
 * @GS_PLUGIN_ACTION_DESTROY:			Destroy the plugin
 * @GS_PLUGIN_ACTION_DOWNLOAD:			Download an application
 * @GS_PLUGIN_ACTION_GET_ALTERNATES:		Get the alternates for a specific application
 * @GS_PLUGIN_ACTION_GET_LANGPACKS:		Get appropriate language pack
 * @GS_PLUGIN_ACTION_INSTALL_REPO:		Install a repository (Since: 41)
 * @GS_PLUGIN_ACTION_REMOVE_REPO:		Remove a repository (Since: 41)
 * @GS_PLUGIN_ACTION_ENABLE_REPO:		Enable a repository (Since: 41)
 * @GS_PLUGIN_ACTION_DISABLE_REPO:		Disable a repository (Since: 41)
 *
 * The plugin action.
 **/
typedef enum {
	GS_PLUGIN_ACTION_UNKNOWN,
	GS_PLUGIN_ACTION_SETUP,
	GS_PLUGIN_ACTION_INSTALL,
	GS_PLUGIN_ACTION_REMOVE,
	GS_PLUGIN_ACTION_UPDATE,
	GS_PLUGIN_ACTION_SET_RATING,
	GS_PLUGIN_ACTION_UPGRADE_DOWNLOAD,
	GS_PLUGIN_ACTION_UPGRADE_TRIGGER,
	GS_PLUGIN_ACTION_LAUNCH,
	GS_PLUGIN_ACTION_UPDATE_CANCEL,
	GS_PLUGIN_ACTION_ADD_SHORTCUT,
	GS_PLUGIN_ACTION_REMOVE_SHORTCUT,
	GS_PLUGIN_ACTION_GET_UPDATES,
	GS_PLUGIN_ACTION_GET_DISTRO_UPDATES,
	GS_PLUGIN_ACTION_GET_SOURCES,
	GS_PLUGIN_ACTION_GET_INSTALLED,
	GS_PLUGIN_ACTION_GET_POPULAR,
	GS_PLUGIN_ACTION_GET_FEATURED,
	GS_PLUGIN_ACTION_SEARCH,
	GS_PLUGIN_ACTION_SEARCH_FILES,
	GS_PLUGIN_ACTION_SEARCH_PROVIDES,
	GS_PLUGIN_ACTION_GET_CATEGORIES,
	GS_PLUGIN_ACTION_GET_CATEGORY_APPS,
	GS_PLUGIN_ACTION_REFINE,
	GS_PLUGIN_ACTION_REFRESH,
	GS_PLUGIN_ACTION_FILE_TO_APP,
	GS_PLUGIN_ACTION_URL_TO_APP,
	GS_PLUGIN_ACTION_GET_RECENT,
	GS_PLUGIN_ACTION_GET_UPDATES_HISTORICAL,
	GS_PLUGIN_ACTION_INITIALIZE,
	GS_PLUGIN_ACTION_DESTROY,
	GS_PLUGIN_ACTION_DOWNLOAD,
	GS_PLUGIN_ACTION_GET_ALTERNATES,
	GS_PLUGIN_ACTION_GET_LANGPACKS,
	GS_PLUGIN_ACTION_INSTALL_REPO,
	GS_PLUGIN_ACTION_REMOVE_REPO,
	GS_PLUGIN_ACTION_ENABLE_REPO,
	GS_PLUGIN_ACTION_DISABLE_REPO,
	GS_PLUGIN_ACTION_LAST  /*< skip >*/
} GsPluginAction;

G_END_DECLS
