// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "ycc_uio.h"
#include "utils.h"
#include "ske.h"

struct skcipher_test {
	const char *name;
	const char *key;
	const char *iv;
	const char *iv_out;
	const char *ptext;
	const char *ctext;
	unsigned int key_len; /* length of key */
	unsigned int len; /* length of ptext or ctext*/
};

static const struct skcipher_test ske = {
	.name = "ske-unit",
	.key = "\x06\xa9\x21\x40\x36\xb8\xa1\x5b"
	       "\x51\x2e\x03\xd5\x34\x12\x00\x06",
	.key_len = 16,
	.iv = "\x3d\xaf\xba\x42\x9d\x9e\xb4\x30"
	      "\xb4\x22\xda\x80\x2c\x9f\xac\x41",
	.ptext = "Single block msg",
	.ctext = "\xe3\x53\x77\x9c\x10\x79\xae\xb8"
		 "\x27\x08\x94\x2d\xbe\x77\x18\x1a",
	.len = 16,
};

static unsigned int done;
static void test_ske_complete(struct skcipher_req *req, int err)
{
	if (!err) {
		hex_dump("Dump async ske dst:", req->dst, req->cryptlen);
		printf("Sample code async ske unit passed\n");
	}

	skcipher_free_req(req);
	done = 1;
}

int main(int argc, char *argv[])
{
	struct skcipher_ctx *ctx;
	struct skcipher_req *req;
	unsigned char ptext[16];
	unsigned char ctext[16];
	unsigned char iv[16];
	unsigned char key[16];
	int ret;

	ret = ycc_drv_init(1);
	if (ret < 0)
		return ret;

	ret = -1;
	ctx = skcipher_alloc_ctx("cbc(aes)", CRYPTO_ASYNC);
	if (!ctx)
		goto out;

	req = skcipher_alloc_req(ctx);
	if (!req)
		goto free_ctx;

	memcpy(key, ske.key, ske.key_len);
	skcipher_setkey(ctx, key, ske.key_len);

	memcpy(ptext, ske.ptext, ske.len);
	memcpy(iv, ske.iv, sizeof(ske.iv) - 1);
	skcipher_req_set_callback(req, test_ske_complete, NULL);
	skcipher_set_req(req, ptext, ctext, ske.len, iv);

	ret = skcipher_encrypt(req);
	if (ret == -EINPROGRESS) {
		/* Wait for completion */
		while (!done);

		/* Test passed, reset ret to 0 */
		ret = 0;
	} else {
		skcipher_free_req(req);
	}

free_ctx:
	skcipher_free_ctx(ctx);
out:
	ycc_drv_exit();
	return ret;
}
