"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GitHubRelease = void 0;
const tslib_1 = require("tslib");
const fs = require("fs");
const path = require("path");
const getPluginId_1 = require("../../config/utils/getPluginId");
const pluginValidation_1 = require("../../config/utils/pluginValidation");
const env_1 = require("../../plugins/env");
const githubClient_1 = tslib_1.__importDefault(require("./githubClient"));
const resolveContentType = (extension) => {
    if (extension.startsWith('.')) {
        extension = extension.slice(1);
    }
    switch (extension) {
        case 'zip':
            return 'application/zip';
        case 'json':
            return 'application/json';
        case 'sha1':
            return 'text/plain';
        default:
            return 'application/octet-stream';
    }
};
class GitHubRelease {
    constructor(token, username, repository, releaseNotes, commitHash) {
        this.token = token;
        this.username = username;
        this.repository = repository;
        this.releaseNotes = releaseNotes;
        this.commitHash = commitHash;
        this.git = new githubClient_1.default({
            required: true,
            owner: username,
            repo: repository,
        });
    }
    publishAssets(srcLocation, destUrl) {
        // Add the assets. Loop through files in the ci/dist folder and upload each asset.
        const files = fs.readdirSync(srcLocation);
        return files.map((file) => tslib_1.__awaiter(this, void 0, void 0, function* () {
            const fileStat = fs.statSync(`${srcLocation}/${file}`);
            const fileData = fs.readFileSync(`${srcLocation}/${file}`);
            return this.git.client.post(`${destUrl}?name=${file}`, fileData, {
                headers: {
                    'Content-Type': resolveContentType(path.extname(file)),
                    'Content-Length': fileStat.size,
                },
                maxContentLength: fileStat.size * 2 * 1024 * 1024,
            });
        }));
    }
    release() {
        var _a, _b, _c, _d;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const ciDir = (0, env_1.getCiFolder)();
            const distDir = path.resolve(ciDir, 'dist');
            const distContentDir = path.resolve(distDir, (0, getPluginId_1.getPluginId)());
            const pluginJsonFile = path.resolve(distContentDir, 'plugin.json');
            const pluginInfo = (0, pluginValidation_1.getPluginJson)(pluginJsonFile).info;
            const PUBLISH_DIR = path.resolve((0, env_1.getCiFolder)(), 'packages');
            const commitHash = this.commitHash || ((_a = pluginInfo.build) === null || _a === void 0 ? void 0 : _a.hash);
            try {
                const latestRelease = yield this.git.client.get(`releases/tags/v${pluginInfo.version}`);
                // Re-release if the version is the same as an existing release
                if (latestRelease.data.tag_name === `v${pluginInfo.version}`) {
                    yield this.git.client.delete(`releases/${latestRelease.data.id}`);
                }
            }
            catch (reason) {
                if (reason.response.status !== 404) {
                    // 404 just means no release found. Not an error. Anything else though, re throw the error
                    throw reason;
                }
            }
            try {
                // Now make the release
                const newReleaseResponse = yield this.git.client.post('releases', {
                    tag_name: `v${pluginInfo.version}`,
                    target_commitish: commitHash,
                    name: `v${pluginInfo.version}`,
                    body: this.releaseNotes,
                    draft: false,
                    prerelease: false,
                });
                const publishPromises = this.publishAssets(PUBLISH_DIR, `https://uploads.github.com/repos/${this.username}/${this.repository}/releases/${newReleaseResponse.data.id}/assets`);
                yield Promise.all(publishPromises);
            }
            catch (reason) {
                console.error((_d = (_c = (_b = reason.data) === null || _b === void 0 ? void 0 : _b.message) !== null && _c !== void 0 ? _c : reason.response.data) !== null && _d !== void 0 ? _d : reason);
                // Rethrow the error so that we can trigger a non-zero exit code to circle-ci
                throw reason;
            }
        });
    }
}
exports.GitHubRelease = GitHubRelease;
//# sourceMappingURL=githubRelease.js.map