// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use super::CommonMetricData;

/// A string metric.
///
/// Record an Unicode string value with arbitrary content.
/// Strings are length-limited to `MAX_LENGTH_VALUE` bytes.
///
/// # Example
///
/// The following piece of code will be generated by `glean_parser`:
///
/// ```rust,ignore
/// use glean::metrics::{StringMetric, CommonMetricData, Lifetime};
/// use once_cell::sync::Lazy;
///
/// mod browser {
///     pub static search_engine: Lazy<StringMetric> = Lazy::new(|| StringMetric::new(CommonMetricData {
///         name: "search_engine".into(),
///         category: "browser".into(),
///         lifetime: Lifetime::Ping,
///         disabled: false,
///         dynamic_label: None
///     }));
/// }
/// ```
///
/// It can then be used with:
///
/// ```rust,ignore
/// browser::search_engine.set("websearch");
/// ```
#[derive(Clone, Debug)]
pub struct StringMetric(pub(crate) glean_core::metrics::StringMetric);

impl StringMetric {
    /// Create a new string metric.
    pub fn new(meta: CommonMetricData) -> Self {
        Self(glean_core::metrics::StringMetric::new(meta))
    }

    /// Set to the specified value.
    ///
    /// ## Arguments
    ///
    /// * `value` - The string to set the metric to.
    ///
    /// ## Notes
    ///
    /// Truncates the value if it is longer than `MAX_STRING_LENGTH` bytes and logs an error.
    /// See [String metric limits](https://mozilla.github.io/glean/book/user/metrics/string.html#limits).
    pub fn set<S: Into<String>>(&self, value: S) {
        crate::with_glean(move |glean| self.0.set(glean, value))
    }

    /// **Test-only API.**
    ///
    /// Get the currently stored value as a string.
    /// This doesn't clear the stored value.
    ///
    /// ## Arguments
    ///
    /// * `storage_name` - the storage name to look into.
    ///
    /// ## Return value
    ///
    /// Returns the stored value or `None` if nothing stored.
    pub fn test_get_value(&self, storage_name: &str) -> Option<String> {
        crate::with_glean(move |glean| self.0.test_get_value(glean, storage_name))
    }
}
