"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererGitΩ = void 0;
const tslib_1 = require("tslib");
const errors_1 = require("@betterer/errors");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const fs_1 = require("fs");
const path_1 = (0, tslib_1.__importDefault)(require("path"));
const simple_git_1 = (0, tslib_1.__importDefault)(require("simple-git"));
const hasher_1 = require("../hasher");
const utils_1 = require("../utils");
const file_cache_1 = require("./file-cache");
const reader_1 = require("./reader");
class BettererGitΩ {
    constructor() {
        this._cache = null;
        this._configPaths = [];
        this._fileMap = {};
        this._filePaths = [];
        this._git = null;
        this._gitDir = null;
        this._rootDir = null;
        this._syncing = null;
    }
    async add(resultsPath) {
        (0, assert_1.default)(this._git);
        await this._git.add(resultsPath);
    }
    filterCached(testName, filePaths) {
        (0, assert_1.default)(this._cache);
        return this._cache.filterCached(testName, filePaths);
    }
    filterIgnored(filePaths) {
        return filePaths.filter((absolutePath) => this._fileMap[absolutePath]);
    }
    clearCache(testName) {
        (0, assert_1.default)(this._cache);
        this._cache.clearCache(testName);
    }
    async enableCache(cachePath) {
        (0, assert_1.default)(this._cache);
        return await this._cache.enableCache(cachePath);
    }
    updateCache(testName, filePaths) {
        (0, assert_1.default)(this._cache);
        return this._cache.updateCache(testName, filePaths);
    }
    writeCache() {
        (0, assert_1.default)(this._cache);
        return this._cache.writeCache();
    }
    getFilePaths() {
        return this._filePaths;
    }
    async init(configPaths) {
        this._configPaths = configPaths;
        this._gitDir = await this._findGitRoot();
        this._rootDir = path_1.default.dirname(this._gitDir);
        this._git = (0, simple_git_1.default)(this._rootDir);
        this._cache = new file_cache_1.BettererFileCacheΩ(this._configPaths);
        await this._init(this._git);
        await this.sync();
        return this._rootDir;
    }
    async sync() {
        if (this._syncing) {
            return await this._syncing;
        }
        this._syncing = this._sync();
        await this._syncing;
        this._syncing = null;
    }
    async _findGitRoot() {
        let dir = process.cwd();
        while (dir !== path_1.default.parse(dir).root) {
            try {
                const gitPath = path_1.default.join(dir, '.git');
                await fs_1.promises.access(gitPath);
                return gitPath;
            }
            catch (err) {
                dir = path_1.default.join(dir, '..');
            }
        }
        throw new errors_1.BettererError('.git directory not found. Betterer must be used within a git repository.');
    }
    async _getFileHash(filePath) {
        const content = await (0, reader_1.read)(filePath);
        if (content == null) {
            return null;
        }
        return (0, hasher_1.createHash)(content);
    }
    async _init(git) {
        const retries = 3;
        for (let i = 0; i < retries; i++) {
            try {
                await git.init();
            }
            catch (error) {
                if (i >= retries) {
                    throw error;
                }
            }
        }
    }
    _toFilePaths(rootDir, lines) {
        return lines.map((line) => toAbsolutePath(rootDir, line));
    }
    _toLines(output) {
        if (output.length === 0) {
            return [];
        }
        return Array.from(new Set(output.trimEnd().split('\n')));
    }
    async _sync() {
        this._fileMap = {};
        this._filePaths = [];
        (0, assert_1.default)(this._cache);
        (0, assert_1.default)(this._git);
        (0, assert_1.default)(this._rootDir);
        const treeOutput = await this._git.raw(['ls-tree', '--full-tree', '-r', 'HEAD']);
        const fileInfo = this._toLines(treeOutput).map((info) => info.split(/\s/));
        const fileHashes = {};
        // Collect hashes from git:
        fileInfo.forEach((fileInfo) => {
            const [, , hash, relativePath] = fileInfo;
            (0, assert_1.default)(this._rootDir);
            const absolutePath = toAbsolutePath(this._rootDir, relativePath);
            fileHashes[absolutePath] = hash;
            return absolutePath;
        });
        // Collect hashes for modified files:
        const modifiedOutput = await this._git.raw(['ls-files', '--modified']);
        const modifiedFilePaths = this._toFilePaths(this._rootDir, this._toLines(modifiedOutput));
        await Promise.all(modifiedFilePaths.map(async (absolutePath) => {
            fileHashes[absolutePath] = await this._getFileHash(absolutePath);
        }));
        // Collect all tracked files, excluding files that have been deleted, *and* all untracked files:
        const allFilesOutput = await this._git.raw(['ls-files', '--cached', '--others', '--exclude-standard']);
        const allFilePaths = this._toFilePaths(this._rootDir, this._toLines(allFilesOutput));
        await Promise.all(allFilePaths.map(async (absolutePath) => {
            // If file is tracked:
            //    `fileHashes[absolutePath]` = the git hash.
            // If file was tracked and is now deleted:
            //    `fileHashes[absolutePath]` = null
            //    `this._getFileHash(absolutePath)` = null
            // If file is untracked and is new:
            //    `fileHashes[absolutePath]` = null
            //    `this._getFileHash(absolutePath) = basic hash
            const hash = fileHashes[absolutePath] || (await this._getFileHash(absolutePath));
            // If hash is null then the file was deleted so it shouldn't be included:
            if (hash == null) {
                return;
            }
            this._fileMap[absolutePath] = hash;
            this._filePaths.push(absolutePath);
        }));
        this._cache.setHashes(this._fileMap);
    }
}
exports.BettererGitΩ = BettererGitΩ;
function toAbsolutePath(rootDir, relativePath) {
    return (0, utils_1.normalisedPath)(path_1.default.join(rootDir, relativePath.trimStart()));
}
//# sourceMappingURL=git.js.map