import { BettererFileGlobs, BettererFilePatterns } from '../../fs';
import { BettererTestConstraint, BettererTestDeadline, BettererTestGoal } from '../types';
import { BettererFileTestBase, BettererFileTestConfig, BettererFileTestFunction, BettererFileTestResult } from './types';
/**
 * @public A very common usecase for **Betterer** is to track issues across all the files in a
 * project. `BettererFileTest` provides a wrapper around {@link @betterer/betterer#BettererTest | `BettererTest` }
 * that makes it easier to implement such a test.
 *
 * @remarks `BettererFileTest` provides a useful example for the more complex possibilities of the {@link @betterer/betterer#BettererTestOptions | `BettererTestOptions` }
 * interface.
 *
 * @example
 * ```typescript
 * const fileTest = new BettererFileTest(async (filePaths, fileTestResult) => {
 *    await Promise.all(
 *      filePaths.map(async (filePath) => {
 *        const fileContents = await fs.readFile(filePath, 'utf8');
 *        const file = fileTestResult.addFile(filePath, fileContents);
 *        file.addIssue(0, 1, `There's something wrong with this file!`);
 *      })
 *    );
 *  });
 * ```
 *
 * @param fileTest - The test function that will detect issues in specific files.
 */
export declare class BettererFileTest implements BettererFileTestBase {
    /**
     * The complete configuration for the test.
     */
    readonly config: BettererFileTestConfig;
    private _isOnly;
    private _isSkipped;
    private _resolver;
    constructor(fileTest: BettererFileTestFunction);
    /**
     * When `true`, all other tests will be skipped. Calling `only()` will set this to `true`.
     */
    get isOnly(): boolean;
    /**
     * When `true`, this test will be skipped. Calling `skip()` will set this to `true`.
     */
    get isSkipped(): boolean;
    /**
     * Override the constraint in the test configuration.
     *
     * @param constraintOverride - The new constraint for the test.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    constraint(constraintOverride: BettererTestConstraint<BettererFileTestResult>): this;
    /**
     * Override the deadline in the test configuration.
     *
     * @param deadlineOverride - The new deadline for the test.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    deadline(deadlineOverride: BettererTestDeadline): this;
    /**
     * Add a list of {@link https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions | Regular Expression } filters for files to exclude when running the test.
     *
     * @param excludePatterns - RegExp filters to match file paths that should be excluded.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    exclude(...excludePatterns: BettererFilePatterns): this;
    /**
     * Override the goal in the test configuration.
     *
     * @param goalOverride - The new goal for the test.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    goal(goalOverride: BettererTestGoal<BettererFileTestResult>): this;
    /**
     * Add a list of {@link https://www.npmjs.com/package/glob#user-content-glob-primer | glob }
     * patterns for files to include when running the test.
     *
     * @param includePatterns - Glob patterns to match file paths that should be included. All
     * `includes` should be relative to the {@link https://phenomnomnominal.github.io/betterer/docs/test-definition-file | test definition file}.
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    include(...includePatterns: BettererFileGlobs): this;
    /**
     * Run only this test. All other tests will be marked as skipped.
     *
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    only(): this;
    /**
     * Skip this test.
     *
     * @returns This {@link @betterer/betterer#BettererFileTest | `BettererFileTest`}, so it is chainable.
     */
    skip(): this;
}
export declare function isBettererFileTest(test: unknown): test is BettererFileTest;
//# sourceMappingURL=file-test.d.ts.map