/**
 * Copyright (c) 2021 OceanBase
 * OceanBase CE is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *          http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */

#define USING_LOG_PREFIX SQL_DAS
#include "sql/das/ob_das_rpc_processor.h"
#include "sql/ob_sql.h"
#include "sql/das/ob_data_access_service.h"
#include "sql/engine/px/ob_px_interruption.h"
#include "share/detect/ob_detect_manager_utils.h"

namespace oceanbase
{
namespace sql
{
template<obrpc::ObRpcPacketCode pcode>
int ObDASBaseAccessP<pcode>::init()
{
  int ret = OB_SUCCESS;
  ObDASTaskArg &task = RpcProcessor::arg_;
  ObDASBaseAccessP<pcode>::get_das_factory() = &das_factory_;
  memset(monitor_val_, 0, sizeof(monitor_val_));
  tsc_monitor_info_.init(&monitor_val_[0], &monitor_val_[1], &monitor_val_[2], &monitor_val_[3]);
  das_remote_info_.tsc_monitor_info_ = &tsc_monitor_info_;
  das_remote_info_.exec_ctx_ = &exec_ctx_;
  das_remote_info_.frame_info_ = &frame_info_;
  task.set_remote_info(&das_remote_info_);
  ObDASRemoteInfo::get_remote_info() = &das_remote_info_;
  return ret;
}

template<obrpc::ObRpcPacketCode pcode>
int ObDASBaseAccessP<pcode>::before_process()
{
  int ret = OB_SUCCESS;
  ObDASTaskArg &task = RpcProcessor::arg_;
  ObDASTaskResp &task_resp = RpcProcessor::result_;
  ObMemAttr mem_attr;
  mem_attr.tenant_id_ = task.get_task_op()->get_tenant_id();
  mem_attr.label_ = "DASRpcPCtx";
  exec_ctx_.get_allocator().set_attr(mem_attr);
  // ash stat should be setted already in rpc thread.
  ObDiagnosticInfo *di = ObLocalDiagnosticInfo::get();
  if (OB_NOT_NULL(di)) {
    di->get_ash_stat().in_das_remote_exec_ = true;
    di->get_ash_stat().tenant_id_ = task.get_task_op()->get_tenant_id();
    di->get_ash_stat().trace_id_ = *ObCurTraceId::get_trace_id();
    di->get_ash_stat().user_id_ = das_remote_info_.user_id_;
    di->get_ash_stat().plan_id_ = das_remote_info_.plan_id_;
    di->get_ash_stat().plan_hash_ = das_remote_info_.plan_hash_;
    MEMCPY(di->get_ash_stat().sql_id_, das_remote_info_.sql_id_,
        min(sizeof(di->get_ash_stat().sql_id_), sizeof(das_remote_info_.sql_id_)));
  }

  {
    omt::ObTenantConfigGuard tenant_config(TENANT_CONF(task.get_task_op()->get_tenant_id()));
    if (tenant_config.is_valid()) {
      is_enable_sqlstat_ = tenant_config->_ob_sqlstat_enable && lib::is_diagnose_info_enabled();
      // sqlstat has a dependency on the statistics mechanism, so turning off perf event will turn off sqlstat at the same time.
    }
  }

  if (is_enable_sqlstat_) {
    sqlstat_record_.record_sqlstat_start_value();
    sqlstat_key_.set_sql_id(ObString::make_string(das_remote_info_.sql_id_));
    sqlstat_key_.set_plan_hash(das_remote_info_.plan_hash_);
    sqlstat_key_.set_source_addr(ObRpcProcessorBase::get_peer());
  }

  if (OB_FAIL(RpcProcessor::before_process())) {
    LOG_WARN("do rpc processor before_process failed", K(ret));
  } else if (das_remote_info_.need_calc_expr_ &&
      OB_FAIL(GCTX.schema_service_->get_tenant_schema_guard(MTL_ID(), schema_guard_))) {
    LOG_WARN("fail to get schema guard", K(ret));
  } else {
    exec_ctx_.get_sql_ctx()->schema_guard_ = &schema_guard_;
  }
  return ret;
}

ERRSIM_POINT_DEF(EN_DAS_SIMULATE_EXTRA_RESULT_MEMORY_LIMIT);

template<obrpc::ObRpcPacketCode pcode>
int ObDASBaseAccessP<pcode>::process()
{
  int ret = OB_SUCCESS;
  LOG_DEBUG("DAS base access remote process", K_(RpcProcessor::arg));
  ObDASTaskArg &task = RpcProcessor::arg_;
  ObDASTaskResp &task_resp = RpcProcessor::result_;
  SQL_INFO_GUARD(ObString("DAS REMOTE PROCESS"), task.get_remote_info()->sql_id_);
  const common::ObSEArray<ObIDASTaskOp*, 2> &task_ops = task.get_task_ops();
  common::ObSEArray<ObIDASTaskResult*, 2> &task_results = task_resp.get_op_results();
  ObDASTaskFactory *das_factory = ObDASBaseAccessP<pcode>::get_das_factory();
  ObIDASTaskResult *op_result = nullptr;
  ObIDASTaskOp *task_op = nullptr;
  bool has_more = false;
  int64_t memory_limit = das::OB_DAS_MAX_PACKET_SIZE;
  ObDASTCBInterruptInfo interrupt_info;
  bool has_set_interrupt = false;
  ObInterruptibleTaskID interrupt_id;
  ObRegisterDmInfo dm_info;
  dm_info.detectable_id_ = task.get_remote_info()->detectable_id_;
  dm_info.addr_ = task.get_ctrl_svr();

#ifdef ERRSIM
  if (EN_DAS_SIMULATE_EXTRA_RESULT_MEMORY_LIMIT) {
    memory_limit = -EN_DAS_SIMULATE_EXTRA_RESULT_MEMORY_LIMIT;
    LOG_INFO("das simulate extra result memory limit", K(memory_limit));
  }
#endif
  //regardless of the success of the task execution, the following meta info must be set
  task_resp.set_ctrl_svr(task.get_ctrl_svr());
  task_resp.set_runner_svr(task.get_runner_svr());
  if (task_ops.count() == 0 || task_results.count() != 0) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("task op unexpected", K(ret), K(task_ops), K(task_results));
  } else {
    for (int i = 0; OB_SUCC(ret) && i < task_ops.count(); i++) {
      if (OB_ISNULL(task_op = task_ops.at(i))) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("task op is null", KR(ret));
      } else {
        DASTCBInfo tcb_info(task_op->task_id_);
        uint64_t check_node_sequence_id = 0;
        bool has_register_check_item = false;

        GET_DIAGNOSTIC_INFO->get_ash_stat().plan_line_id_ = task_op->plan_line_id_;
        ACTIVE_SESSION_RETRY_DIAG_INFO_SETTER(ls_id_, task_op->get_ls_id().id());

        if ((task_op->get_type() == DAS_OP_TABLE_BATCH_SCAN ||
             task_op->get_type() == DAS_OP_TABLE_SCAN) &&
            !dm_info.detectable_id_.is_invalid()) {
          if (!has_set_interrupt) {
            if (OB_ISNULL(GCTX.sql_engine_)) {
              ret = OB_ERR_UNEXPECTED;
              LOG_WARN("sql engine is null", K(ret), KPC(task_op));
            } else if (OB_FAIL(ObInterruptUtil::generate_query_interrupt_id((uint32_t)GCTX.get_server_index(),
                                                                            (uint64_t)GCTX.sql_engine_->get_px_sequence_id(),
                                                                            interrupt_id))) {
              LOG_WARN("generate interrupt id failed", KR(ret));
            } else if (OB_FAIL(SET_INTERRUPTABLE(interrupt_id))) {
              LOG_WARN("register interrupt failed", KR(ret));
            } else {
              has_set_interrupt = true;
              interrupt_info.interrupt_id_ = interrupt_id;
              interrupt_info.self_addr_ = task.get_runner_svr();
            }
          }

          if (OB_FAIL(ret)) {
          } else if (OB_FAIL(ObDetectManagerUtils::das_task_register_check_item_into_dm(dm_info,
                                                                                        interrupt_id,
                                                                                        tcb_info,
                                                                                        check_node_sequence_id))) {
            LOG_WARN("[DM] register check item result to dm failed", K(ret));
          } else {
            has_register_check_item = true;
            interrupt_info.detectable_id_ = dm_info.detectable_id_;
            interrupt_info.check_node_sequence_id_ = check_node_sequence_id;
          }
        }

        if (OB_FAIL(ret)) {
        } else if (OB_FAIL(das_factory->create_das_task_result(task_op->get_type(), op_result))) {
          LOG_WARN("create das task result failed", K(ret));
        } else if (OB_FAIL(task_resp.add_op_result(op_result))) {
          LOG_WARN("failed to add op result", K(ret));
        } else if (OB_FAIL(op_result->init(*task_op, CURRENT_CONTEXT->get_arena_allocator()))) {
          LOG_WARN("failed to init op result", K(ret));
        } else if (FALSE_IT(op_result->set_task_id(task_op->get_task_id()))) {
        } else if (OB_FAIL(task_op->start_das_task())) {
          LOG_WARN("start das task failed", K(ret));
        } else if (OB_FAIL(task_op->fill_task_result(*task_results.at(i), has_more, memory_limit))) {
          LOG_WARN("fill task result to controller failed", K(ret));
        } else if (OB_UNLIKELY(has_more) && OB_FAIL(task_op->fill_extra_result(interrupt_info))) {
          LOG_WARN("fill extra result to controller failed", KR(ret));
        } else {
          task_resp.set_has_more(has_more);
          ObWarningBuffer *wb = ob_get_tsi_warning_buffer();
          if (wb != nullptr) {
            //ignore the errcode of storing warning msg
            (void)task_resp.store_warning_msg(*wb);
          }
        }

        if (OB_FAIL(ret) && OB_NOT_NULL(op_result)) {
          // accessing failed das task result is undefined behavior.
          op_result->reuse();
        }
        //因为end_task还有可能失败，需要通过RPC将end_task的返回值带回到scheduler上
        int tmp_ret = task_op->end_das_task();
        if (OB_SUCCESS != tmp_ret) {
          LOG_WARN("end das task failed", K(ret), K(tmp_ret), K(task));
        }
        ret = COVER_SUCC(tmp_ret);
        if (OB_NOT_NULL(task_op->get_trans_desc())) {
          tmp_ret = MTL(transaction::ObTransService*)
            ->get_tx_exec_result(*task_op->get_trans_desc(),
                                task_resp.get_trans_result());
          if (OB_SUCCESS != tmp_ret) {
            LOG_WARN("get trans exec result failed", K(ret), K(task));
          }
          ret = COVER_SUCC(tmp_ret);
        }
        if (OB_SUCCESS != ret && is_schema_error(ret)) {
          ret = GSCHEMASERVICE.is_schema_error_need_retry(NULL, task_op->get_tenant_id()) ?
          OB_ERR_REMOTE_SCHEMA_NOT_FULL : OB_ERR_WAIT_REMOTE_SCHEMA_REFRESH;
        }
        task_resp.set_err_code(ret);
        if (OB_SUCCESS != ret) {
          task_resp.store_err_msg(ob_get_tsi_err_msg(ret));
          LOG_WARN("process das access task failed", K(ret),
                  K(task.get_ctrl_svr()), K(task.get_runner_svr()));
        }

        // Error or no more results, need to unregister check item from dm
        if (has_register_check_item && (OB_FAIL(ret) || !has_more)) {
          ObDetectManagerUtils::das_task_unregister_check_item_from_dm(dm_info.detectable_id_, check_node_sequence_id);
        }

        if (has_more || memory_limit < 0) {
          /**
          * das serialized execution.
          * If current resp buffer is overflow. We would reply result
          * directly. following un-executed tasks would be executed
          * later remotely.
          *
          * the insert op won't set has_more flag, but if it exceed the
          * threshold of memory_limit, we should reply anyway.
          */
          LOG_DEBUG("reply das result due to memory limit exceeded.",
              K(has_more), K(memory_limit), K(i), K(task_ops.count()));
          break;
        }
        LOG_DEBUG("process das base access task", K(ret), KPC(task_op), KPC(op_result), K(has_more));
      }
    }

    if (has_set_interrupt) {
      // unregister interrupt checker
      UNSET_INTERRUPTABLE(interrupt_id);
    }
  }
  return OB_SUCCESS;
}

template<obrpc::ObRpcPacketCode pcode>
int ObDASBaseAccessP<pcode>::after_process(int error_code)
{
  int ret = OB_SUCCESS;
  const int64_t elapsed_time = common::ObTimeUtility::current_time() - RpcProcessor::get_receive_timestamp();
  if (OB_FAIL(RpcProcessor::after_process(error_code))) {
    LOG_WARN("do das base rpc process failed", K(ret));
  } else if (elapsed_time >= ObServerConfig::get_instance().trace_log_slow_query_watermark) {
    //slow das task, print trace info
    FORCE_PRINT_TRACE(THE_TRACE, "[slow das rpc process]");
  } else {
    if (is_enable_sqlstat_) {
      sqlstat_record_.record_sqlstat_end_value();
      sqlstat_record_.move_to_sqlstat_cache(sqlstat_key_);
    }
  }
  //执行相关的错误信息不用传递给RPC框架，RPC框架不处理具体的RPC执行错误信息，始终返回OB_SUCCESS
  return OB_SUCCESS;
}

template<obrpc::ObRpcPacketCode pcode>
void ObDASBaseAccessP<pcode>::cleanup()
{
  GET_DIAGNOSTIC_INFO->get_ash_stat().in_das_remote_exec_ = false;
  das_factory_.cleanup();
  ObDASBaseAccessP<pcode>::get_das_factory() = nullptr;
  if (das_remote_info_.trans_desc_ != nullptr) {
    MTL(transaction::ObTransService*)->release_tx(*das_remote_info_.trans_desc_);
    das_remote_info_.trans_desc_ = nullptr;
  }
  RpcProcessor::cleanup();
}

int ObDASSyncAccessP::process()
{
  int ret = OB_SUCCESS;
  LOG_DEBUG("DAS sync access remote process", K_(arg));
  FLTSpanGuard(das_sync_rpc_process);
  if (OB_FAIL(ObDASSyncRpcProcessor::process())) {
    LOG_WARN("failed to process das sync rpc", K(ret));
  }
  return OB_SUCCESS;
}

int ObDASAsyncAccessP::process()
{
  int ret = OB_SUCCESS;
  LOG_DEBUG("DAS async access remote process", K_(arg));
  FLTSpanGuard(das_async_rpc_process);
  if (OB_FAIL(ObDASAsyncRpcProcessor::process())) {
    LOG_WARN("failed to process das async rpc", K(ret));
  }
  return OB_SUCCESS;
}

void ObRpcDasAsyncAccessCallBack::on_timeout()
{
  int ret = OB_TIMEOUT;
  int64_t current_ts = ObTimeUtility::current_time();
  int64_t timeout_ts = context_->get_timeout_ts();
  is_timeout_ = true;
  // ESTIMATE_PS_RESERVE_TIME = 100 * 1000
  if (timeout_ts - current_ts > 100 * 1000) {
    LOG_DEBUG("rpc return OB_TIMEOUT before actual timeout, change error code to OB_RPC_CONNECT_ERROR", KR(ret),
              K(timeout_ts), K(current_ts));
    ret = OB_RPC_CONNECT_ERROR;
  }
  LOG_WARN("das async task timeout", KR(ret), K(get_task_ops()));
  result_.set_err_code(ret);
  result_.get_op_results().reuse();
  context_->get_ref_count_ctx().inc_concurrency_limit_with_signal();
}

void ObRpcDasAsyncAccessCallBack::on_invalid()
{
  int ret = OB_SUCCESS;
  is_invalid_ = true;
  // a valid packet on protocol level, but can't decode it.
  ret = get_error() == OB_ALLOCATE_MEMORY_FAILED ?
        OB_ALLOCATE_MEMORY_FAILED : OB_INVALID_ERROR;
  LOG_WARN("das async task invalid", K(get_task_ops()));
  result_.set_err_code(ret);
  result_.get_op_results().reuse();
  context_->get_ref_count_ctx().inc_concurrency_limit_with_signal();
}

void ObRpcDasAsyncAccessCallBack::set_args(const Request &arg)
{
  UNUSED(arg);
}

int ObRpcDasAsyncAccessCallBack::process()
{
  int ret = OB_SUCCESS;
  is_processed_ = true;
  LOG_DEBUG("DAS async access callback process", K_(result));
  if (OB_FAIL(get_rcode())) {
    result_.set_err_code(get_rcode());
    // we need to clear op results because they are not decoded from das async rpc due to rpc error.
    result_.get_op_results().reuse();
    LOG_WARN("das async rpc execution failed", K(get_rcode()), K_(result));
  }
  context_->get_ref_count_ctx().inc_concurrency_limit_with_signal();
  return ret;
}

oceanbase::rpc::frame::ObReqTransport::AsyncCB *ObRpcDasAsyncAccessCallBack::clone(
    const oceanbase::rpc::frame::SPAlloc &alloc) const {
  UNUSED(alloc);
  return const_cast<rpc::frame::ObReqTransport::AsyncCB *>(
      static_cast<const rpc::frame::ObReqTransport::AsyncCB * const>(this));
}

int ObDasAsyncRpcCallBackContext::init(const ObMemAttr &attr)
{
  alloc_.set_attr(attr);
  return task_ops_.get_copy_assign_ret();
};

int ObDASSyncFetchP::before_process()
{
  int ret = OB_SUCCESS;
  ObDASDataFetchReq &req = arg_;
  ObDiagnosticInfo *di = ObLocalDiagnosticInfo::get();
  if (OB_NOT_NULL(di)) {
    di->get_ash_stat().in_das_remote_exec_ = true;
    di->get_ash_stat().tenant_id_ = req.get_tenant_id();
    di->get_ash_stat().trace_id_ = *ObCurTraceId::get_trace_id();
  }

  if (OB_FAIL(ObDASSyncFetchResRpcProcessor::before_process())) {
    LOG_WARN("do rpc processor before_process failed", K(ret));
  }
  return ret;
}

int ObDASSyncFetchP::process()
{
  int ret = OB_SUCCESS;
  FLTSpanGuard(fetch_das_result_process);
  ObDASDataFetchReq &req = arg_;
  ObDASDataFetchRes &res = result_;
  ObDataAccessService *das = NULL;
  const uint64_t tenant_id = req.get_tenant_id();
  const int64_t task_id = req.get_task_id();
  if (tenant_id != MTL_ID()) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("wrong tenant id", KR(ret), K(req));
  } else if (OB_FAIL(res.init(tenant_id, task_id))) {
    LOG_WARN("init res failed", KR(ret), K(req));
  } else if (OB_ISNULL(das = MTL(ObDataAccessService *))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("das is null", KR(ret), KP(das));
  } else if (OB_FAIL(das->get_task_res_mgr().iterator_task_result(res,
                                                                  res.io_read_bytes_,
                                                                  res.ssstore_read_bytes_,
                                                                  res.ssstore_read_row_cnt_,
                                                                  res.memstore_read_row_cnt_))) {
    if (OB_UNLIKELY(OB_ENTRY_NOT_EXIST == ret)) {
      // After server reboot, the hash map containing task results was gone.
      // We need to retry for such cases.
      LOG_WARN("task result was gone due to server reboot, will retry", KR(ret), K(res));
      ret = OB_RPC_SEND_ERROR;
    } else {
      LOG_WARN("get task result failed", KR(ret), K(res));
    }
  }
  return ret;
}

int ObDASSyncFetchP::after_process(int error_code)
{
  int ret = OB_SUCCESS;
  const int64_t elapsed_time = common::ObTimeUtility::current_time() - get_receive_timestamp();
  if (OB_FAIL(ObDASSyncFetchResRpcProcessor::after_process(error_code))) {
    LOG_WARN("do das sync base rpc process failed", K(ret));
  } else if (elapsed_time >= ObServerConfig::get_instance().trace_log_slow_query_watermark) {
    //slow das task, print trace info
    FORCE_PRINT_TRACE(THE_TRACE, "[slow das rpc process]");
  }
  //执行相关的错误信息不用传递给RPC框架，RPC框架不处理具体的RPC执行错误信息，始终返回OB_SUCCESS
  return OB_SUCCESS;
}

void ObDASSyncFetchP::cleanup()
{
  GET_DIAGNOSTIC_INFO->get_ash_stat().in_das_remote_exec_ = false;
  ObDASSyncFetchResRpcProcessor::cleanup();
}

int ObDASAsyncEraseP::process()
{
  int ret = OB_SUCCESS;
  ObDASDataEraseReq &req = arg_;
  ObDataAccessService *das = NULL;
  const uint64_t tenant_id = req.get_tenant_id();
  const int64_t task_id = req.get_task_id();
  if (OB_ISNULL(das = MTL(ObDataAccessService *))) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("das is null", KR(ret), KP(das));
  } else if (OB_FAIL(das->get_task_res_mgr().erase_task_result(task_id, true))) {
    LOG_WARN("erase task result failed", KR(ret), K(task_id));
  }
  return ret;
}
}  // namespace sql
}  // namespace oceanbase
