// Copyright (c) 2023 Alibaba Cloud
//
// SPDX-License-Identifier: Apache-2.0
//

//! Fetch confidential resources from KBS (Relying Party).

//! All the fetched resources will be stored in a local filepath:
//! `/run/image-security/kbs/<filename>`
//!
//! The `<filename>` will be generated by hash256sum the KBS Resource URI/
//! For example:
//! `kbs://example.org/alice/key/1` will be stored in
//! `/run/image-security/kbs/cde48578964b30b0aa8cecf04c020f64f7cce36fc391b24f45cf8d4e5368e229`

use std::path::{Path, PathBuf};

use anyhow::Result;
use async_trait::async_trait;
use log::info;
use sha2::{Digest, Sha256};
use tokio::fs;

#[cfg(feature = "keywrap-grpc")]
mod grpc;

#[cfg(feature = "keywrap-ttrpc")]
mod ttrpc;

#[cfg(feature = "keywrap-ttrpc")]
mod ttrpc_proto;

#[cfg(feature = "keywrap-native")]
mod native;

/// Default workdir to store downloaded kbs resources
const STORAGE_PATH: &str = "image-security/kbs";

/// SecureChannel to connect with KBS
pub struct SecureChannel {
    /// Get Resource Service client.
    client: Box<dyn Client>,

    /// The path to store downloaded kbs resources
    pub storage_path: PathBuf,
}

impl Default for SecureChannel {
    fn default() -> Self {
        Self::new(
            "sample_kbc",
            "null",
            Path::new(crate::config::DEFAULT_WORK_DIR),
        )
        .expect("initialize default secure channel")
    }
}

#[async_trait]
trait Client: Send + Sync {
    async fn get_resource(&self, resource_path: &str) -> Result<Vec<u8>>;
}

impl SecureChannel {
    /// Create a new [`SecureChannel`], the input parameter:
    /// * `decrypt_config`: s string with format `provider:attestation-agent:<kbc_name>::<kbs_uri>`.
    ///   This parameter is only used when in native secure channel (for enclave-cc)
    pub fn new(_kbc_name: &str, _kbs_uri: &str, work_dir: &Path) -> Result<Self> {
        let client: Box<dyn Client> = {
            cfg_if::cfg_if! {
                if #[cfg(feature = "keywrap-ttrpc")] {
                    info!("secure channel uses ttrpc");
                    Box::<ttrpc::Ttrpc>::default()
                } else if #[cfg(feature = "keywrap-native")] {
                    info!("secure channel uses native-aa");
                    Box::new(native::Native::new(_kbc_name, _kbs_uri)?)
                } else if #[cfg(feature = "keywrap-grpc")] {
                    info!("secure channel uses gRPC");
                    Box::<grpc::Grpc>::default()
                } else  {
                    compile_error!("At last one feature of `keywrap-grpc`, `keywrap-ttrpc`, and `keywrap-native` must be enabled.");
                }
            }
        };

        let storage_path = work_dir.join(STORAGE_PATH);
        std::fs::create_dir_all(&storage_path)?;

        Ok(Self {
            client,
            storage_path,
        })
    }

    /// Check whether the resource of the uri has been downloaded.
    /// Return Some(_) if exists, and return None if not.
    async fn check_local(&self, uri: &str) -> Result<Option<Vec<u8>>> {
        let file_path = self.get_filepath(uri);
        match Path::new(&file_path).exists() {
            true => {
                let contents = fs::read(&file_path).await?;
                Ok(Some(contents))
            }
            false => Ok(None),
        }
    }

    /// Get the localpath to store the kbs resource in the local filesystem
    fn get_filepath(&self, uri: &str) -> PathBuf {
        let mut sha256 = Sha256::new();
        sha256.update(uri.as_bytes());
        let filename = format!("{:x}", sha256.finalize());
        self.storage_path.join(filename)
    }
}

impl SecureChannel {
    /// Get resource from using, using `resource_name` as `name` in a ResourceDescription,
    /// then save the gathered data into `path`
    ///
    /// Please refer to https://github.com/confidential-containers/guest-components/blob/main/image-rs/docs/ccv1_image_security_design.md#get-resource-service
    /// for more information.
    pub async fn get_resource(&self, resource_uri: &str) -> Result<Vec<u8>> {
        if let Some(res) = self.check_local(resource_uri).await? {
            return Ok(res);
        }

        let res = self.client.get_resource(resource_uri).await?;

        let path = self.get_filepath(resource_uri);
        fs::write(path, &res).await?;
        Ok(res)
    }
}
