"use strict";
// Copyright (c) Microsoft Corporation. All rights reserved. Licensed under the MIT license.
// See LICENSE in the project root for license information.
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiClass = void 0;
const DeclarationReference_1 = require("@microsoft/tsdoc/lib/beta/DeclarationReference");
const ApiDeclaredItem_1 = require("../items/ApiDeclaredItem");
const ApiItemContainerMixin_1 = require("../mixins/ApiItemContainerMixin");
const ApiReleaseTagMixin_1 = require("../mixins/ApiReleaseTagMixin");
const HeritageType_1 = require("./HeritageType");
const ApiNameMixin_1 = require("../mixins/ApiNameMixin");
const ApiTypeParameterListMixin_1 = require("../mixins/ApiTypeParameterListMixin");
/**
 * Represents a TypeScript class declaration.
 *
 * @remarks
 *
 * This is part of the {@link ApiModel} hierarchy of classes, which are serializable representations of
 * API declarations.
 *
 * `ApiClass` represents a TypeScript declaration such as this:
 *
 * ```ts
 * export class X { }
 * ```
 *
 * @public
 */
class ApiClass extends ApiItemContainerMixin_1.ApiItemContainerMixin(ApiNameMixin_1.ApiNameMixin(ApiTypeParameterListMixin_1.ApiTypeParameterListMixin(ApiReleaseTagMixin_1.ApiReleaseTagMixin(ApiDeclaredItem_1.ApiDeclaredItem)))) {
    constructor(options) {
        super(options);
        this._implementsTypes = [];
        if (options.extendsTokenRange) {
            this.extendsType = new HeritageType_1.HeritageType(this.buildExcerpt(options.extendsTokenRange));
        }
        else {
            this.extendsType = undefined;
        }
        for (const implementsTokenRange of options.implementsTokenRanges) {
            this._implementsTypes.push(new HeritageType_1.HeritageType(this.buildExcerpt(implementsTokenRange)));
        }
    }
    static getContainerKey(name) {
        return `${name}|${"Class" /* Class */}`;
    }
    /** @override */
    static onDeserializeInto(options, context, jsonObject) {
        super.onDeserializeInto(options, context, jsonObject);
        options.extendsTokenRange = jsonObject.extendsTokenRange;
        options.implementsTokenRanges = jsonObject.implementsTokenRanges;
    }
    /** @override */
    get kind() {
        return "Class" /* Class */;
    }
    /** @override */
    get containerKey() {
        return ApiClass.getContainerKey(this.name);
    }
    /**
     * The list of interfaces that this class implements using the `implements` keyword.
     */
    get implementsTypes() {
        return this._implementsTypes;
    }
    /** @override */
    serializeInto(jsonObject) {
        super.serializeInto(jsonObject);
        // Note that JSON does not support the "undefined" value, so we simply omit the field entirely if it is undefined
        if (this.extendsType) {
            jsonObject.extendsTokenRange = this.extendsType.excerpt.tokenRange;
        }
        jsonObject.implementsTokenRanges = this.implementsTypes.map((x) => x.excerpt.tokenRange);
    }
    /** @beta @override */
    buildCanonicalReference() {
        const nameComponent = DeclarationReference_1.DeclarationReference.parseComponent(this.name);
        return (this.parent ? this.parent.canonicalReference : DeclarationReference_1.DeclarationReference.empty())
            .addNavigationStep("." /* Exports */, nameComponent)
            .withMeaning("class" /* Class */);
    }
}
exports.ApiClass = ApiClass;
//# sourceMappingURL=ApiClass.js.map