"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseResults = void 0;
const tslib_1 = require("tslib");
const errors_1 = require("@betterer/errors");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const fs_1 = require("../fs");
const merge_1 = require("./merge");
const require_1 = require("./require");
const MERGE_CONFLICT_ANCESTOR = '|||||||';
const MERGE_CONFLICT_END = '>>>>>>>';
const MERGE_CONFLICT_SEP = '=======';
const MERGE_CONFLICT_START = '<<<<<<<';
/**
 * Parses the contents of a given results file path. If the file doesn't exist, it will
 * return an empty object. If the file exists, but has merge conflicts, it will merge the
 * files using {@link mergeResults | `mergeResults`}.
 *
 * @throws {@link @betterer/errors#BettererError | `BettererError` }
 * Throws if the results file cannot be parsed, or if it contains merge conflicts that
 * can't be resolved.
 */
async function parseResults(resultsPath) {
    const contents = await (0, fs_1.read)(resultsPath);
    if (!contents) {
        return {};
    }
    if (hasMergeConflicts(contents)) {
        try {
            const [ours, theirs] = extractConflicts(contents);
            return (0, merge_1.mergeResults)(ours, theirs);
        }
        catch (error) {
            throw new errors_1.BettererError(`could not resolve merge conflict in "${resultsPath}". 😔`, error);
        }
    }
    try {
        return await (0, require_1.requireText)(contents);
    }
    catch (_a) {
        throw new errors_1.BettererError(`could not read results from "${resultsPath}". 😔`);
    }
}
exports.parseResults = parseResults;
function hasMergeConflicts(str) {
    return str.includes(MERGE_CONFLICT_START) && str.includes(MERGE_CONFLICT_SEP) && str.includes(MERGE_CONFLICT_END);
}
function extractConflicts(file) {
    const ours = [];
    const theirs = [];
    const lines = file.split(/\r?\n/g);
    let skip = false;
    while (lines.length) {
        const line = lines.shift();
        (0, assert_1.default)(line != null);
        if (line.startsWith(MERGE_CONFLICT_START)) {
            // get our file
            while (lines.length) {
                const conflictLine = lines.shift();
                (0, assert_1.default)(conflictLine != null);
                if (conflictLine === MERGE_CONFLICT_SEP) {
                    skip = false;
                    break;
                }
                else if (skip || conflictLine.startsWith(MERGE_CONFLICT_ANCESTOR)) {
                    skip = true;
                    continue;
                }
                else {
                    ours.push(conflictLine);
                }
            }
            // get their file
            while (lines.length) {
                const conflictLine = lines.shift();
                (0, assert_1.default)(conflictLine != null);
                if (conflictLine.startsWith(MERGE_CONFLICT_END)) {
                    break;
                }
                else {
                    theirs.push(conflictLine);
                }
            }
        }
        else {
            ours.push(line);
            theirs.push(line);
        }
    }
    return [ours.join('\n'), theirs.join('\n')];
}
//# sourceMappingURL=parse.js.map