"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererSuiteΩ = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const utils_1 = require("../utils");
const suite_summary_1 = require("./suite-summary");
const NEGATIVE_FILTER_TOKEN = '!';
class BettererSuiteΩ {
    constructor(_config, _resultsFile, filePaths, runs) {
        this._config = _config;
        this._resultsFile = _resultsFile;
        this.filePaths = filePaths;
        this.runs = runs;
        this._reporter = this._config.reporter;
    }
    async run(isRunOnce = false) {
        // Attach lifecycle promises for Reporters:
        const runLifecycles = this.runs.map((run) => {
            const lifecycle = (0, utils_1.defer)();
            run.lifecycle = lifecycle.promise;
            return lifecycle;
        });
        const runsLifecycle = (0, utils_1.defer)();
        // Don't await here! A custom reporter could be awaiting
        // the lifecycle promise which is unresolved right now!
        const reportSuiteStart = this._reporter.suiteStart(this, runsLifecycle.promise);
        try {
            const runSummaries = await this._runTests(runLifecycles);
            const changed = this._resultsFile.getChanged(runSummaries);
            const suiteSummaryΩ = new suite_summary_1.BettererSuiteSummaryΩ(this.filePaths, this.runs, runSummaries, changed);
            runsLifecycle.resolve(suiteSummaryΩ);
            await reportSuiteStart;
            await this._reporter.suiteEnd(suiteSummaryΩ);
            if (!isRunOnce && suiteSummaryΩ && !this._config.ci) {
                await this._resultsFile.writeNew(suiteSummaryΩ);
            }
            return suiteSummaryΩ;
        }
        catch (error) {
            runsLifecycle.reject(error);
            await reportSuiteStart;
            await this._reporter.suiteError(this, error);
            throw error;
        }
    }
    async _runTests(runLifecycles) {
        const runsΩ = this.runs;
        const hasOnly = !!runsΩ.find((run) => run.testMeta.isOnly);
        const { filters } = this._config;
        return await Promise.all(runsΩ.map(async (runΩ, index) => {
            const lifecycle = runLifecycles[index];
            // This is all a bit backwards because "filters" is named badly.
            const hasFilters = !!filters.length;
            // And this is some madness which applies filters and negative filters in
            // the order they are read:
            //
            // ["foo"] => [/foo/] => ["foo"]
            // ["foo"] => [/bar/] => []
            // ["foo"] => [/!foo/] => []
            // ["foo"] => [/!bar/] => ["foo"]
            // ["foo"] => [/foo/, /!foo/] => []
            // ["foo"] => [/!foo/, /foo/] => ["foo"]
            const isSelected = filters.reduce((selected, filter) => {
                const isNegated = filter.source.startsWith(NEGATIVE_FILTER_TOKEN);
                if (selected) {
                    if (isNegated) {
                        const negativeFilter = new RegExp(filter.source.substr(1), filter.flags);
                        return !negativeFilter.test(runΩ.name);
                    }
                    return selected;
                }
                else {
                    if (isNegated) {
                        const negativeFilter = new RegExp(filter.source.substr(1), filter.flags);
                        return !negativeFilter.test(runΩ.name);
                    }
                    return filter.test(runΩ.name);
                }
            }, false);
            const isOtherTestOnly = hasOnly && !runΩ.testMeta.isOnly;
            const isSkipped = (hasFilters && !isSelected) || isOtherTestOnly || runΩ.testMeta.isSkipped;
            // Don't await here! A custom reporter could be awaiting
            // the lifecycle promise which is unresolved right now!
            const reportRunStart = this._reporter.runStart(runΩ, lifecycle.promise);
            const runSummary = await runΩ.run(isSkipped);
            // `filePaths` will be updated in the worker if the test filters the files
            // so it needs to be updated
            runΩ.filePaths = runSummary.filePaths;
            if (runSummary.isFailed) {
                const { error } = runSummary;
                (0, assert_1.default)(error);
                lifecycle.reject(error);
                await reportRunStart;
                await this._reporter.runError(runΩ, error);
            }
            else {
                lifecycle.resolve(runSummary);
                await reportRunStart;
                await this._reporter.runEnd(runSummary);
            }
            return runSummary;
        }));
    }
}
exports.BettererSuiteΩ = BettererSuiteΩ;
//# sourceMappingURL=suite.js.map