"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isBettererError = exports.BettererError = void 0;
/**
 * @public A custom Error for use in **Betterer**. It attaches some extra details to a standard
 * JavaScript error for better logging and debugging.
 *
 * @param message - The error message.
 * @param details - Extra details for the error.
 */
class BettererError extends Error {
    constructor(message, ...details) {
        super(message);
        /**
         * Used by {@link @betterer/errors#isBettererError | `isBettererError()`} to detect that an
         * object is an instance of `BettererError`.
         */
        this.isBettererError = true;
        Error.captureStackTrace(this, BettererError);
        Object.setPrototypeOf(this, new.target.prototype);
        this.details = details;
    }
}
exports.BettererError = BettererError;
/**
 * @public Check if an object is a {@link BettererError | `BettererError`}.
 *
 * @example
 * ```typescript
 * import { BettererError, isBettererError } from '@betterer/errors';
 *
 * isBettererError(new Error()); // false
 * isBettererError(new BettererError()); // true
 * ```
 */
function isBettererError(err) {
    return !!err.isBettererError;
}
exports.isBettererError = isBettererError;
//# sourceMappingURL=error.js.map