from utils import pushd
import os
from random import randint
import shutil
import logging
import random
import string
from fallocate import fallocate, FALLOC_FL_PUNCH_HOLE, FALLOC_FL_KEEP_SIZE
from utils import Size, Unit
import xattr

"""
Generate and distribute target files(regular, symlink, directory), link files.
File with holes(sparse file)
Hardlink

1. Generate directory tree structure firstly.
"""

CHINESE_TABLE = "搀掺蝉馋谗缠铲产阐颤昌猖场尝常长偿肠厂敞畅唱倡超抄钞朝嘲潮巢吵炒车扯撤掣彻澈郴臣辰尘晨忱沉\
愤粪丰封枫蜂峰锋风疯烽逢冯缝讽奉凤佛否夫敷肤孵扶拂辐幅氟符伏俘服浮涪福袱弗甫抚辅俯釜斧脯腑\
楔些歇蝎鞋协挟携邪斜胁谐写械卸蟹懈泄泻谢屑薪芯锌欣辛新忻心信衅星腥猩惺兴刑型形邢行醒幸杏性\
寅饮尹引隐印英樱婴鹰应缨莹萤营荧蝇迎赢盈影颖硬映哟拥佣臃痈庸雍踊蛹咏泳涌永恿勇用幽优悠忧尤\
庥庠庹庵庾庳赓廒廑廛廨廪膺忄忉忖忏怃忮怄忡忤忾怅怆忪忭忸怙怵怦怛怏怍怩怫怊怿怡恸恹恻恺恂恪"


def gb2312(length):
    for i in range(0, length):
        c = random.choice(CHINESE_TABLE)
        yield c.encode("gb2312")


class Distributor:
    def __init__(self, top_dir: str, levels: int, max_sub_directories: int):
        self.top_dir = top_dir
        self.levels = levels
        self.max_sub_directories = max_sub_directories
        # All files generated by this distributor, no matter `_put_single_file()`
        # or `put_multiple_files` wll be recorded in this list.
        self.files = []
        self.symlinks = []
        self.dirs = []
        self.hardlinks = {}

    def _relative_path_to_top(self, path: str) -> str:
        return os.path.relpath(path, start=self.top_dir)

    def _generate_one_level(self, level, cur_dir):
        dirs = []
        with pushd(cur_dir):
            # At least, each level has a child directory
            for index in range(0, randint(1, self.max_sub_directories)):
                d_name = f"DIR.{level}.{index}"
                try:
                    d = os.mkdir(d_name)
                except FileExistsError:
                    pass
                dirs.append(d_name)

            if level >= self.levels:
                return

            for d in dirs:
                self._generate_one_level(level + 1, d)
        # This is top level planted tree.
        return dirs

    def generate_tree(self):
        """DIR.LEVEL.INDEX"""
        dirs = self._generate_one_level(0, self.top_dir)
        self.planted_tree_root = dirs[:]

    def _random_pos_dir(self):
        level = randint(0, self.levels)

        with pushd(os.path.join(self.top_dir, random.choice(self.planted_tree_root))):
            while level:
                files = os.listdir()
                level -= 1
                files = [f for f in files if os.path.isdir(f)]
                if len(files) != 0:
                    next_level = files[randint(0, len(files) - 1)]
                else:
                    break

                os.chdir(next_level)

            return os.getcwd()

    def put_hardlinks(self, count):
        def _create_new_source():
            source_file = os.path.join(
                self._random_pos_dir(), Distributor.generate_random_name(60)
            )
            fd = os.open(source_file, os.O_CREAT | os.O_RDWR)
            os.write(fd, os.urandom(randint(0, 1024 * 1024 + 7)))
            os.close(fd)
            return source_file

        source_file = _create_new_source()
        self.hardlinks[source_file] = []
        self.hardlink_aliases = []
        for i in range(0, count):

            if randint(0, 16) % 4 == 0:
                source_file = _create_new_source()
                self.hardlinks[source_file] = []

            link = os.path.join(
                self._random_pos_dir(),
                Distributor.generate_random_name(50, suffix="hardlink"),
            )
            logging.debug(link)
            # TODO: `link` may be too long to link, so better to change directory first!
            os.link(source_file, link)
            self.hardlinks[source_file].append(self._relative_path_to_top(link))
            self.hardlink_aliases.append(self._relative_path_to_top(link))
        return self.hardlink_aliases[-count:]

    def put_symlinks(self, count, chinese=False):
        """
        Generate symlinks pointing to regular files or directories.
        """

        def _create_new_source():
            this_path = ""
            if randint(0, 123) % 4 == 0:
                self.put_directories(1)
                this_path = self.dirs[-1]
                del self.dirs[-1]
            else:
                _, this_path = self._put_single_file(
                    self._random_pos_dir(),
                    Size(randint(0, 100), Unit.KB),
                    chinese=chinese,
                )
                del self.files[-1]

            return this_path

        source_file = _create_new_source()
        for i in range(0, count):
            if randint(0, 12) % 3 == 0:
                source_file = _create_new_source()

            symlink = os.path.join(
                self._random_pos_dir(),
                Distributor.generate_random_length_name(20, suffix="symlink"),
            )

            # XFS limits symlink target path which is stored within symlink length at 1024bytes.
            if len(source_file) >= 1024:
                continue

            if randint(0, 12) % 5 == 0:
                source_file = os.path.relpath(source_file, start=self.top_dir)

            try:
                os.symlink(source_file, symlink)
            except FileExistsError as e:
                # Sometimes, symlink fails due to an existed symlink file met.
                # This should rarely happen if `generate_random_length_name` truly randoms
                logging.exception(e)
                continue

            if randint(0, 12) % 4 == 0:
                try:
                    if os.path.isdir(source_file):
                        try:
                            os.rmdir(source_file)
                        except Exception:
                            pass
                    else:
                        os.unlink(source_file)
                except FileNotFoundError:
                    pass

            # Save symlink relative path so that we can tell which symlinks were put.
            self.symlinks.append(self._relative_path_to_top(symlink))

        return self.symlinks[-count:]

    def put_directories(self, count):
        for i in range(0, count):
            dst_path = os.path.join(
                self._random_pos_dir(),
                Distributor.generate_random_name(30, suffix="dir"),
            )

            # We might have a very long name of `dst_path`. So better to mkdir one by one
            dst_relpath = os.path.relpath(dst_path, start=self.top_dir)
            with pushd(self.top_dir):
                for d in dst_relpath.split("/")[0:]:
                    try:
                        os.chdir(d)
                    except FileNotFoundError:
                        os.mkdir(d)
                        os.chdir(d)
            self.dirs.append(os.path.relpath(dst_path, start=self.top_dir))
        return self.dirs[-count:]

    @staticmethod
    def generate_random_name(length, suffix=None, chinese=False):
        if chinese:
            result_str = "".join([s.decode("gb2312") for s in gb2312(length)])
        else:
            letters = string.ascii_letters
            result_str = "".join(random.choice(letters) for i in range(length))
        if suffix is not None:
            result_str += f".{suffix}"
        return result_str

    @staticmethod
    def generate_random_length_name(max_length, suffix=None, chinese=False):
        # Shrink the max_length since it has a suffix
        # Use max_length - 9 as the minimum length to reduce name conflict.
        len = randint((max_length - 9) // 2, max_length - 9)
        return Distributor.generate_random_name(len, suffix, chinese)

    def _put_single_file(
        self,
        parent_dir,
        file_size: Size,
        specified_name=None,
        letters=False,
        chinese=False,
        name_len=32,
    ):
        if specified_name is None:
            name = Distributor.generate_random_length_name(
                name_len, suffix="regular", chinese=chinese
            )
        else:
            name = specified_name

        this_path = os.path.join(parent_dir, name)

        with pushd(parent_dir):
            if chinese:
                fd = os.open(name.encode("gb2312"), os.O_CREAT | os.O_RDWR)
            else:
                fd = os.open(name.encode("ascii"), os.O_CREAT | os.O_RDWR)

            if file_size.B != 0:
                left = file_size.B
                logging.debug("Putting file %s", this_path)
                while left:
                    length = Size(1, Unit.MB).B if Size(1, Unit.MB).B < left else left
                    if not letters:
                        left -= os.write(fd, os.urandom(length))
                    else:
                        picked_list = "".join(
                            random.choices(string.ascii_lowercase[1:4], k=length)
                        )
                        left -= os.write(fd, picked_list.encode())

            os.close(fd)

        self.files.append(self._relative_path_to_top(this_path))
        return name, this_path

    def put_single_file(self, file_size: Size, pos=None, name=None):
        self._put_single_file(
            self._random_pos_dir() if pos is None else pos,
            file_size,
            letters=True,
            specified_name=name,
        )
        return self.files[-1]

    def put_single_file_with_xattr(self, file_size: Size, kv, pos=None, name=None):
        self._put_single_file(
            self._random_pos_dir() if pos is None else pos,
            file_size,
            letters=True,
            specified_name=name,
        )
        p = os.path.join(self.top_dir, self.files[-1])
        xattr.setxattr(p, kv[0].encode(), kv[1].encode())

    def put_multiple_files(self, count: int, max_size: Size):
        for i in range(0, count):
            cur_size = Size.from_B(randint(0, max_size.B))
            self._put_single_file(self._random_pos_dir(), cur_size)
        return self.files[-count:]

    def put_multiple_chinese_files(self, count: int, max_size: Size):
        for i in range(0, count):
            cur_size = Size.from_B(randint(0, max_size.B))
            self._put_single_file(self._random_pos_dir(), cur_size, chinese=True)
        return self.files[-count:]

    def put_multiple_empty_files(self, count):
        for i in range(0, count):
            self._put_single_file(self._random_pos_dir(), Size(0, Unit.Byte))
        return self.files[-count:]


if __name__ == "__main__":
    top_dir = "/mnt/gen_tree"
    if os.path.exists(top_dir):
        shutil.rmtree(top_dir)
    try:
        os.makedirs(top_dir, exist_ok=True)
    except FileExistsError:
        pass

    dist = Distributor(top_dir, 2, 5)
    dist.generate_tree()
    print(dist._random_pos_dir())
    dist.put_hardlinks(10)
    Distributor.generate_random_name(2000, suffix="sym")
    dist._put_single_file(top_dir, Size(100, Unit.MB))
    dist.put_multiple_files(1000, Size(4, Unit.KB))
