/*
 * driver/base/clockevent.c - Populate sysfs with cpu clockevent information
 *
 * Written by: xuyun.xy@alibaba-inc.com
 *
 *
 */
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/cpu.h>
#include <linux/module.h>
#include <linux/clockchips.h>
#include <linux/tick.h>

extern void tick_broadcast_hrtimer_control(void *arg);
struct cpumask *tick_get_broadcast_mask(void);

static ssize_t current_clockevent_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	unsigned int cpu = dev->id;

	if (cpumask_test_cpu(cpu, tick_get_broadcast_mask()))
		return sprintf(buf, "global");
	return sprintf(buf, "local");
}

static ssize_t current_clockevent_store(struct device *dev,
		struct device_attribute *attr, const char *buf, size_t n)
{
	unsigned int cpu = dev->id;
	enum tick_broadcast_mode op;

	if (!strncmp(buf, "local", 5))
		op = TICK_BROADCAST_OFF;
	else if (!strncmp(buf, "global", 6))
		op = TICK_BROADCAST_ON;
	else
		return -ENODEV;

	smp_call_function_single(cpu, tick_broadcast_hrtimer_control, &op, 1);
	return n;
}

static DEVICE_ATTR_RW(current_clockevent);

static ssize_t available_clockevent_show(struct device *dev,
		struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%s", "local global");
}

static DEVICE_ATTR_RO(available_clockevent);

static struct attribute *default_attrs[] = {
	&dev_attr_current_clockevent.attr,
	&dev_attr_available_clockevent.attr,
	NULL
};

static struct attribute_group clockevent_attr_group = {
	.attrs = default_attrs,
	.name = "clockevent"
};

/* Add/Remove cpu_clockevent interface for CPU device */
static int clockevent_add_dev(unsigned int cpu)
{
	struct device *dev = get_cpu_device(cpu);

	return sysfs_create_group(&dev->kobj, &clockevent_attr_group);
}

static int clockevent_remove_dev(unsigned int cpu)
{
	struct device *dev = get_cpu_device(cpu);

	sysfs_remove_group(&dev->kobj, &clockevent_attr_group);
	return 0;
}

static int clockevent_sysfs_init(void)
{
	return cpuhp_setup_state(CPUHP_CLOCKEVENT_ONLINE,
				 "base/clockevent:online",
				 clockevent_add_dev,
				 clockevent_remove_dev);
}

device_initcall(clockevent_sysfs_init);
