/*
 * Copyright (c) 2020 Yubico AB. All rights reserved.
 * Use of this source code is governed by a BSD-style
 * license that can be found in the LICENSE file.
 */

#include <assert.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "../openbsd-compat/openbsd-compat.h"
#include "mutator_aux.h"

struct param {
	int seed;
	int dev;
	struct blob wiredata;
};

/*
 * Sample netlink messages. These are unlikely to get the harness very far in
 * terms of coverage, but serve to give libFuzzer a sense of the underlying
 * structure.
 */
static const uint8_t sample_netlink_wiredata[] = {
	0xd8, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x9d, 0x2e, 0x00, 0x00,
	0x01, 0x02, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x6e, 0x66, 0x63, 0x00, 0x06, 0x00, 0x01, 0x00,
	0x1e, 0x00, 0x00, 0x00, 0x08, 0x00, 0x03, 0x00,
	0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x05, 0x00,
	0x1f, 0x00, 0x00, 0x00, 0x80, 0x01, 0x06, 0x00,
	0x14, 0x00, 0x01, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0e, 0x00, 0x00, 0x00, 0x14, 0x00, 0x02, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x03, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x03, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x04, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x06, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x05, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x07, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x06, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x07, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x05, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x08, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0c, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x09, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x0f, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x0a, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x0b, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x13, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x0c, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x15, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x0d, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x11, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x0e, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x12, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x0f, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x1a, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0c, 0x00, 0x00, 0x00, 0x14, 0x00, 0x10, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x1b, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x11, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x1c, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x14, 0x00, 0x12, 0x00,
	0x08, 0x00, 0x01, 0x00, 0x1d, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x02, 0x00, 0x0a, 0x00, 0x00, 0x00,
	0x14, 0x00, 0x13, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x1e, 0x00, 0x00, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x0a, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x07, 0x00,
	0x18, 0x00, 0x01, 0x00, 0x08, 0x00, 0x02, 0x00,
	0x05, 0x00, 0x00, 0x00, 0x0b, 0x00, 0x01, 0x00,
	0x65, 0x76, 0x65, 0x6e, 0x74, 0x73, 0x00, 0x00,
	0x24, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01,
	0x00, 0x00, 0x00, 0x00, 0x9d, 0x2e, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x00,
	0x1e, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x24, 0x00, 0x00, 0x00,
	0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
	0x9d, 0x2e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x24, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x05, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x1c, 0x00, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x09, 0x01, 0x00, 0x00, 0x08, 0x00, 0x01, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
	0x1e, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x9d, 0x2e, 0x00, 0x00, 0x08, 0x01, 0x00, 0x00,
	0x08, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x08, 0x00, 0x03, 0x00, 0x10, 0x00, 0x00, 0x00,
	0x06, 0x00, 0x05, 0x00, 0x44, 0x00, 0x00, 0x00,
	0x05, 0x00, 0x06, 0x00, 0x20, 0x00, 0x00, 0x00,
	0x0b, 0x00, 0x07, 0x00, 0x27, 0x00, 0x00, 0x00,
	0x93, 0xb9, 0x25, 0x00
};

struct param *
unpack(const uint8_t *ptr, size_t len)
{
	cbor_item_t *item = NULL, **v;
	struct cbor_load_result cbor;
	struct param *p;
	int ok = -1;

	if ((p = calloc(1, sizeof(*p))) == NULL ||
	    (item = cbor_load(ptr, len, &cbor)) == NULL ||
	    cbor.read != len ||
	    cbor_isa_array(item) == false ||
	    cbor_array_is_definite(item) == false ||
	    cbor_array_size(item) != 3 ||
	    (v = cbor_array_handle(item)) == NULL)
		goto fail;

	if (unpack_int(v[0], &p->seed) < 0 ||
	    unpack_int(v[1], &p->dev) < 0 ||
	    unpack_blob(v[2], &p->wiredata) < 0)
		goto fail;

	ok = 0;
fail:
	if (ok < 0) {
		free(p);
		p = NULL;
	}

	if (item)
		cbor_decref(&item);

	return p;
}

size_t
pack(uint8_t *ptr, size_t len, const struct param *p)
{
	cbor_item_t *argv[3], *array = NULL;
	size_t cbor_alloc_len, cbor_len = 0;
	unsigned char *cbor = NULL;

	memset(argv, 0, sizeof(argv));

	if ((array = cbor_new_definite_array(3)) == NULL ||
	    (argv[0] = pack_int(p->seed)) == NULL ||
	    (argv[1] = pack_int(p->dev)) == NULL ||
	    (argv[2] = pack_blob(&p->wiredata)) == NULL)
		goto fail;

	for (size_t i = 0; i < 3; i++)
		if (cbor_array_push(array, argv[i]) == false)
			goto fail;

	if ((cbor_len = cbor_serialize_alloc(array, &cbor,
	    &cbor_alloc_len)) > len) {
		cbor_len = 0;
		goto fail;
	}

	memcpy(ptr, cbor, cbor_len);
fail:
	for (size_t i = 0; i < 3; i++)
		if (argv[i])
			cbor_decref(&argv[i]);

	if (array)
		cbor_decref(&array);

	free(cbor);

	return cbor_len;
}

size_t
pack_dummy(uint8_t *ptr, size_t len)
{
	struct param dummy;
	uint8_t	blob[4096];
	size_t blob_len;

	memset(&dummy, 0, sizeof(dummy));

	dummy.wiredata.len = sizeof(sample_netlink_wiredata);
	memcpy(&dummy.wiredata.body, &sample_netlink_wiredata,
	    dummy.wiredata.len);

	assert((blob_len = pack(blob, sizeof(blob), &dummy)) != 0);

	if (blob_len > len) {
		memcpy(ptr, blob, len);
		return len;
	}

	memcpy(ptr, blob, blob_len);

	return blob_len;
}

void
test(const struct param *p)
{
	fido_nl_t *nl;
	uint32_t target;

	prng_init((unsigned int)p->seed);
	fido_init(FIDO_DEBUG);
	fido_set_log_handler(consume_str);

	set_netlink_io_functions(fd_read, fd_write);
	set_wire_data(p->wiredata.body, p->wiredata.len);

	if ((nl = fido_nl_new()) == NULL)
		return;

	consume(&nl->fd, sizeof(nl->fd));
	consume(&nl->nfc_type, sizeof(nl->nfc_type));
	consume(&nl->nfc_mcastgrp, sizeof(nl->nfc_mcastgrp));
	consume(&nl->saddr, sizeof(nl->saddr));

	fido_nl_power_nfc(nl, (uint32_t)p->dev);

	if (fido_nl_get_nfc_target(nl, (uint32_t)p->dev, &target) == 0)
		consume(&target, sizeof(target));

	fido_nl_free(&nl);
}

void
mutate(struct param *p, unsigned int seed, unsigned int flags) NO_MSAN
{
	if (flags & MUTATE_SEED)
		p->seed = (int)seed;

	if (flags & MUTATE_PARAM)
		mutate_int(&p->dev);

	if (flags & MUTATE_WIREDATA)
		mutate_blob(&p->wiredata);
}
