"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererContextΩ = void 0;
const config_1 = require("../config");
const fs_1 = require("../fs");
const run_1 = require("../run");
const suite_1 = require("../suite");
const test_1 = require("../test");
const utils_1 = require("../utils");
const context_summary_1 = require("./context-summary");
class BettererContextΩ {
    constructor(_globals, _watcher) {
        this._globals = _globals;
        this._watcher = _watcher;
        this._isDestroyed = false;
        this._suiteSummaries = [];
        this.config = this._globals.config;
        this._runWorkerPool = new run_1.BettererRunWorkerPoolΩ(this.config.workers);
        this._resultsFile = this._globals.resultsFile;
        this._reporter = this.config.reporter;
        this._versionControl = this._globals.versionControl;
        this._started = this._start();
    }
    get isDestroyed() {
        return this._isDestroyed;
    }
    async options(optionsOverride) {
        // Wait for any pending run to finish, and any existing reporter to render:
        await this._started.end();
        // Override the config:
        (0, config_1.overrideConfig)(this.config, optionsOverride);
        // Start everything again, and trigger a new reporter:
        this._started = this._start();
        // eslint-disable-next-line @typescript-eslint/no-misused-promises
        process.on('SIGTERM', () => this.stop());
    }
    async run(specifiedFilePaths, isRunOnce = false) {
        try {
            await this._resultsFile.sync();
            await this._versionControl.sync();
            const { cwd, includes, excludes } = this.config;
            const globalResolver = new fs_1.BettererFileResolverΩ(cwd, this._versionControl);
            globalResolver.include(...includes);
            globalResolver.exclude(...excludes);
            const hasSpecifiedFiles = specifiedFilePaths.length > 0;
            const hasGlobalIncludesExcludes = includes.length || excludes.length;
            let filePaths;
            if (hasSpecifiedFiles && hasGlobalIncludesExcludes) {
                // Validate specified files based on global `includes`/`excludes and gitignore rules:
                filePaths = await globalResolver.validate(specifiedFilePaths);
            }
            else if (hasSpecifiedFiles) {
                // Validate specified files based on gitignore rules:
                filePaths = await globalResolver.validate(specifiedFilePaths);
            }
            else if (hasGlobalIncludesExcludes) {
                // Resolve files based on global `includes`/`excludes and gitignore rules:
                filePaths = await globalResolver.files();
            }
            else {
                // When `filePaths` is `[]` the test will use its specific resolver:
                filePaths = [];
            }
            const testMeta = (0, test_1.loadTestMeta)(this.config);
            const testNames = Object.keys(testMeta);
            const workerRunConfig = (0, run_1.createWorkerRunConfig)(this.config);
            const runs = await Promise.all(testNames.map(async (testName) => {
                return await run_1.BettererRunΩ.create(this._runWorkerPool, testName, workerRunConfig, filePaths, this._versionControl);
            }));
            const suite = new suite_1.BettererSuiteΩ(this.config, this._resultsFile, filePaths, runs);
            const suiteSummary = await suite.run(isRunOnce);
            this._suiteSummaries = [...this._suiteSummaries, suiteSummary];
        }
        catch (error) {
            await this._started.error(error);
            throw error;
        }
    }
    async runOnce() {
        try {
            await this.run([], true);
            const summary = await this.stop();
            return summary;
        }
        finally {
            await this._destroy();
        }
    }
    async stop() {
        try {
            const contextSummary = await this._started.end();
            return contextSummary.lastSuite;
        }
        finally {
            await this._destroy();
        }
    }
    async _destroy() {
        this._isDestroyed = true;
        if (this._watcher) {
            await this._watcher.close();
        }
        await this._versionControl.destroy();
        await this._runWorkerPool.destroy();
    }
    _start() {
        // Update `this._reporter` here because `this.options()` may have been called:
        this._reporter = this.config.reporter;
        const contextLifecycle = (0, utils_1.defer)();
        // Don't await here! A custom reporter could be awaiting
        // the lifecycle promise which is unresolved right now!
        const reportContextStart = this._reporter.contextStart(this, contextLifecycle.promise);
        return {
            end: async () => {
                const contextSummary = new context_summary_1.BettererContextSummaryΩ(this.config, this._suiteSummaries);
                contextLifecycle.resolve(contextSummary);
                await reportContextStart;
                await this._reporter.contextEnd(contextSummary);
                const suiteSummaryΩ = contextSummary.lastSuite;
                if (suiteSummaryΩ && !this.config.ci) {
                    await this._resultsFile.write(suiteSummaryΩ, this.config.precommit);
                }
                await this._versionControl.writeCache();
                return contextSummary;
            },
            error: async (error) => {
                contextLifecycle.reject(error);
                await reportContextStart;
                await this._reporter.contextError(this, error);
            }
        };
    }
}
exports.BettererContextΩ = BettererContextΩ;
//# sourceMappingURL=context.js.map