from __future__ import with_statement
from __future__ import division
from __future__ import absolute_import
import concurrent.futures
import functools
from httplib import HTTPConnection
import json
import logging
import math
import multiprocessing
import requests
from requests.auth import HTTPBasicAuth
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
import sys
import time
import threading
import traceback
import os
from urllib import quote
from urlparse import urlparse, urljoin
import xml.etree.ElementTree as ET

from redhat_support_lib.infrastructure.aws_auth import AWSRequestsAuth
import redhat_support_lib.utils.confighelper as confighelper
from redhat_support_lib.infrastructure.errors import S3UploadFailedError, DontWaitForCompletion, MetadataException
from redhat_support_lib.infrastructure.file_operations import open_file_chunk_reader
from redhat_support_lib.utils.timerhelper import TimerThread


__author__ = 'Pranita Ghole pghole@redhat.com'

logger = logging.getLogger("redhat_support_lib.infrastructure.s3_uploader")


config = confighelper.get_config_helper()


class ProgressPercentage(object):
    def __init__(self, filename):
        self._filename = filename
        self._size = float(os.path.getsize(filename))
        self._seen_so_far = 0
        self._lock = threading.Lock()

    def __call__(self, bytes_amount):
        # To simplify we'll assume this is hooked up
        # to a single filename.
        with self._lock:
            self._seen_so_far += bytes_amount
            percentage = (self._seen_so_far / self._size) * 100
            sys.stdout.write(
                "\r %.2f%%" % percentage)
            sys.stdout.flush()


def refresh_cred(method):
    """
    If any AWS requests fails due to credentials expired (inspite of the credentials getting refreshed by the thread),
    refresh the credentials and perform that request once more.
    :param method: The function which sends request to aws
    :type method: function to be executed.
    :return: Response of the AWS request.
    :rtype: Response object.
    """

    def run_method(*args, **kw):
        try:
            result = method(*args, **kw)
        except requests.exceptions.HTTPError as err:
            if err.response.status_code == 400 and ET.fromstring(err.response.content).find(
                    "Code").text == 'ExpiredToken':
                logger.warning(err.response.content)
                logger.debug("Refreshing credentials after they expired.")
                for arg in args:
                    if isinstance(arg, AWSUploader):
                        arg.refresh_upload_credentials()
                        break
                result = method(*args, **kw)
            else:
                logger.warning(err.response.content)
                raise
        except requests.exceptions.ConnectionError, cerr:
            raise Exception("Connection Error: {}".format(cerr))
        except requests.exceptions.Timeout, terr:
            raise Exception("Timeout Error: {}".format(terr))
        except requests.exceptions.RequestException, rerr:
            raise Exception(rerr)
        except Exception, e:
            raise
        return result

    return run_method


def check_for_proxy():
    '''
    Check if redhat-support-tool has a proxy configured. If yes, return that in dict format for the requests library.

    :return:Proxy dictionary
    '''
    proxy_dict = {}
    config = confighelper.get_config_helper()
    if config.proxy_url:
        if config.proxy_user and config.proxy_pass:
            http_proxy_url = urlparse(config.proxy_url)
            proxyurl = http_proxy_url.scheme + "://" + quote(config.proxy_user) + ":" + quote(config.proxy_pass) + "@" +\
                       http_proxy_url.netloc
            proxy_dict['https'] = proxyurl
            proxy_dict['http'] = proxyurl
        else:
            proxy_dict['https'] = config.proxy_url
            proxy_dict['http'] = config.proxy_url
    return proxy_dict if proxy_dict else None


def check_ssl_params():
    '''
    Checks the SSL verify param.

    :return: False or path of the cert.

    '''
    verify = True
    if config.no_verify_ssl:
        verify = False
    else:
        if os.access('/etc/pki/tls/certs/ca-bundle.crt', os.R_OK):
            verify = '/etc/pki/tls/certs/ca-bundle.crt'
        if config.ssl_ca and os.access(config.ssl_ca, os.R_OK):
            verify = config.ssl_ca
    return verify


class AWSUploader:
    """
    The AWSUploader class.

    """

    def __init__(self, case_number=None, uuid=None, isPrivate=False, description=None, auth=None,
                 part_size=8*1024*1024, s3_aws_host='s3-accelerate.amazonaws.com', filepath=None, use_threads=True,
                 refresh_frequency=600, max_workers=10, fileChunk=None, endpoint_url='https://redhat.com',
                 max_status_retries=10, multipart_threshold=8*1024*1024, aws_verify=True, s3_failure_retries=5):

        config = confighelper.get_config_helper()
        self.part_size = int(config.s3_part_size)*1024*1024 if config.s3_part_size else part_size
        self.s3_aws_host = config.s3_aws_host if config.s3_aws_host else s3_aws_host
        self.endpoint_url = config.s3_endpoint_url if config.s3_endpoint_url else endpoint_url
        self.use_threads = config.s3_use_threads if config.s3_use_threads is not None else use_threads
        self.max_workers = int(config.s3_max_workers) if config.s3_max_workers else min(max_workers, multiprocessing.cpu_count() + 4)
        self.max_status_retries = int(config.s3_max_status_retries) if config.s3_max_status_retries else max_status_retries
        self.multipart_threshold = int(config.s3_multipart_threshold)*1024*1024 if config.s3_multipart_threshold else multipart_threshold
        self.refresh_frequency = int(config.s3_refresh_frequency) if config.s3_refresh_frequency else refresh_frequency
        self.s3_failure_retries = int(config.s3_failure_retries) if config.s3_failure_retries else s3_failure_retries
        self.proxies = check_for_proxy()
        self.verify = check_ssl_params()
        self.aws_verify = config.s3_aws_verify if config.s3_aws_verify is not None else aws_verify
        self.case_number = case_number
        self.uuid = uuid
        self.isPrivate = isPrivate
        self.description = description
        self.auth = auth
        self.filepath = filepath
        self.bytes_transferred = 0
        self.fileChunk = fileChunk
        self.attachment_host = config.url
        self.hydra_proxy_aws_host = None
        self.callback = ProgressPercentage(self.filepath)
        self.bucket = None
        self.aws_host = None
        self.key = None
        self.completed_metadata = None
        self.all_uuids = []
        self.encryption_enabled = False
        self.enc_type = None
        self.encryp_auth = None

    def send_hydra_request(self, method, uri, **kwargs):
        """
        The function to send requests to Hydra.

        :param method: GET, PUT, POST etc.
        :param uri: The relative url for request.
        :param kwargs: The keyword arguments like data=fileobj etc.
        :return: response obj.

        """
        url = urljoin(self.attachment_host, uri)
        headers = (config.userAgent)
        if not config.auth:
            kwargs.update(auth=HTTPBasicAuth(config.username, config.password), verify=self.verify,
                          proxies=self.proxies, headers=config.userAgent)
        else:
            headers.update({"Authorization": "Bearer %s" % config.auth.get_access_token()})
            kwargs.update(verify=self.verify,
                          proxies=self.proxies, headers=headers)
        with requests.Session() as s:
            HTTPConnection.debuglevel = int(config.http_debug)
            response = s.request(method, url, **kwargs)
        if response.status_code == 400 and '/credentials/refresh' in url:
            logger.debug(response.json())
            return
        response.raise_for_status()
        if response.status_code == 200:
            return response
        else:
            raise Exception("The Hydra request for url {} returned {}".format(url, response.status_code))

    @refresh_cred
    def send_aws_request(self, method, uri="", **kwargs):
        """

        Send the request to AWS with retries for some status codes. The backoff of 1 second means
        the successive sleeps will be 0.5, 1, 2, 4, 8, 16, 32, 64, 128, 256.

        Retry logic has also been implemented.

        :param method: GET, PUT, POST, DELETE etc.
        :param uri: The url for request. Pls. note we are sending the query param in the url, rather than the "params"
        because the params are quoted using quote_plus, which might cause some issue while calculating the S4 signature
        :param kwargs: The keyword arguments like data=fileobj etc.
        :return: response object

        """
        if not kwargs.get('auth'):
            kwargs.update(auth=self.auth)
        if config.s3_aws_verify is not None:
            kwargs.update(verify=self.aws_verify)
        kwargs.update(proxies=self.proxies)
        base_url = urljoin(self.hydra_proxy_aws_host, self.key)
        url = base_url + uri
        retries = Retry(total=self.s3_failure_retries, backoff_factor=1, status_forcelist=[500, 502, 503, 504, 501, 429])
        with requests.Session() as s:
            s.mount('https://', HTTPAdapter(max_retries=retries))
            s.mount('http://', HTTPAdapter(max_retries=retries))
            HTTPConnection.debuglevel = int(config.http_debug)
            response = s.request(method, url,  allow_redirects=True, headers=config.userAgent, **kwargs)
        response.raise_for_status()
        return response

    def form_sigv4_auth(self, access_key, secret_key, aws_session_token, region, ss_algorithm=None,
                        encryption_key=None, encryption_key_md5=None):
        """
        This function returns the AWS auth object. This can be directly passed to the "auth" in the requests.

        :param access_key: The AWS access key
        :param secret_key: The AWS secret access key
        :param aws_session_token: The AWS token.
        :param region: The region of the s3. for eg- 'us-east-1'
        :return: The AWS auth object

        """
        logger.debug("Forming new signature v4 auth object to sign the requests.")
        auth = AWSRequestsAuth(aws_access_key=access_key,
                               aws_secret_access_key=secret_key,
                               aws_token=aws_session_token,
                               aws_host=self.aws_host,
                               aws_region=region,
                               aws_service='s3',
                               ss_algorithm=ss_algorithm,
                               encryption_key=encryption_key,
                               encryption_key_md5=encryption_key_md5
                               )
        return auth

    def update_vars(self, resp):
        """
        This function will form update the class variables and form the 'aws_host' and 'hydra_proxy_aws_host' runtime.
        :param resp: The json response from the fetch_upload_credentials()
        :return:True

        """
        self.key = resp['key']
        self.uuid = resp['attachmentId']
        self.bucket = resp['bucketName']
        self.aws_host = ".".join([self.bucket, self.s3_aws_host])
        split_hydra_proxy = urlparse(self.endpoint_url)
        self.hydra_proxy_aws_host = '%s://%s.%s' % (split_hydra_proxy.scheme, self.bucket, split_hydra_proxy.netloc)

        logger.debug("key: {}, uuid: {}, bucket: {}, Host used in AWS signature calculation: {}, "
                     "Host used for upload request: {}".format(
                      self.key, self.uuid, self.bucket, self.aws_host, self.hydra_proxy_aws_host))

        return True

    def fetch_upload_credentials(self, filename=None):
        """
        This fetches the credentials needed for uploading to s3 from Hydra.

        :param filename: The path of the file to be uploaded. Optional

        :return: None.

        """
        logger.debug("Fetching the AWS credentials for upload from Hydra")

        uri = "/support/v1/cases/{case_number}/attachments/upload/credentials".format(case_number=self.case_number)

        filename = self.filepath if filename is None else filename

        data = {
            "fileName": os.path.basename(filename),
            "isPrivate": self.isPrivate,
            "description": self.description
        }

        res = self.send_hydra_request(method="POST", uri=uri, data=json.dumps(data))

        upload_cred = res.json()

        if self.update_vars(upload_cred):
            self.auth = self.form_sigv4_auth(access_key=upload_cred.get('accessKey'),
                                             secret_key=upload_cred.get('secretKey'),
                                             aws_session_token=upload_cred.get('sessionToken'),
                                             region=upload_cred.get('region')
                                             )
            if upload_cred.get('serverSideEncryption'):
                self.encryption_enabled = True
                self.encryp_auth = self.form_sigv4_auth(access_key=upload_cred.get('accessKey'),
                                                        secret_key=upload_cred.get('secretKey'),
                                                        aws_session_token=upload_cred.get('sessionToken'),
                                                        region=upload_cred.get('region'),
                                                        ss_algorithm=upload_cred.get('serverSideEncryption'),
                                                        encryption_key=upload_cred.get('serverSideEncryptionKmsKey'),
                                                        encryption_key_md5=upload_cred.get('serverSideEncryptionKeyMD5')
                                                        )
                if upload_cred.get('serverSideEncryption') == 'AES256' and upload_cred.get('serverSideEncryptionKmsKey'):
                    self.enc_type = 'SSE-C'

    def refresh_upload_credentials(self):
        """
        This fetches the refreshed credentials for the specific case_number and uuid upload. Since the credentials are
        expired every 15 min., we need to call this function to refresh the credentials, form the new AWS auth object &
        update the self.auth to be passed in the AWS requests.

        :return: None

        """
        logger.debug("Refreshing the AWS credentials")

        uri = "/support/v1/cases/{case_number}/attachments/{attachment_id}/upload/credentials/refresh".format(
            case_number=self.case_number,
            attachment_id=self.uuid
        )

        res = self.send_hydra_request(method='GET', uri=uri)

        if res:
            new_cred = res.json()
            with threading.Lock():
                self.auth = self.form_sigv4_auth(access_key=new_cred.get('accessKey'),
                                                 secret_key=new_cred.get('secretKey'),
                                                 aws_session_token=new_cred.get('sessionToken'),
                                                 region=new_cred.get('region')
                                                 )

                if new_cred.get('serverSideEncryption'):
                    self.encryption_enabled = True
                    self.encryp_auth = self.form_sigv4_auth(access_key=new_cred.get('accessKey'),
                                                            secret_key=new_cred.get('secretKey'),
                                                            aws_session_token=new_cred.get('sessionToken'),
                                                            region=new_cred.get('region'),
                                                            ss_algorithm=new_cred.get('serverSideEncryption'),
                                                            encryption_key=new_cred.get('serverSideEncryptionKmsKey'),
                                                            encryption_key_md5=new_cred.get('serverSideEncryptionKeyMD5')
                                                            )
                    if new_cred.get('serverSideEncryption') == 'AES256' and new_cred.get(
                            'serverSideEncryptionKmsKey'):
                        self.enc_type = 'SSE-C'
                logger.debug("The credentials are refreshed.")

    def create_multipart_upload(self):
        '''

        This initiates a multipart upload and returns as uploadid. AWS sends HTTP 200 response for success.
        For any server side encryption send encryp_auth.

        :return: UploadID, type - str

        '''

        auth = self.encryp_auth if self.encryption_enabled else self.auth

        resp = self.send_aws_request(method='POST',
                                     uri='?uploads',
                                     auth=auth
                                     )
        if resp.status_code == 200:
            root = ET.fromstring(resp.content)
            uploadid = [i.text for i in list(root) if i.tag.endswith('UploadId')][0]
            logger.debug("The multipart upload has been initiated with Uploadid {}".format(uploadid))
            return uploadid

    def upload_parts(self, uploadid):
        """
        Divides the file in parts and uploads. This has support for both with and without using threads.

        :param uploadid: The uploadId
        :return: parts. type - list of dicts. For eg- [{"PartNumber": 1, "ETag": "b54357faf0632cce46e942fa68356b38"},..]

        """
        parts = []
        num_parts = int(math.ceil(os.path.getsize(self.filepath) / float(self.part_size)))
        if not self.use_threads:
            logger.debug("Proceeding to upload the parts without threads")
            for i in range(1, num_parts + 1):
                part = self._upload_one_part(upload_id=uploadid, part_number=i)
                parts.append(part)
        else:
            logger.debug("Proceeding to upload the parts using threading")
            with concurrent.futures.ThreadPoolExecutor(max_workers=self.max_workers) as executor:
                upload_partial = functools.partial(self._upload_one_part, uploadid)
                for part in executor.map(upload_partial, range(1, num_parts + 1)):
                    parts.append(part)

        # checking if all parts got successfully uploaded.
        successful_uploads = [i for i in parts if i is not None]
        if len(parts) == len(successful_uploads):
            sorted_parts = sorted(parts, key=lambda k: k['PartNumber'])
            return sorted_parts
        else:
            logger.warning("All the parts were not uploaded successfully. Hence, aborting the upload")
            raise Exception("All the parts were not uploaded successfully. Hence, aborting the upload")

    def _upload_one_part(self, upload_id, part_number):
        """

        :param upload_id (str): The uploadID
        :param part_number (int): The file to be uploaded is divided in parts. This specifies the number associated with the part
        AWS sends a HTTP 200 response on success.
        For SSE-C, send the Encryption auth else send the auth without encryption headers.

        :return: part type (dict): For eg. - {"PartNumber": 1, "ETag": "b54357faf0632cce46e942fa68356b38"}
        """
        open_chunk_reader = open_file_chunk_reader

        auth = self.encryp_auth if self.encryption_enabled and self.enc_type == 'SSE-C' else self.auth

        with open_chunk_reader(self.filepath, start_byte=self.part_size * (part_number - 1), size=self.part_size,
                               callback=self.callback) as fileobj:
            part_uploads_resp = self.send_aws_request(method='PUT',
                                                      uri='?partNumber=' + str(part_number) + '&uploadId=' + upload_id,
                                                      data=fileobj, auth=auth)
            if part_uploads_resp.status_code == 200:
                return {"PartNumber": part_number, "ETag": part_uploads_resp.headers['ETag']}
            else:
                logger.warning("There is some error uploading this part - {}".format(part_uploads_resp.content))
                raise Exception("Unexpected error while uploading a part to s3. The API returned "
                                "{}".format(part_uploads_resp.status_code))

    def abort_all(self, uploadid):
        """
        In case of any exception while uploading the parts for multipart uploads, the uploads should be aborted by calling
        the abort api. AWS sends HTTP 204 response for successful aborts.

        If there is any exception in the abort operation, we are ignoring that as we don't want to retry for abort
        and let it raise the Exception that follows.

        :param uploadid: The uploadid

        :return:

        """
        try:
            res = self.send_aws_request(method='DELETE',
                                        uri='?uploadId=' + uploadid)

            if res.status_code == 204:
                logger.debug("Upload aborted successfully.")
            else:
                raise Exception("The abort API returned response {}".format(res.status_code))
        except Exception, e:
            logger.warning("There is a exception while aborting the file. {}".format(e))

    def complete_upload_request_body(self, parts):
        """
        :param parts: The list of dictionary of the parts uploaded to AWS.
        eg - [{"PartNumber": 1, "ETag": "b54357faf0632cce46e942fa68356b38"}, ...]

        :return: The xml to be sent to the complete multipart upload request

        <?xml version="1.0" encoding="UTF-8"?>
        <CompleteMultipartUpload xmlns="http://s3.amazonaws.com/doc/2006-03-01/">
            <Part>
                <ETag>string</ETag>
                <PartNumber>integer</PartNumber>
            </Part>
            ...
        </CompleteMultipartUpload>

        """
        logger.debug("Forming the request body to complete the upload")
        data = []
        croot = ET.Element("CompleteMultipartUpload")
        if parts:
            for i in parts:
                if i is not None:
                    part = ET.SubElement(croot, "Part")
                    ET.SubElement(part, "PartNumber").text = str(i["PartNumber"])
                    ET.SubElement(part, "ETag").text = i["ETag"]

            tree = ET.ElementTree(croot)

            data = ET.tostring(croot)

        return data

    def valid_response_body(self, resp):
        """
        It may be possible that for CompleteUpload requests, the status code returned is 200, however there is a  error,
        hence we need to check that there is no Error in the response xml.
        Refer https://docs.aws.amazon.com/AmazonS3/latest/API/API_CompleteMultipartUpload.html
        :param resp:
        :type resp:
        :return:
        :rtype:
        """
        if ET.fromstring(resp.content).tag.endswith('Error'):
            logger.warning(resp.content)
            return False
        return True

    def complete_multipart_upload_req(self, uploadid, data):
        """

        :param uploadid: The uploadid associated with the multipart upload
        :param data: The XML requests body
        :return: Response object if complete request is successful.

        """
        logger.debug("Making AWS request for completing the multipart upload")
        resp = self.send_aws_request(method='POST',
                                     uri='?uploadId=' + uploadid,
                                     data=data
                                     )
        if resp.status_code == 200 and self.valid_response_body(resp):
            return resp
        else:
            msg = "Unexpected exception while making the complete multipart upload to s3"
            logger.warning(msg)
            raise Exception(msg)

    def complete_multipart_upload(self, uploadid, parts):
        """

        :param uploadid: The Uploadid associated with the multipart upload.
        :param parts: The list of dictionaries of PartNumber and ETag
        :param parts: The list of dictionaries of PartNumber and ETag
        :return: True if the response is 200
        """
        request_body = self.complete_upload_request_body(parts)
        response = self.complete_multipart_upload_req(uploadid=uploadid, data=request_body)
        if response.status_code == 200:
            logger.debug("The complete multipart request was successful")
            return True

    def check_attachment_status(self, filename=None, uuid=None):
        """

        :param filename: The name of the file for which status has to be checked. Various statuses are
           # -upload_pending, -metadata_creation_pending, -metadata_creation_failed -completed -deleted
        :return: True if the metadata is created successfully for that file.
        :rtype:
        """

        filename = self.filepath if filename is None else filename
        uuid = self.uuid if uuid is None else uuid
        minDelay = 1
        computedDelay = 10 * (1 - 1 / os.stat(filename).st_size)
        polling_time = minDelay if computedDelay < minDelay else computedDelay
        retry = 0
        flag = 1

        uri = '/support/v1/cases/{case_number}/attachments/{attachment_id}/status'.format(
            case_number=self.case_number,
            attachment_id=uuid
        )

        try:
            while not self.completed_metadata:
                res = self.send_hydra_request(method='GET', uri=uri)
                json_res = res.json()
                retry += 1
                logger.debug("The status of the attachment is {}".format(json_res['status']))
                if json_res['status'] == 'COMPLETED':
                    self.completed_metadata = True
                    return True
                if json_res['status'] == 'METADATA_CREATION_FAILED':
                    return False
                if json_res['status'] == 'DELETED':
                    raise Exception("The uuid {} has returned DELETED status while verification".format(uuid))
                valid_status = json_res['status'] == 'UPLOAD_PENDING' or json_res['status'] == 'METADATA_CREATION_PENDING'
                if retry > self.max_status_retries and valid_status:
                    if flag == 1:
                        logger.warning(
                            "The file verification for case_number: {} and uuid: {} is taking longer than expected."
                            .format(self.case_number, self.uuid))
                        print "\nThe verification is taking longer than expected for the file {}. Press Ctrl+C to stop" \
                              " waiting and check your case attachments " \
                              "after a few minutes.".format(os.path.basename(filename))
                        flag += 1
                time.sleep(polling_time)
        except KeyboardInterrupt:
            logger.info("The file verification for case_number: {} and uuid: {} is not complete yet."
                        " The user has opted to not wait for it.".format(self.case_number, self.uuid))
            raise DontWaitForCompletion("")
        except Exception, e:
            logger.error(e)
            raise MetadataException(e)

    def check_split_attachment_status(self, filename=None, uuid=None):
        """

        :param filename: The name of the file for which status has to be checked.
        :return: True if the metadata is created successfully for that file.
        :rtype:
        """
        filename = self.filepath if filename is None else filename
        minDelay = 1
        computedDelay = 10 * (1 - 1 / os.stat(filename).st_size)
        polling_time = minDelay if computedDelay < minDelay else computedDelay
        retry = 0
        flag = 1

        uri = '/support/v1/cases/{case_number}/attachments/status'.format(case_number=self.case_number)

        try:
            while not self.completed_metadata:
                metadata_dict = {'COMPLETED': [], 'METADATA_CREATION_FAILED': [], 'DELETED': [], 'UPLOAD_PENDING': [],
                                 'METADATA_CREATION_PENDING': []}
                res = self.send_hydra_request(method='GET', uri=uri)
                json_res = res.json()
                retry += 1

                for uuid in self.all_uuids:
                    for res_dict in json_res:
                        if res_dict.get('attachmentId') == uuid:
                            metadata_dict[res_dict['status']].append(uuid)
                            break
                if len(metadata_dict['COMPLETED']) == len(self.all_uuids):
                    self.completed_metadata = True
                    return True
                elif len(metadata_dict['METADATA_CREATION_FAILED']) > 0:
                    return False
                elif len(metadata_dict['DELETED']) > 0:
                    raise Exception("Some uuids have returned DELETED status while "
                                    "verification - {}".format(metadata_dict['DELETED']))
                if retry > self.max_status_retries:
                    logger.debug(metadata_dict)
                    if flag == 1:
                        logger.warning(
                            "The file verification for case_number:{} and uuids: {} is taking longer "
                            "than expected.".format(self.case_number, ", ".join(self.all_uuids)))
                        print "\nThe verification is taking longer than expected for the split files of {}. "\
                              "Press Ctrl+C to stop waiting and check your case attachments after a few minutes."\
                              .format(os.path.basename(filename))
                        flag += 1
                time.sleep(polling_time)
        except KeyboardInterrupt:
            logger.info("The file verification for case_number:{} and uuids: {} is not complete yet."
                        " The user has opted to not wait for it.".format(self.case_number, ", ".join(self.all_uuids)))
            raise DontWaitForCompletion("")
        except Exception, e:
            logger.error(e)
            raise MetadataException(e)

    def multipart_upload_file(self):
        """

        :return:True if upload to s3 and metadata creation is successful.
        :rtype:Boolean or None
        """
        refresh_cred_timer = None
        upload_id = None
        try:
            self.fetch_upload_credentials()
            refresh_cred_timer = TimerThread(self.refresh_frequency,
                                             self.refresh_upload_credentials)
            upload_id = self.create_multipart_upload()
            parts = self.upload_parts(upload_id)
            complete_s3_upload = self.complete_multipart_upload(upload_id, parts)
            if complete_s3_upload:
                if refresh_cred_timer: refresh_cred_timer.stop()
                logger.info("File uploaded to s3. Verifying now...")
                return True
        except KeyboardInterrupt:
            logger.debug("The multipart upload as been interrupted by user")
            if upload_id: self.abort_all(upload_id)
            raise
        except Exception, e:
            logger.warning(e)
            if upload_id: self.abort_all(upload_id)
            raise
        finally:
            if refresh_cred_timer: refresh_cred_timer.stop()

    def single_put_operation(self):
        """
        This is a single PUT request to AWS to upload file. AWS sends HTTP 200 response for success.
        For all the server side encryption types, use the encrypted auth.

        :return: True if uploading the file was successful or raise exception.
        """

        self.fetch_upload_credentials()
        auth = self.encryp_auth if self.encryption_enabled else self.auth
        total_bytes = os.stat(self.filepath).st_size
        with open_file_chunk_reader(self.filepath, start_byte=0, size=total_bytes,
                                    callback=self.callback) as fileobj:
            response = self.send_aws_request(method='PUT', uri="", data=fileobj, auth=auth)
            if response.status_code == 200:
                logger.info("The file is uploaded to S3. Verifying...")
                return True
            raise Exception(response)

    def upload_split_files(self):

        """
        This function handles the split uploads in case of RHST i.e the user wants to split the file and upload it.
        Pls. note that the files are uploaded as separate with different uuids.

        chunk = {'num': 0, 'names': [], 'size': self._options.get('splitsize', self.max_split_size)}

        AWS sends HTTP 200 response for success. For all the server side encryption types, use the encrypted auth.

        :return: filechunk dict updated with the filechunk names
        :rtype: dict
        """

        num_parts = int(math.ceil(os.path.getsize(self.filepath) / float(self.fileChunk['size'])))
        for i in range(1, num_parts + 1):
            with open_file_chunk_reader(self.filepath, start_byte=int(self.fileChunk['size'] * (i - 1)),
                                        size=int(self.fileChunk['size']), callback=self.callback) as fileobj:
                chunkName = ("%s.%03d" % (os.path.basename(self.filepath),
                                          self.fileChunk['num']))
                self.fileChunk['names'].append(chunkName)
                self.fetch_upload_credentials(filename=chunkName)
                auth = self.encryp_auth if self.encryption_enabled else self.auth
                response = self.send_aws_request(method='PUT', uri="", data=fileobj, auth=auth)
                if response.status_code == 200:
                    self.all_uuids.append(self.uuid)
                    self.fileChunk['num'] += 1
                    logger.info("s3 upload {} was successful".format(chunkName))
                else:
                    raise Exception("There is some exception while uploading the split parts")
        return True

    def upload_file(self):
        """
        This uploads a file to the AWS using the s3 REST APIs. Checks if user has requested the for splitting the
        files in chunks. If not, then if the file is smaller than the multipart_threshold, uploads using a single
        PUT request (has a limit of 5GB) or else go for the multipart upload.

        :return: True for successful uploads.

        """
        logger.info("Uploading file using AWS S3 APIs")
        s3_config = {'s3_aws_host': self.s3_aws_host,
                     's3_endpoint_url': self.endpoint_url,
                     's3_multipart_threshold': self.multipart_threshold,
                     's3_part_size': self.part_size,
                     's3_use_threads': self.use_threads,
                     's3_max_workers': self.max_workers,
                     's3_refresh_frequency': self.refresh_frequency,
                     's3_failure_retries': self.s3_failure_retries
                     }
        logger.debug("S3 config - {}".format(s3_config))
        try:
            if self.fileChunk:
                s3_upload = self.upload_split_files()
            else:
                if os.stat(self.filepath).st_size > self.multipart_threshold:
                    s3_upload = self.multipart_upload_file()
                else:
                    s3_upload = self.single_put_operation()

            if not s3_upload:
                return s3_upload

            # Wait for 5 sec to hit the status api
            time.sleep(5)

            if not self.fileChunk:
                if not self.check_attachment_status():
                    raise MetadataException("Unexpected Exception Occurred while verifying the file"
                                            " - case_number: {}, uuid: {}".format(self.case_number, self.uuid))
                logger.info("File verified successfully.")
                return True

            if self.fileChunk:
                if not self.check_split_attachment_status():
                    raise MetadataException("Unexpected Exception Occurred while verifying the split files "
                                            "- case_number:{}, uuid: {}".format(self.case_number, ", ".join(self.all_uuids)))
                logger.info("File verified successfully.")
                return self.fileChunk
        except KeyboardInterrupt:
            logger.debug("User has interrupted the upload.")
            raise
        except MetadataException, me:
            raise
        except DontWaitForCompletion:
            logger.debug("The user has opted not to wait for attachment verification.")
            raise
        except Exception, e:
            raise S3UploadFailedError(traceback.format_exc())


if __name__ == '__main__':
    obj = AWSUploader(case_number='123456',
                    filepath='/tmp/1MB.test',
                    use_threads=True, refresh_frequency=600, max_workers=10,
                    description='test file', isPrivate=False,
                    # fileChunk={'num': 0, 'names': [], 'size': 1000000}
                    )
    obj.upload_file()



