// Copyright 2020 The TensorFlow Runtime Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Thin wrapper around the cuFFT API that adapts the library for TFRT
// conventions.
#ifndef TFRT_GPU_WRAPPER_CUFFT_WRAPPER_H_
#define TFRT_GPU_WRAPPER_CUFFT_WRAPPER_H_

#include "cufft.h"  // from @cuda_headers
#include "tfrt/gpu/wrapper/fft_wrapper.h"

namespace tfrt {
namespace gpu {
namespace wrapper {

raw_ostream& Print(raw_ostream& os, cufftResult result);
raw_ostream& Print(raw_ostream& os, cufftType value);
raw_ostream& Print(raw_ostream& os, cufftDirection value);

Expected<cufftType> Parse(llvm::StringRef name, cufftType);
Expected<cufftDirection> Parse(llvm::StringRef name, cufftDirection);

namespace internal {
template <>
struct EnumPlatform<FftType, cufftType> : CudaPlatformType {};
template <>
struct EnumPlatform<FftDirection, cufftDirection> : CudaPlatformType {};

template <>
struct EnumStream<FftType, Platform::CUDA>
    : EnumStreamPtrs<cufftType, Parse, Print> {};
template <>
struct EnumStream<FftDirection, Platform::CUDA>
    : EnumStreamPtrs<cufftDirection, Parse, Print> {};
}  // namespace internal

llvm::Expected<LibraryVersion> CufftGetVersion();

// Creates an opaque handle and allocates small data for the plan. Use
// CufftMakePlanMany to do the plan generation. (See
// https://docs.nvidia.com/cuda/cufft/index.html#plan-extensible).
llvm::Expected<OwningFftHandle> CufftCreate(CurrentContext current);

// Frees all GPU resources associated with the handle and destroys internal data
// structures.
llvm::Error CufftDestroy(cufftHandle handle);

// Sets the stream for execution of cuFFT functions. Note that these functions
// may consist of many kernel invocations.
llvm::Error CufftSetStream(cufftHandle handle, cudaStream_t stream);

// Following a call to CufftCreate, makes a plan for the specified signal size
// and type. Work size contains the size(s) in bytes of the work areas for each
// GPU used.
llvm::Expected<size_t> CufftMakePlanMany(
    cufftHandle handle, cufftType type, int64_t batch, ArrayRef<int64_t> dims,
    ArrayRef<int64_t> input_embed, int64_t input_stride, int64_t input_dist,
    ArrayRef<int64_t> output_embed, int64_t output_stride, int64_t output_dist);

// Functions for getting estimated size of work area for temporary results
// during plan execution.
// See https://docs.nvidia.com/cuda/cufft/index.html#work-estimate.

llvm::Expected<size_t> CufftEstimateMany(
    cufftType type, int batch, int rank, ArrayRef<int> dims,
    ArrayRef<int> input_embed, int input_stride, ArrayRef<int> output_embed,
    int output_stride, int input_dist, int output_dist);

llvm::Expected<size_t> CufftGetSizeMany(
    cufftHandle handle, cufftType type, int batch, int rank,
    ArrayRef<int64_t> dims, ArrayRef<int64_t> input_embed, int64_t input_stride,
    ArrayRef<int64_t> output_embed, int64_t output_stride, int64_t input_dist,
    int64_t output_dist);

llvm::Expected<size_t> CufftGetSize(cufftHandle handle);

// Lower level memory management support.
// See https://docs.nvidia.com/cuda/cufft/index.html#unique_772799016.

llvm::Error CufftDisableAutoAllocation(cufftHandle handle);
llvm::Error CufftEnableAutoAllocation(cufftHandle handle);

llvm::Error CufftSetWorkArea(cufftHandle handle, Pointer<void> work_area);

llvm::Error CufftExec(CurrentContext current, cufftHandle handle,
                      Pointer<const void> raw_input, Pointer<void> raw_output,
                      cufftType type, cufftDirection direction);

}  // namespace wrapper
}  // namespace gpu
}  // namespace tfrt

#endif  // TFRT_GPU_WRAPPER_CUFFT_WRAPPER_H_
