// Copyright 2020 The TensorFlow Runtime Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Thin wrapper around the cuBLAS API adding llvm::Error.
#include "tfrt/gpu/wrapper/cublas_wrapper.h"

#include "wrapper_detail.h"

namespace tfrt {
namespace gpu {
namespace wrapper {

llvm::Expected<OwningBlasHandle> CublasCreate(CurrentContext current) {
  CheckCudaContext(current);
  cublasHandle_t handle = nullptr;
  RETURN_IF_ERROR(cublasCreate_v2(&handle));
  return OwningBlasHandle(handle);
}

llvm::Error CublasDestroy(cublasHandle_t handle) {
  return TO_ERROR(cublasDestroy_v2(handle));
}

llvm::Expected<int> CublasGetVersion(cublasHandle_t handle) {
  int version = 0;
  RETURN_IF_ERROR(cublasGetVersion_v2(handle, &version));
  return version;
}

llvm::Error CublasSetStream(cublasHandle_t handle, cudaStream_t stream) {
  return TO_ERROR(cublasSetStream_v2(handle, stream));
}

llvm::Expected<Stream> CublasGetStream(cublasHandle_t handle) {
  cudaStream_t stream = nullptr;
  RETURN_IF_ERROR(cublasGetStream_v2(handle, &stream));
  return Stream(stream);
}

llvm::Error CublasSetPointerMode(cublasHandle_t handle,
                                 cublasPointerMode_t mode) {
  return TO_ERROR(cublasSetPointerMode_v2(handle, mode));
}

llvm::Expected<cublasPointerMode_t> CublasGetPointerMode(
    cublasHandle_t handle) {
  cublasPointerMode_t mode;
  RETURN_IF_ERROR(cublasGetPointerMode_v2(handle, &mode));
  return mode;
}

llvm::Error CublasSetMathMode(cublasHandle_t handle, cublasMath_t math_type) {
  return TO_ERROR(cublasSetMathMode(handle, math_type));
}

llvm::Expected<cublasMath_t> CublasGetMathMode(cublasHandle_t handle) {
  cublasMath_t mode;
  RETURN_IF_ERROR(cublasGetMathMode(handle, &mode));
  return mode;
}

llvm::Error CublasAxpyEx(CurrentContext current, cublasHandle_t handle, int n,
                         Pointer<const void> alpha, /* host or device pointer */
                         cudaDataType alphaType, Pointer<const void> x,
                         cudaDataType typeX, int strideX, Pointer<void> y,
                         cudaDataType typeY, int strideY,
                         cudaDataType executionType) {
  CheckCudaContext(current);
  return TO_ERROR(cublasAxpyEx(handle, n, ToCuda(alpha), alphaType, ToCuda(x),
                               typeX, strideX, ToCuda(y), typeY, strideY,
                               executionType));
}

llvm::Error CublasGemmEx(CurrentContext current, cublasHandle_t handle,
                         cublasOperation_t transA, cublasOperation_t transB,
                         int m, int n, int k, Pointer<const void> alpha,
                         Pointer<const void> A, cudaDataType typeA, int heightA,
                         Pointer<const void> B, cudaDataType typeB, int heightB,
                         Pointer<const void> beta, Pointer<void> C,
                         cudaDataType typeC, int heightC,
                         cublasComputeType_t computeType,
                         cublasGemmAlgo_t algo) {
  CheckCudaContext(current);
  return TO_ERROR(cublasGemmEx(handle, transA, transB, m, n, k, ToCuda(alpha),
                               ToCuda(A), typeA, heightA, ToCuda(B), typeB,
                               heightB, ToCuda(beta), ToCuda(C), typeC, heightC,
                               computeType, algo));
}

llvm::Error CublasGemmBatchedEx(
    CurrentContext current, cublasHandle_t handle, cublasOperation_t transA,
    cublasOperation_t transB, int m, int n, int k, Pointer<const void> alpha,
    llvm::ArrayRef<Pointer<const void>> Aarray, cudaDataType typeA, int heightA,
    llvm::ArrayRef<Pointer<const void>> Barray, cudaDataType typeB, int heightB,
    Pointer<const void> beta, llvm::ArrayRef<Pointer<void>> Carray,
    cudaDataType typeC, int heightC, int batchCount,
    cublasComputeType_t computeType, cublasGemmAlgo_t algo) {
  CheckCudaContext(current);
  return TO_ERROR(cublasGemmBatchedEx(
      handle, transA, transB, m, n, k, ToCuda(alpha), ToCuda(Aarray).data(),
      typeA, heightA, ToCuda(Barray).data(), typeB, heightB, ToCuda(beta),
      ToCuda(Carray).data(), typeC, heightC, batchCount, computeType, algo));
}

llvm::Error CublasGemmStridedBatchedEx(
    CurrentContext current, cublasHandle_t handle, cublasOperation_t transA,
    cublasOperation_t transB, int m, int n, int k, Pointer<const void> alpha,
    Pointer<const void> A, cudaDataType typeA, int heightA, int64_t strideA,
    Pointer<const void> B, cudaDataType typeB, int heightB, int64_t strideB,
    Pointer<const void> beta, Pointer<void> C, cudaDataType typeC, int heightC,
    int64_t strideC, int batchCount, cublasComputeType_t computeType,
    cublasGemmAlgo_t algo) {
  CheckCudaContext(current);
  return TO_ERROR(cublasGemmStridedBatchedEx(
      handle, transA, transB, m, n, k, ToCuda(alpha), ToCuda(A), typeA, heightA,
      strideA, ToCuda(B), typeB, heightB, strideB, ToCuda(beta), ToCuda(C),
      typeC, heightC, strideC, batchCount, computeType, algo));
}

llvm::Error CublasScalEx(CurrentContext current, cublasHandle_t handle, int n,
                         Pointer<const void> alpha, cudaDataType alphaType,
                         Pointer<void> x, cudaDataType xType, int incx,
                         cudaDataType executionType) {
  CheckCudaContext(current);
  return TO_ERROR(cublasScalEx(handle, n, ToCuda(alpha), alphaType, ToCuda(x),
                               xType, incx, executionType));
}

llvm::Error CublasTrsmBatched(CurrentContext current, cublasHandle_t handle,
                              cudaDataType dataType, cublasSideMode_t sideMode,
                              cublasFillMode_t fillMode,
                              cublasOperation_t trans, cublasDiagType_t diag,
                              int m, int n, Pointer<const void> alpha,
                              Pointer<const void*> A, int lda, Pointer<void*> B,
                              int ldb, int batchCount) {
  CheckCudaContext(current);
  switch (dataType) {
    case CUDA_R_32F:
      return TO_ERROR(cublasStrsmBatched(
          handle, sideMode, fillMode, trans, diag, m, n,
          reinterpret_cast<const float*>(ToCuda(alpha)),
          reinterpret_cast<const float* const*>(ToCuda(A)), lda,
          reinterpret_cast<float* const*>(ToCuda(B)), ldb, batchCount));
    case CUDA_C_32F:
      return TO_ERROR(cublasCtrsmBatched(
          handle, sideMode, fillMode, trans, diag, m, n,
          reinterpret_cast<const cuComplex*>(ToCuda(alpha)),
          reinterpret_cast<const cuComplex* const*>(ToCuda(A)), lda,
          reinterpret_cast<cuComplex* const*>(ToCuda(B)), ldb, batchCount));
    case CUDA_R_64F:
      return TO_ERROR(cublasDtrsmBatched(
          handle, sideMode, fillMode, trans, diag, m, n,
          reinterpret_cast<const double*>(ToCuda(alpha)),
          reinterpret_cast<const double* const*>(ToCuda(A)), lda,
          reinterpret_cast<double* const*>(ToCuda(B)), ldb, batchCount));
    case CUDA_C_64F:
      return TO_ERROR(cublasZtrsmBatched(
          handle, sideMode, fillMode, trans, diag, m, n,
          reinterpret_cast<const cuDoubleComplex*>(ToCuda(alpha)),
          reinterpret_cast<const cuDoubleComplex* const*>(ToCuda(A)), lda,
          reinterpret_cast<cuDoubleComplex* const*>(ToCuda(B)), ldb,
          batchCount));
    default:
      return MakeStringError("Unsupported data type: ", Printed(dataType));
  }
}

}  // namespace wrapper
}  // namespace gpu
}  // namespace tfrt
