"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ciPluginReportTask = exports.ciPackagePluginTask = exports.ciBuildPluginTask = void 0;
const tslib_1 = require("tslib");
const execa = require("execa");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const path = require("path");
const rimraf_1 = tslib_1.__importDefault(require("rimraf"));
const util_1 = require("util");
const getPluginId_1 = require("../../config/utils/getPluginId");
const pluginValidation_1 = require("../../config/utils/pluginValidation");
const env_1 = require("../../plugins/env");
const manifest_1 = require("../../plugins/manifest");
const utils_1 = require("../../plugins/utils");
const workflow_1 = require("../../plugins/workflow");
const plugin_build_1 = require("./plugin.build");
const task_1 = require("./task");
const rimraf = (0, util_1.promisify)(rimraf_1.default);
/**
 * 1. BUILD
 *
 *  when platform exists it is building backend, otherwise frontend
 *
 *  Each build writes data:
 *   ~/ci/jobs/build_xxx/
 *
 *  Anything that should be put into the final zip file should be put in:
 *   ~/ci/jobs/build_xxx/dist
 *
 * @deprecated -- this task was written with a specific circle-ci build in mind.  That system
 * has been replaced with Drone, and this is no longer the best practice.  Any new work
 * should be defined in the grafana build pipeline tool or drone configs directly.
 */
const buildPluginRunner = ({ finish, maxJestWorkers }) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const start = Date.now();
    if (finish) {
        const workDir = (0, env_1.getJobFolder)();
        yield rimraf(workDir);
        fs_extra_1.default.mkdirSync(workDir);
        // Move local folders to the scoped job folder
        for (const name of ['dist', 'coverage']) {
            const dir = path.resolve(process.cwd(), name);
            if (fs_extra_1.default.existsSync(dir)) {
                fs_extra_1.default.moveSync(dir, path.resolve(workDir, name));
            }
        }
        (0, env_1.writeJobStats)(start, workDir);
    }
    else {
        // Do regular build process with coverage
        yield (0, plugin_build_1.pluginBuildRunner)({ coverage: true, maxJestWorkers });
    }
});
exports.ciBuildPluginTask = new task_1.Task('Build Plugin', buildPluginRunner);
/**
 * 2. Package
 *
 *  Take everything from `~/ci/job/{any}/dist` and
 *  1. merge it into: `~/ci/dist`
 *  2. zip it into packages in `~/ci/packages`
 *  3. prepare grafana environment in: `~/ci/grafana-test-env`
 *
 *
 * @deprecated -- this task was written with a specific circle-ci build in mind.  That system
 * has been replaced with Drone, and this is no longer the best practice.  Any new work
 * should be defined in the grafana build pipeline tool or drone configs directly.
 */
const packagePluginRunner = ({ signatureType, rootUrls }) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const start = Date.now();
    const ciDir = (0, env_1.getCiFolder)();
    const packagesDir = path.resolve(ciDir, 'packages');
    const distDir = path.resolve(ciDir, 'dist');
    const docsDir = path.resolve(ciDir, 'docs');
    const jobsDir = path.resolve(ciDir, 'jobs');
    fs_extra_1.default.exists(jobsDir, (jobsDirExists) => {
        if (!jobsDirExists) {
            throw new Error('You must run plugin:ci-build prior to running plugin:ci-package');
        }
    });
    const grafanaEnvDir = path.resolve(ciDir, 'grafana-test-env');
    yield execa('rimraf', [packagesDir, distDir, grafanaEnvDir]);
    fs_extra_1.default.mkdirSync(packagesDir);
    fs_extra_1.default.mkdirSync(distDir);
    // Updating the dist dir to have a pluginId named directory in it
    // The zip needs to contain the plugin code wrapped in directory with a pluginId name
    const distContentDir = path.resolve(distDir, (0, getPluginId_1.getPluginId)());
    fs_extra_1.default.mkdirSync(grafanaEnvDir);
    console.log('Build Dist Folder');
    // 1. Check for a local 'dist' folder
    const d = path.resolve(process.cwd(), 'dist');
    if (fs_extra_1.default.existsSync(d)) {
        yield execa('cp', ['-rn', d + '/.', distContentDir]);
    }
    // 2. Look for any 'dist' folders under ci/job/XXX/dist
    const dirs = fs_extra_1.default.readdirSync(path.resolve(ciDir, 'jobs'));
    for (const j of dirs) {
        const contents = path.resolve(ciDir, 'jobs', j, 'dist');
        if (fs_extra_1.default.existsSync(contents)) {
            try {
                yield execa('cp', ['-rn', contents + '/.', distContentDir]);
            }
            catch (er) {
                throw new Error('Duplicate files found in dist folders');
            }
        }
    }
    console.log('Save the source info in plugin.json');
    const pluginJsonFile = path.resolve(distContentDir, 'plugin.json');
    const pluginInfo = (0, pluginValidation_1.getPluginJson)(pluginJsonFile);
    pluginInfo.info.build = yield (0, env_1.getPluginBuildInfo)();
    fs_extra_1.default.writeFileSync(pluginJsonFile, JSON.stringify(pluginInfo, null, 2), { encoding: 'utf-8' });
    // Write a MANIFEST.txt file in the dist folder
    try {
        const manifest = yield (0, manifest_1.buildManifest)(distContentDir);
        if (signatureType) {
            manifest.signatureType = signatureType;
        }
        if (rootUrls && rootUrls.length > 0) {
            rootUrls.forEach(pluginValidation_1.assertRootUrlIsValid);
            manifest.rootUrls = rootUrls;
        }
        const signedManifest = yield (0, manifest_1.signManifest)(manifest);
        yield (0, manifest_1.saveManifest)(distContentDir, signedManifest);
    }
    catch (err) {
        console.warn(`Error signing manifest: ${distContentDir}`, err);
    }
    console.log('Building ZIP');
    let zipName = pluginInfo.id + '-' + pluginInfo.info.version + '.zip';
    let zipFile = path.resolve(packagesDir, zipName);
    yield execa('zip', ['-r', zipFile, '.'], { cwd: distDir });
    const zipStats = fs_extra_1.default.statSync(zipFile);
    if (zipStats.size < 100) {
        throw new Error('Invalid zip file: ' + zipFile);
    }
    // Make a copy so it is easy for report to read
    yield execa('cp', [pluginJsonFile, distDir]);
    const info = {
        plugin: yield (0, utils_1.getPackageDetails)(zipFile, distDir),
    };
    console.log('Setup Grafana Environment');
    let p = path.resolve(grafanaEnvDir, 'plugins', pluginInfo.id);
    fs_extra_1.default.mkdirSync(p, { recursive: true });
    yield execa('unzip', [zipFile, '-d', p]);
    // If docs exist, zip them into packages
    if (fs_extra_1.default.existsSync(docsDir)) {
        console.log('Creating documentation zip');
        zipName = pluginInfo.id + '-' + pluginInfo.info.version + '-docs.zip';
        zipFile = path.resolve(packagesDir, zipName);
        yield execa('zip', ['-r', zipFile, '.'], { cwd: docsDir });
        info.docs = yield (0, utils_1.getPackageDetails)(zipFile, docsDir);
    }
    p = path.resolve(packagesDir, 'info.json');
    fs_extra_1.default.writeFileSync(p, JSON.stringify(info, null, 2), { encoding: 'utf-8' });
    // Write the custom settings
    p = path.resolve(grafanaEnvDir, 'custom.ini');
    const customIniBody = `# Autogenerated by @grafana/toolkit \n` +
        `[paths] \n` +
        `plugins = ${path.resolve(grafanaEnvDir, 'plugins')}\n` +
        `\n`; // empty line
    fs_extra_1.default.writeFileSync(p, customIniBody, { encoding: 'utf-8' });
    (0, env_1.writeJobStats)(start, (0, env_1.getJobFolder)());
});
exports.ciPackagePluginTask = new task_1.Task('Bundle Plugin', packagePluginRunner);
/**
 * 4. Report
 *
 *  Create a report from all the previous steps
 *
 * @deprecated -- this task was written with a specific circle-ci build in mind.  That system
 * has been replaced with Drone, and this is no longer the best practice.  Any new work
 * should be defined in the grafana build pipeline tool or drone configs directly.
 */
const pluginReportRunner = ({ upload }) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const ciDir = path.resolve(process.cwd(), 'ci');
    const packageDir = path.resolve(ciDir, 'packages');
    const packageInfo = require(path.resolve(packageDir, 'info.json'));
    const pluginJsonFile = path.resolve(ciDir, 'dist', 'plugin.json');
    console.log('Load info from: ' + pluginJsonFile);
    const pluginMeta = (0, pluginValidation_1.getPluginJson)(pluginJsonFile);
    const report = {
        plugin: pluginMeta,
        packages: packageInfo,
        workflow: (0, workflow_1.agregateWorkflowInfo)(),
        coverage: (0, workflow_1.agregateCoverageInfo)(),
        tests: (0, workflow_1.agregateTestInfo)(),
        artifactsBaseURL: yield (0, env_1.getCircleDownloadBaseURL)(),
        grafanaVersion: (0, utils_1.getGrafanaVersions)(),
        git: yield (0, utils_1.readGitLog)(),
    };
    const pr = (0, env_1.getPullRequestNumber)();
    if (pr) {
        report.pullRequest = pr;
    }
    // Save the report to disk
    const file = path.resolve(ciDir, 'report.json');
    fs_extra_1.default.writeFileSync(file, JSON.stringify(report, null, 2), { encoding: 'utf-8' });
    const GRAFANA_API_KEY = process.env.GRAFANA_API_KEY;
    if (!GRAFANA_API_KEY) {
        console.log('Enter a GRAFANA_API_KEY to upload the plugin report');
        return;
    }
    const url = `https://grafana.com/api/plugins/${report.plugin.id}/ci`;
    console.log('Sending report to:', url);
    const axios = require('axios');
    const info = yield axios.post(url, report, {
        headers: { Authorization: 'Bearer ' + GRAFANA_API_KEY },
    });
    if (info.status === 200) {
        console.log('OK: ', info.data);
    }
    else {
        console.warn('Error: ', info);
    }
});
exports.ciPluginReportTask = new task_1.Task('Generate Plugin Report', pluginReportRunner);
//# sourceMappingURL=plugin.ci.js.map