import { Context } from '@opentelemetry/api';
import { BatchObservableResult, ObservableResult } from './ObservableResult';
/**
 * Options needed for metric creation
 */
export interface MetricOptions {
    /**
     * The description of the Metric.
     * @default ''
     */
    description?: string;
    /**
     * The unit of the Metric values.
     * @default ''
     */
    unit?: string;
    /**
     * Indicates the type of the recorded value.
     * @default {@link ValueType.DOUBLE}
     */
    valueType?: ValueType;
}
/** The Type of value. It describes how the data is reported. */
export declare enum ValueType {
    INT = 0,
    DOUBLE = 1
}
/**
 * Counter is the most common synchronous instrument. This instrument supports
 * an `Add(increment)` function for reporting a sum, and is restricted to
 * non-negative increments. The default aggregation is Sum, as for any additive
 * instrument.
 *
 * Example uses for Counter:
 * <ol>
 *   <li> count the number of bytes received. </li>
 *   <li> count the number of requests completed. </li>
 *   <li> count the number of accounts created. </li>
 *   <li> count the number of checkpoints run. </li>
 *   <li> count the number of 5xx errors. </li>
 * <ol>
 */
export interface Counter {
    /**
     * Increment value of counter by the input. Inputs may not be negative.
     */
    add(value: number, attributes?: MetricAttributes, context?: Context): void;
}
export interface UpDownCounter {
    /**
     * Increment value of counter by the input. Inputs may be negative.
     */
    add(value: number, attributes?: MetricAttributes, context?: Context): void;
}
export interface Histogram {
    /**
     * Records a measurement. Value of the measurement must not be negative.
     */
    record(value: number, attributes?: MetricAttributes, context?: Context): void;
}
/**
 * key-value pairs passed by the user.
 */
export declare type MetricAttributes = {
    [key: string]: string;
};
/**
 * The observable callback for Observable instruments.
 */
export declare type ObservableCallback = (observableResult: ObservableResult) => void | Promise<void>;
/**
 * The observable callback for a batch of Observable instruments.
 */
export declare type BatchObservableCallback = (observableResult: BatchObservableResult) => void | Promise<void>;
export interface Observable {
    /**
     * Sets up a function that will be called whenever a metric collection is initiated.
     *
     * If the function is already in the list of callbacks for this Observable, the function is not added a second time.
     */
    addCallback(callback: ObservableCallback): void;
    /**
     * Removes a callback previously registered with {@link Observable.addCallback}.
     */
    removeCallback(callback: ObservableCallback): void;
}
export declare type ObservableCounter = Observable;
export declare type ObservableUpDownCounter = Observable;
export declare type ObservableGauge = Observable;
//# sourceMappingURL=Metric.d.ts.map