// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "google/cloud/storage/client.h"
#include "google/cloud/storage/examples/storage_examples_common.h"
#include "google/cloud/internal/getenv.h"
#include <iostream>

namespace {

void CreateGetSignedUrlV2(google::cloud::storage::Client client,
                          std::vector<std::string> const& argv) {
  //! [sign url v2] [START storage_generate_signed_url_v2]
  namespace gcs = google::cloud::storage;
  using ::google::cloud::StatusOr;
  [](gcs::Client client, std::string const& bucket_name,
     std::string const& object_name) {
    StatusOr<std::string> signed_url = client.CreateV2SignedUrl(
        "GET", std::move(bucket_name), std::move(object_name),
        gcs::ExpirationTime(std::chrono::system_clock::now() +
                            std::chrono::minutes(15)));

    if (!signed_url) throw std::runtime_error(signed_url.status().message());
    std::cout << "The signed url is: " << *signed_url << "\n\n"
              << "You can use this URL with any user agent, for example:\n"
              << "curl '" << *signed_url << "'\n";
  }
  //! [sign url v2] [END storage_generate_signed_url_v2]
  (std::move(client), argv.at(0), argv.at(1));
}

void CreatePutSignedUrlV2(google::cloud::storage::Client client,
                          std::vector<std::string> const& argv) {
  //! [create put signed url v2]
  namespace gcs = google::cloud::storage;
  using ::google::cloud::StatusOr;
  [](gcs::Client client, std::string const& bucket_name,
     std::string const& object_name) {
    StatusOr<std::string> signed_url = client.CreateV2SignedUrl(
        "PUT", std::move(bucket_name), std::move(object_name),
        gcs::ExpirationTime(std::chrono::system_clock::now() +
                            std::chrono::minutes(15)),
        gcs::ContentType("application/octet-stream"));

    if (!signed_url) throw std::runtime_error(signed_url.status().message());
    std::cout << "The signed url is: " << *signed_url << "\n\n"
              << "You can use this URL with any user agent, for example:\n"
              << "curl -X PUT -H 'Content-Type: application/octet-stream'"
              << " --upload-file my-file '" << *signed_url << "'\n";
  }
  //! [create put signed url v2]
  (std::move(client), argv.at(0), argv.at(1));
}

void RunAll(std::vector<std::string> const& argv) {
  namespace examples = ::google::cloud::storage::examples;
  namespace gcs = ::google::cloud::storage;

  if (!argv.empty()) throw examples::Usage{"auto"};
  examples::CheckEnvironmentVariablesAreSet({
      "GOOGLE_CLOUD_PROJECT",
      "GOOGLE_CLOUD_CPP_STORAGE_TEST_BUCKET_NAME",
  });
  auto const project_id =
      google::cloud::internal::GetEnv("GOOGLE_CLOUD_PROJECT").value();
  auto const bucket_name = google::cloud::internal::GetEnv(
                               "GOOGLE_CLOUD_CPP_STORAGE_TEST_BUCKET_NAME")
                               .value();
  auto generator = google::cloud::internal::DefaultPRNG(std::random_device{}());
  auto const object_name =
      examples::MakeRandomObjectName(generator, "cloud-cpp-test-examples-");

  auto client = gcs::Client::CreateDefaultClient().value();

  if (examples::UsingTestbench()) {
    std::cout << "Signed URL examples are only runnable against production\n";
    return;
  }

  std::cout << "\nRunning CreatePutSignedUrlV2() example" << std::endl;
  CreatePutSignedUrlV2(client, {bucket_name, object_name});

  std::cout << "\nRunning CreateGetSignedUrlV2() example" << std::endl;
  CreateGetSignedUrlV2(client, {bucket_name, object_name});
}

}  // namespace

int main(int argc, char* argv[]) {
  namespace examples = ::google::cloud::storage::examples;
  auto make_entry = [](std::string const& name,
                       examples::ClientCommand const& cmd) {
    return examples::CreateCommandEntry(
        name, {"<bucket-name>", "<object-name>"}, cmd);
  };
  examples::Example example({
      make_entry("create-get-signed-url-v2", CreateGetSignedUrlV2),
      make_entry("create-put-signed-url-v2", CreatePutSignedUrlV2),
      {"auto", RunAll},
  });
  return example.Run(argc, argv);
}
