# CMCEnroll 1 "July 20, 2016" PKI "PKI CMC Enrollment Tool"

## NAME

CMCEnroll - Used to sign a certificate request with an agent's certificate.

**Note:**
This tool has not yet been updated to work with the latest improvement in the CA to conform to RFC 5272.
Please use **CMCRequest** instead.

## SYNOPSIS

**CMCEnroll** **-d** *NSS-database* **-n** *certificate-nickname* **-r** *certificate-request-file* **-p** *NSS-database-passwd*

## DESCRIPTION

The Certificate Management over Cryptographic Message Syntax (CMC) Enrollment utility, **CMCEnroll**,
provides a command-line utility used to sign a certificate request with an agent's certificate.
This can be used in conjunction with the CA end-entity CMC Enrollment form to sign and enroll certificates for users.

**CMCEnroll** takes a standard PKCS #10 certificate request and signs it with an agent certificate.
The output is also a certificate request which can be submitted through the appropriate profile.

## OPTIONS

The following parameters are mandatory:

**Note:**
Surround values that include spaces with quotation marks.

**-d** *NSS-database*  
    The directory containing the NSS database associated with the agent certificate.
    This is usually the agent's personal directory, such as their browser certificate database in the home directory.

**-n** *certificate-nickname*  
    The nickname of the agent certificate that is used to sign the request.

**-r** *certificate-request-file*  
    The filename of the certificate request.

**-p** *NSS-database-passwd*  
    The password to the NSS certificate database which contains the agent certificate,
    given in **-d** *NSS-database*.

## EXAMPLES

Signed requests must be submitted to the CA to be processed.

**Note:**
For this example to work automatically, the **CMCAuth** plug-in must be enabled on the CA server (which it is by default).

(1) Create a PKCS #10 certificate request using a tool like **certutil**:

```
$ cd $HOME/.mozilla/firefox/<profile>

$ certutil -L -d .
Certificate Nickname                                         Trust Attributes
                                                             SSL,S/MIME,JAR/XPI

Google Internet Authority G2                                 ,,
COMODO RSA Domain Validation Secure Server CA                ,,
pki.example.com                                              ,,
DigiCert SHA2 Secure Server CA                               ,,
DigiCert SHA2 Extended Validation Server CA                  ,,
COMODO RSA Extended Validation Secure Server CA 2            ,,
Symantec Class 3 Secure Server CA - G4                       ,,
Go Daddy Secure Certificate Authority - G2                   ,,
Oracle SSL CA - G2                                           ,,
GeoTrust EV SSL CA - G4                                      ,,
Symantec Class 3 Secure Server SHA256 SSL CA                 ,,
GeoTrust SSL CA - G3                                         ,,
PKI Administrator for example.com                            u,u,u
DigiCert SHA2 High Assurance Server CA                       ,,
COMODO RSA Organization Validation Secure Server CA          ,,
CA Signing Certificate - example.com Security Domain         CT,C,C

$ certutil -R -d . -s "CN=CMCEnroll Test Certificate" -a

A random seed must be generated that will be used in the
creation of your key.  One of the easiest ways to create a
random seed is to use the timing of keystrokes on a keyboard.

To begin, type keys on the keyboard until this progress meter
is full.  DO NOT USE THE AUTOREPEAT FUNCTION ON YOUR KEYBOARD!


Continue typing until the progress meter is full:

|************************************************************|

Finished.  Press enter to continue:


Generating key.  This may take a few moments...


Certificate request generated by Netscape certutil
Phone: (not specified)

Common Name: CMCEnroll Test Certificate
Email: (not specified)
Organization: (not specified)
State: (not specified)
Country: (not specified)

-----BEGIN CERTIFICATE REQUEST-----
MIICajCCAVICAQAwJTEjMCEGA1UEAxMaQ01DRW5yb2xsIFRlc3QgQ2VydGlmaWNh
dGUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDamQA6psK7Tnic3DAt
IyAMCk7FK3PuSseJSrR/C7W05tPvrlp5vUKxpmcA+Pg3AANp5gVMQOps6riAvoK7
6NKTkw4Me09Cowad7ay9IBBY4QqqBmRnfT3Mm6U5tJWeqvq1cIkwoxzHllgsGBGM
QduI7URjhQYx3p+srGSe0fM7bqK+AU6aJh4r0jc1A6pCv/2XMOY1IUzmjIEnNq2R
WOpnsWQ4UDma1r8sUzKgNhkuhjPU5U5YGt9+0jiuqv14dbKi7UJN3DPtkEXZNOrF
rGgqKhdUqLhrdm+x/Hgw/aZoSDFYXON9jFTFyMUyUkWXZq5sfwghWUC2q4DsbfvH
68h1AgMBAAGgADANBgkqhkiG9w0BAQsFAAOCAQEAQ9aHQvPDcDuOJOL62pQeoDJp
YtFmsDaksdhedG27usjPuX06XmzSIV3/D2zfPib2fpfdrHB5901TdehlghQVOkN6
sSoih60GSD9zCkFD1eESywJJeZssRfDG4gk2Ls9wXz5ZY/QwSx6C97SodF0cuDHL
FsymesuxhePL7sYkkmazjgQTkA/JXLe6FYX213xQ+FGfQvmAqc9xHu5jvnBXX+Ub
ucixaLKUiRIVHfTmuUb/qenEBQM2vzWDZawHL5SBSa/Zxjy2iVMrQBeOiLcu8bTL
TAmSCbonRTilFrKFVG0H+Y9+5bulOdJc64XOvj9DRJd1FJoocw0eGhw31I5rJA==
-----END CERTIFICATE REQUEST-----
```


(2) Copy the PKCS #10 ASCII output to a text file.

```
$ vi cert.req
-----BEGIN CERTIFICATE REQUEST-----
MIICajCCAVICAQAwJTEjMCEGA1UEAxMaQ01DRW5yb2xsIFRlc3QgQ2VydGlmaWNh
dGUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDamQA6psK7Tnic3DAt
IyAMCk7FK3PuSseJSrR/C7W05tPvrlp5vUKxpmcA+Pg3AANp5gVMQOps6riAvoK7
6NKTkw4Me09Cowad7ay9IBBY4QqqBmRnfT3Mm6U5tJWeqvq1cIkwoxzHllgsGBGM
QduI7URjhQYx3p+srGSe0fM7bqK+AU6aJh4r0jc1A6pCv/2XMOY1IUzmjIEnNq2R
WOpnsWQ4UDma1r8sUzKgNhkuhjPU5U5YGt9+0jiuqv14dbKi7UJN3DPtkEXZNOrF
rGgqKhdUqLhrdm+x/Hgw/aZoSDFYXON9jFTFyMUyUkWXZq5sfwghWUC2q4DsbfvH
68h1AgMBAAGgADANBgkqhkiG9w0BAQsFAAOCAQEAQ9aHQvPDcDuOJOL62pQeoDJp
YtFmsDaksdhedG27usjPuX06XmzSIV3/D2zfPib2fpfdrHB5901TdehlghQVOkN6
sSoih60GSD9zCkFD1eESywJJeZssRfDG4gk2Ls9wXz5ZY/QwSx6C97SodF0cuDHL
FsymesuxhePL7sYkkmazjgQTkA/JXLe6FYX213xQ+FGfQvmAqc9xHu5jvnBXX+Ub
ucixaLKUiRIVHfTmuUb/qenEBQM2vzWDZawHL5SBSa/Zxjy2iVMrQBeOiLcu8bTL
TAmSCbonRTilFrKFVG0H+Y9+5bulOdJc64XOvj9DRJd1FJoocw0eGhw31I5rJA==
-----END CERTIFICATE REQUEST-----
```


(3) Run the **CMCEnroll** command to sign the certificate request.
If the input file is "$HOME/.mozilla/firefox/&lt;profile&gt;/cert.req",
the agent's certificate is stored in the "$HOME/.mozilla/firefox/&lt;profile&gt;" directory,
the certificate common name for this CA is "PKI Administrator for example.com",
and the password for the certificate database is "Secret.123", the command is as follows:

```
$ CMCEnroll -d "$HOME/.mozilla/firefox/<profile>" \
    -n "PKI Administrator for example.com" \
    -r "$HOME/.mozilla/firefox/<profile>/cert.req" \
    -p "Secret.123"
cert/key prefix =
path = <home>/.mozilla/firefox/<profile>
-----BEGIN CERTIFICATE REQUEST-----
MIICajCCAVICAQAwJTEjMCEGA1UEAxMaQ01DRW5yb2xsIFRlc3QgQ2VydGlmaWNh
dGUwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDamQA6psK7Tnic3DAt
IyAMCk7FK3PuSseJSrR/C7W05tPvrlp5vUKxpmcA+Pg3AANp5gVMQOps6riAvoK7
6NKTkw4Me09Cowad7ay9IBBY4QqqBmRnfT3Mm6U5tJWeqvq1cIkwoxzHllgsGBGM
QduI7URjhQYx3p+srGSe0fM7bqK+AU6aJh4r0jc1A6pCv/2XMOY1IUzmjIEnNq2R
WOpnsWQ4UDma1r8sUzKgNhkuhjPU5U5YGt9+0jiuqv14dbKi7UJN3DPtkEXZNOrF
rGgqKhdUqLhrdm+x/Hgw/aZoSDFYXON9jFTFyMUyUkWXZq5sfwghWUC2q4DsbfvH
68h1AgMBAAGgADANBgkqhkiG9w0BAQsFAAOCAQEAQ9aHQvPDcDuOJOL62pQeoDJp
YtFmsDaksdhedG27usjPuX06XmzSIV3/D2zfPib2fpfdrHB5901TdehlghQVOkN6
sSoih60GSD9zCkFD1eESywJJeZssRfDG4gk2Ls9wXz5ZY/QwSx6C97SodF0cuDHL
FsymesuxhePL7sYkkmazjgQTkA/JXLe6FYX213xQ+FGfQvmAqc9xHu5jvnBXX+Ub
ucixaLKUiRIVHfTmuUb/qenEBQM2vzWDZawHL5SBSa/Zxjy2iVMrQBeOiLcu8bTL
TAmSCbonRTilFrKFVG0H+Y9+5bulOdJc64XOvj9DRJd1FJoocw0eGhw31I5rJA==
-----END CERTIFICATE REQUEST-----
```

The output of this command is stored in a file with the same filename
as the request with a **.out** appended to the filename (e.g. cert.req.out):

```
$ cat cert.req.out
-----BEGIN CERTIFICATE REQUEST-----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-----END CERTIFICATE REQUEST-----
```

(4) Submit the signed certificate request through the CA end-entities page:

(a) Open the end-entities page.

(b) Select the "Signed CMC-Authenticated User Certificate Enrollment" profile.

(c) Paste the content of the output file into the first text area of this form.

(d) Remove the "-----BEGIN CERTIFICATE REQUEST-----" header and the "-----END CERTIFICATE REQUEST-----" footer from the pasted content.

(e) Fill in the contact information, and submit the form.


(5) The certificate is immediately processed and returned since a signed request was sent and the CMCAuth plug-in was enabled:

```
Congratulations, your request has been processed successfully

Your request ID is 7.

Outputs

* Certificate Pretty Print

    Certificate:
        Data:
            Version:  v3
            Serial Number: 0x7
            Signature Algorithm: SHA256withRSA - 1.2.840.113549.1.1.11
            Issuer: CN=CA Signing Certificate,O=example.com Security Domain
            Validity:
                Not Before: Thursday, July 21, 2016 6:28:20 PM MDT America/Denver
                Not  After: Tuesday, January 17, 2017 6:28:20 PM MST America/Denver
            Subject: CN=CMCEnroll Test Certificate
            Subject Public Key Info:
                Algorithm: RSA - 1.2.840.113549.1.1.1
                Public Key:
                    Exponent: 65537
                    Public Key Modulus: (2048 bits) :
                        DA:99:00:3A:A6:C2:BB:4E:78:9C:DC:30:2D:23:20:0C:
                        0A:4E:C5:2B:73:EE:4A:C7:89:4A:B4:7F:0B:B5:B4:E6:
                        D3:EF:AE:5A:79:BD:42:B1:A6:67:00:F8:F8:37:00:03:
                        69:E6:05:4C:40:EA:6C:EA:B8:80:BE:82:BB:E8:D2:93:
                        93:0E:0C:7B:4F:42:A3:06:9D:ED:AC:BD:20:10:58:E1:
                        0A:AA:06:64:67:7D:3D:CC:9B:A5:39:B4:95:9E:AA:FA:
                        B5:70:89:30:A3:1C:C7:96:58:2C:18:11:8C:41:DB:88:
                        ED:44:63:85:06:31:DE:9F:AC:AC:64:9E:D1:F3:3B:6E:
                        A2:BE:01:4E:9A:26:1E:2B:D2:37:35:03:AA:42:BF:FD:
                        97:30:E6:35:21:4C:E6:8C:81:27:36:AD:91:58:EA:67:
                        B1:64:38:50:39:9A:D6:BF:2C:53:32:A0:36:19:2E:86:
                        33:D4:E5:4E:58:1A:DF:7E:D2:38:AE:AA:FD:78:75:B2:
                        A2:ED:42:4D:DC:33:ED:90:45:D9:34:EA:C5:AC:68:2A:
                        2A:17:54:A8:B8:6B:76:6F:B1:FC:78:30:FD:A6:68:48:
                        31:58:5C:E3:7D:8C:54:C5:C8:C5:32:52:45:97:66:AE:
                        6C:7F:08:21:59:40:B6:AB:80:EC:6D:FB:C7:EB:C8:75
            Extensions:
                Identifier: Authority Key Identifier - 2.5.29.35
                    Critical: no
                    Key Identifier:
                        BB:36:98:5D:65:CB:88:E0:87:23:37:6F:5B:F7:AF:8B:
                        8A:EB:BA:B5
                Identifier: Authority Info Access: - 1.3.6.1.5.5.7.1.1
                    Critical: no
                    Access Description:
                        Method #0: ocsp
                        Location #0: URIName: http://pki.example.com:8080/ca/ocsp
                Identifier: Key Usage: - 2.5.29.15
                    Critical: yes
                    Key Usage:
                        Digital Signature
                        Non Repudiation
                        Key Encipherment
                Identifier: Extended Key Usage: - 2.5.29.37
                    Critical: no
                    Extended Key Usage:
                        1.3.6.1.5.5.7.3.2
                        1.3.6.1.5.5.7.3.4
        Signature:
            Algorithm: SHA256withRSA - 1.2.840.113549.1.1.11
            Signature:
                6D:8B:99:D2:E9:D3:4E:7F:55:20:A6:7F:80:0C:72:B4:
                30:C5:4F:CB:D4:AC:57:85:D7:D2:CA:75:90:F7:2F:57:
                11:CB:67:16:08:0C:4C:23:D2:A5:A7:2E:4E:21:39:F5:
                D5:C7:6D:0B:DC:AD:48:E2:92:FF:99:C5:FC:CF:0E:89:
                69:B9:09:BA:9F:0E:84:AB:81:32:A7:8B:99:30:DF:75:
                2F:6C:61:5A:9C:87:77:DA:2C:EA:40:85:20:F2:DE:95:
                76:6B:D7:0B:8C:88:25:62:00:2D:04:30:F0:24:4B:64:
                2A:4A:E7:37:04:A2:BC:AD:B7:7F:BA:AA:74:41:2C:55:
                E9:E5:4B:92:18:BC:18:DC:FC:4B:EA:15:18:CE:B0:7A:
                3A:84:64:E2:31:1C:64:0A:79:3E:80:6E:43:12:30:8A:
                2A:67:6F:56:4B:56:55:C7:56:86:87:27:E4:C3:28:CA:
                05:D2:BD:0B:5D:10:A2:4E:96:9D:5B:2A:A0:0B:9B:B6:
                BB:8F:15:1F:D3:AF:79:E0:38:D3:F1:ED:D5:F1:F0:EB:
                F8:66:56:3F:2F:4F:4A:93:0E:2E:11:F3:F7:1B:37:61:
                08:E4:4A:92:4C:60:E3:1E:0A:0D:61:F2:AF:B2:E3:48:
                39:74:AA:5E:32:5B:AB:F3:55:3B:6B:1B:33:48:CB:21
        FingerPrint
            MD2:
                C2:58:80:9F:03:7D:5A:C2:3A:C2:42:D9:B8:CF:2D:17
            MD5:
                5F:D3:7C:1D:1F:59:3D:11:5E:B4:BE:75:D7:61:47:C6
            SHA-1:
                F4:29:98:68:76:3F:41:FD:5E:E9:C3:F6:8A:3A:25:F3:
                5C:A9:71:27
            SHA-256:
                66:8F:00:98:D4:FF:F1:E4:35:F2:8E:54:26:AD:98:02:
                8F:6C:98:02:49:0B:A7:E5:98:41:1D:FE:92:E1:6A:57
            SHA-512:
                E3:DB:3E:FB:9F:5F:CF:6D:79:1A:15:68:1A:42:5E:73:
                9A:ED:15:98:1D:D9:31:AF:00:45:37:1E:8A:98:C1:EA:
                F0:DF:57:E9:A7:F7:19:01:5B:79:2B:79:07:CE:66:D6:
                D6:C3:42:C9:D5:EE:50:71:7D:A5:94:DF:25:E6:CC:49

* Certificate Base-64 Encoded

-----BEGIN CERTIFICATE-----
MIIDkjCCAnqgAwIBAgIBBzANBgkqhkiG9w0BAQsFADBOMSswKQYDVQQKDCJ1c2Vy
c3lzLnJlZGhhdC5jb20gU2VjdXJpdHkgRG9tYWluMR8wHQYDVQQDDBZDQSBTaWdu
aW5nIENlcnRpZmljYXRlMB4XDTE2MDcyMjAwMjgyMFoXDTE3MDExODAxMjgyMFow
JTEjMCEGA1UEAxMaQ01DRW5yb2xsIFRlc3QgQ2VydGlmaWNhdGUwggEiMA0GCSqG
SIb3DQEBAQUAA4IBDwAwggEKAoIBAQDamQA6psK7Tnic3DAtIyAMCk7FK3PuSseJ
SrR/C7W05tPvrlp5vUKxpmcA+Pg3AANp5gVMQOps6riAvoK76NKTkw4Me09Cowad
7ay9IBBY4QqqBmRnfT3Mm6U5tJWeqvq1cIkwoxzHllgsGBGMQduI7URjhQYx3p+s
rGSe0fM7bqK+AU6aJh4r0jc1A6pCv/2XMOY1IUzmjIEnNq2RWOpnsWQ4UDma1r8s
UzKgNhkuhjPU5U5YGt9+0jiuqv14dbKi7UJN3DPtkEXZNOrFrGgqKhdUqLhrdm+x
/Hgw/aZoSDFYXON9jFTFyMUyUkWXZq5sfwghWUC2q4DsbfvH68h1AgMBAAGjgaMw
gaAwHwYDVR0jBBgwFoAUuzaYXWXLiOCHIzdvW/evi4rrurUwTgYIKwYBBQUHAQEE
QjBAMD4GCCsGAQUFBzABhjJodHRwOi8vcGtpLWRlc2t0b3AudXNlcnN5cy5yZWRo
YXQuY29tOjgwODAvY2Evb2NzcDAOBgNVHQ8BAf8EBAMCBeAwHQYDVR0lBBYwFAYI
KwYBBQUHAwIGCCsGAQUFBwMEMA0GCSqGSIb3DQEBCwUAA4IBAQBti5nS6dNOf1Ug
pn+ADHK0MMVPy9SsV4XX0sp1kPcvVxHLZxYIDEwj0qWnLk4hOfXVx20L3K1I4pL/
mcX8zw6JabkJup8OhKuBMqeLmTDfdS9sYVqch3faLOpAhSDy3pV2a9cLjIglYgAt
BDDwJEtkKkrnNwSivK23f7qqdEEsVenlS5IYvBjc/EvqFRjOsHo6hGTiMRxkCnk+
gG5DEjCKKmdvVktWVcdWhocn5MMoygXSvQtdEKJOlp1bKqALm7a7jxUf06954DjT
8e3V8fDr+GZWPy9PSpMOLhHz9xs3YQjkSpJMYOMeCg1h8q+y40g5dKpeMlur81U7
axszSMsh
-----END CERTIFICATE-----

* Certificate Imports
----------------------
| Import Certificate |
----------------------
```

(6) Use the agent page to search for the new certificate:

```
Certificate   0x07

Certificate contents

    Certificate:
        Data:
            Version:  v3
            Serial Number: 0x7
            Signature Algorithm: SHA256withRSA - 1.2.840.113549.1.1.11
            Issuer: CN=CA Signing Certificate,O=example.com Security Domain
            Validity:
                Not Before: Thursday, July 21, 2016 6:28:20 PM MDT America/Denver
                Not  After: Tuesday, January 17, 2017 6:28:20 PM MST America/Denver
            Subject: CN=CMCEnroll Test Certificate
            Subject Public Key Info:
                Algorithm: RSA - 1.2.840.113549.1.1.1
                Public Key:
                    Exponent: 65537
                    Public Key Modulus: (2048 bits) :
                        DA:99:00:3A:A6:C2:BB:4E:78:9C:DC:30:2D:23:20:0C:
                        0A:4E:C5:2B:73:EE:4A:C7:89:4A:B4:7F:0B:B5:B4:E6:
                        D3:EF:AE:5A:79:BD:42:B1:A6:67:00:F8:F8:37:00:03:
                        69:E6:05:4C:40:EA:6C:EA:B8:80:BE:82:BB:E8:D2:93:
                        93:0E:0C:7B:4F:42:A3:06:9D:ED:AC:BD:20:10:58:E1:
                        0A:AA:06:64:67:7D:3D:CC:9B:A5:39:B4:95:9E:AA:FA:
                        B5:70:89:30:A3:1C:C7:96:58:2C:18:11:8C:41:DB:88:
                        ED:44:63:85:06:31:DE:9F:AC:AC:64:9E:D1:F3:3B:6E:
                        A2:BE:01:4E:9A:26:1E:2B:D2:37:35:03:AA:42:BF:FD:
                        97:30:E6:35:21:4C:E6:8C:81:27:36:AD:91:58:EA:67:
                        B1:64:38:50:39:9A:D6:BF:2C:53:32:A0:36:19:2E:86:
                        33:D4:E5:4E:58:1A:DF:7E:D2:38:AE:AA:FD:78:75:B2:
                        A2:ED:42:4D:DC:33:ED:90:45:D9:34:EA:C5:AC:68:2A:
                        2A:17:54:A8:B8:6B:76:6F:B1:FC:78:30:FD:A6:68:48:
                        31:58:5C:E3:7D:8C:54:C5:C8:C5:32:52:45:97:66:AE:
                        6C:7F:08:21:59:40:B6:AB:80:EC:6D:FB:C7:EB:C8:75
            Extensions:
                Identifier: Authority Key Identifier - 2.5.29.35
                    Critical: no
                    Key Identifier:
                        BB:36:98:5D:65:CB:88:E0:87:23:37:6F:5B:F7:AF:8B:
                        8A:EB:BA:B5
                Identifier: Authority Info Access: - 1.3.6.1.5.5.7.1.1
                    Critical: no
                    Access Description:
                        Method #0: ocsp
                        Location #0: URIName: http://pki.example.com:8080/ca/ocsp
                Identifier: Key Usage: - 2.5.29.15
                    Critical: yes
                    Key Usage:
                        Digital Signature
                        Non Repudiation
                        Key Encipherment
                Identifier: Extended Key Usage: - 2.5.29.37
                    Critical: no
                    Extended Key Usage:
                        1.3.6.1.5.5.7.3.2
                        1.3.6.1.5.5.7.3.4
        Signature:
            Algorithm: SHA256withRSA - 1.2.840.113549.1.1.11
            Signature:
                6D:8B:99:D2:E9:D3:4E:7F:55:20:A6:7F:80:0C:72:B4:
                30:C5:4F:CB:D4:AC:57:85:D7:D2:CA:75:90:F7:2F:57:
                11:CB:67:16:08:0C:4C:23:D2:A5:A7:2E:4E:21:39:F5:
                D5:C7:6D:0B:DC:AD:48:E2:92:FF:99:C5:FC:CF:0E:89:
                69:B9:09:BA:9F:0E:84:AB:81:32:A7:8B:99:30:DF:75:
                2F:6C:61:5A:9C:87:77:DA:2C:EA:40:85:20:F2:DE:95:
                76:6B:D7:0B:8C:88:25:62:00:2D:04:30:F0:24:4B:64:
                2A:4A:E7:37:04:A2:BC:AD:B7:7F:BA:AA:74:41:2C:55:
                E9:E5:4B:92:18:BC:18:DC:FC:4B:EA:15:18:CE:B0:7A:
                3A:84:64:E2:31:1C:64:0A:79:3E:80:6E:43:12:30:8A:
                2A:67:6F:56:4B:56:55:C7:56:86:87:27:E4:C3:28:CA:
                05:D2:BD:0B:5D:10:A2:4E:96:9D:5B:2A:A0:0B:9B:B6:
                BB:8F:15:1F:D3:AF:79:E0:38:D3:F1:ED:D5:F1:F0:EB:
                F8:66:56:3F:2F:4F:4A:93:0E:2E:11:F3:F7:1B:37:61:
                08:E4:4A:92:4C:60:E3:1E:0A:0D:61:F2:AF:B2:E3:48:
                39:74:AA:5E:32:5B:AB:F3:55:3B:6B:1B:33:48:CB:21
        FingerPrint
            MD2:
                C2:58:80:9F:03:7D:5A:C2:3A:C2:42:D9:B8:CF:2D:17
            MD5:
                5F:D3:7C:1D:1F:59:3D:11:5E:B4:BE:75:D7:61:47:C6
            SHA-1:
                F4:29:98:68:76:3F:41:FD:5E:E9:C3:F6:8A:3A:25:F3:
                5C:A9:71:27
            SHA-256:
                66:8F:00:98:D4:FF:F1:E4:35:F2:8E:54:26:AD:98:02:
                8F:6C:98:02:49:0B:A7:E5:98:41:1D:FE:92:E1:6A:57
            SHA-512:
                E3:DB:3E:FB:9F:5F:CF:6D:79:1A:15:68:1A:42:5E:73:
                9A:ED:15:98:1D:D9:31:AF:00:45:37:1E:8A:98:C1:EA:
                F0:DF:57:E9:A7:F7:19:01:5B:79:2B:79:07:CE:66:D6:
                D6:C3:42:C9:D5:EE:50:71:7D:A5:94:DF:25:E6:CC:49

Certificate request info

Request ID: 7

Installing this certificate in a server

The following format can be used to install this certificate into a server.

Base 64 encoded certificate

-----BEGIN CERTIFICATE-----
MIIDkjCCAnqgAwIBAgIBBzANBgkqhkiG9w0BAQsFADBOMSswKQYDVQQKDCJ1c2Vy
c3lzLnJlZGhhdC5jb20gU2VjdXJpdHkgRG9tYWluMR8wHQYDVQQDDBZDQSBTaWdu
aW5nIENlcnRpZmljYXRlMB4XDTE2MDcyMjAwMjgyMFoXDTE3MDExODAxMjgyMFow
JTEjMCEGA1UEAxMaQ01DRW5yb2xsIFRlc3QgQ2VydGlmaWNhdGUwggEiMA0GCSqG
SIb3DQEBAQUAA4IBDwAwggEKAoIBAQDamQA6psK7Tnic3DAtIyAMCk7FK3PuSseJ
SrR/C7W05tPvrlp5vUKxpmcA+Pg3AANp5gVMQOps6riAvoK76NKTkw4Me09Cowad
7ay9IBBY4QqqBmRnfT3Mm6U5tJWeqvq1cIkwoxzHllgsGBGMQduI7URjhQYx3p+s
rGSe0fM7bqK+AU6aJh4r0jc1A6pCv/2XMOY1IUzmjIEnNq2RWOpnsWQ4UDma1r8s
UzKgNhkuhjPU5U5YGt9+0jiuqv14dbKi7UJN3DPtkEXZNOrFrGgqKhdUqLhrdm+x
/Hgw/aZoSDFYXON9jFTFyMUyUkWXZq5sfwghWUC2q4DsbfvH68h1AgMBAAGjgaMw
gaAwHwYDVR0jBBgwFoAUuzaYXWXLiOCHIzdvW/evi4rrurUwTgYIKwYBBQUHAQEE
QjBAMD4GCCsGAQUFBzABhjJodHRwOi8vcGtpLWRlc2t0b3AudXNlcnN5cy5yZWRo
YXQuY29tOjgwODAvY2Evb2NzcDAOBgNVHQ8BAf8EBAMCBeAwHQYDVR0lBBYwFAYI
KwYBBQUHAwIGCCsGAQUFBwMEMA0GCSqGSIb3DQEBCwUAA4IBAQBti5nS6dNOf1Ug
pn+ADHK0MMVPy9SsV4XX0sp1kPcvVxHLZxYIDEwj0qWnLk4hOfXVx20L3K1I4pL/
mcX8zw6JabkJup8OhKuBMqeLmTDfdS9sYVqch3faLOpAhSDy3pV2a9cLjIglYgAt
BDDwJEtkKkrnNwSivK23f7qqdEEsVenlS5IYvBjc/EvqFRjOsHo6hGTiMRxkCnk+
gG5DEjCKKmdvVktWVcdWhocn5MMoygXSvQtdEKJOlp1bKqALm7a7jxUf06954DjT
8e3V8fDr+GZWPy9PSpMOLhHz9xs3YQjkSpJMYOMeCg1h8q+y40g5dKpeMlur81U7
axszSMsh
-----END CERTIFICATE-----

Base 64 encoded certificate with CA certificate chain in pkcs7 format

-----BEGIN PKCS7-----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-----END PKCS7-----
```

## SEE ALSO

**CMCRequest(1)**, **CMCResponse(1)**, **CMCRevoke(1)**, **pki(1)**

## AUTHORS

Matthew Harmsen &lt;mharmsen@redhat.com&gt;.

## COPYRIGHT

Copyright (c) 2016 Red Hat, Inc.
This is licensed under the GNU General Public License, version 2 (GPLv2).
A copy of this license is available at http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt.
