// SPDX-License-Identifier: GPL-2.0
#include <linux/tboot.h>

#include <asm/cpufeature.h>
#include <asm/msr-index.h>
#include <asm/processor.h>

static void clear_sgx_caps(void)
{
	setup_clear_cpu_cap(X86_FEATURE_SGX);
	setup_clear_cpu_cap(X86_FEATURE_SGX_LC);
}

static int __init nosgx(char *str)
{
	clear_sgx_caps();

	return 0;
}

early_param("nosgx", nosgx);

void init_ia32_feat_ctl(struct cpuinfo_x86 *c)
{
	bool enable_sgx;
	u64 msr;

	if (rdmsrl_safe(MSR_IA32_FEATURE_CONTROL, &msr)) {
		clear_sgx_caps();
		return;
	}

	/*
	 * Enable SGX if and only if the kernel supports SGX and Launch Control
	 * is supported, i.e. disable SGX if the LE hash MSRs can't be written.
	 */
	enable_sgx = cpu_has(c, X86_FEATURE_SGX) &&
		     cpu_has(c, X86_FEATURE_SGX_LC) &&
		     IS_ENABLED(CONFIG_X86_SGX);

	if (msr & FEATURE_CONTROL_LOCKED)
		goto update_sgx;

	/*
	 * Ignore whatever value BIOS left in the MSR to avoid enabling random
	 * features or faulting on the WRMSR.
	 */
	msr = FEATURE_CONTROL_LOCKED;

	if (enable_sgx)
		msr |= FEAT_CTL_SGX_ENABLED | FEAT_CTL_SGX_LC_ENABLED;

	wrmsrl(MSR_IA32_FEATURE_CONTROL, msr);

update_sgx:
	if (!(msr & FEAT_CTL_SGX_ENABLED) ||
	    !(msr & FEAT_CTL_SGX_LC_ENABLED) || !enable_sgx) {
		if (enable_sgx)
			pr_err_once("SGX disabled by BIOS\n");
		clear_sgx_caps();
	}
}
