/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.license;

import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateTaskConfig;
import org.elasticsearch.cluster.ClusterStateTaskExecutor;
import org.elasticsearch.cluster.ClusterStateTaskListener;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.component.Lifecycle;
import org.elasticsearch.common.hash.MessageDigests;
import org.elasticsearch.common.logging.LoggerMessageFormat;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.time.DateFormatter;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.env.Environment;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.license.ExpirationCallback;
import org.elasticsearch.license.License;
import org.elasticsearch.license.LicenseOverrides;
import org.elasticsearch.license.LicenseUtils;
import org.elasticsearch.license.LicenseVerifier;
import org.elasticsearch.license.LicensesMetadata;
import org.elasticsearch.license.OperationModeFileWatcher;
import org.elasticsearch.license.PostStartBasicRequest;
import org.elasticsearch.license.PostStartBasicResponse;
import org.elasticsearch.license.PostStartTrialRequest;
import org.elasticsearch.license.PostStartTrialResponse;
import org.elasticsearch.license.PutLicenseRequest;
import org.elasticsearch.license.SelfGeneratedLicense;
import org.elasticsearch.license.StartBasicClusterTask;
import org.elasticsearch.license.StartTrialClusterTask;
import org.elasticsearch.license.StartupSelfGeneratedLicenseTask;
import org.elasticsearch.license.XPackLicenseState;
import org.elasticsearch.protocol.xpack.XPackInfoResponse;
import org.elasticsearch.protocol.xpack.license.LicenseStatus;
import org.elasticsearch.protocol.xpack.license.LicensesStatus;
import org.elasticsearch.protocol.xpack.license.PutLicenseResponse;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.watcher.ResourceWatcherService;
import org.elasticsearch.xpack.core.XPackPlugin;
import org.elasticsearch.xpack.core.XPackSettings;
import org.elasticsearch.xpack.core.scheduler.SchedulerEngine;

public class LicenseService
extends AbstractLifecycleComponent
implements ClusterStateListener,
SchedulerEngine.Listener {
    private static final Logger logger = LogManager.getLogger(LicenseService.class);
    public static final Setting<License.LicenseType> SELF_GENERATED_LICENSE_TYPE = new Setting("xpack.license.self_generated.type", License.LicenseType.BASIC.getTypeName(), s -> {
        License.LicenseType type = License.LicenseType.parse(s);
        return SelfGeneratedLicense.validateSelfGeneratedType(type);
    }, new Setting.Property[]{Setting.Property.NodeScope});
    public static final List<License.LicenseType> ALLOWABLE_UPLOAD_TYPES = LicenseService.getAllowableUploadTypes();
    public static final Setting<List<License.LicenseType>> ALLOWED_LICENSE_TYPES_SETTING = Setting.listSetting((String)"xpack.license.upload.types", ALLOWABLE_UPLOAD_TYPES.stream().map(License.LicenseType::getTypeName).toList(), License.LicenseType::parse, LicenseService::validateUploadTypesSetting, (Setting.Property[])new Setting.Property[]{Setting.Property.NodeScope});
    static final TimeValue NON_BASIC_SELF_GENERATED_LICENSE_DURATION = TimeValue.timeValueHours((long)720L);
    static final Set<License.LicenseType> VALID_TRIAL_TYPES = Set.of(License.LicenseType.GOLD, License.LicenseType.PLATINUM, License.LicenseType.ENTERPRISE, License.LicenseType.TRIAL);
    static final TimeValue LICENSE_EXPIRATION_WARNING_PERIOD = LicenseService.days(7);
    public static final long BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS = XPackInfoResponse.BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS;
    private final Settings settings;
    private final ClusterService clusterService;
    private final XPackLicenseState licenseState;
    private final AtomicReference<License> currentLicenseHolder = new AtomicReference();
    private final SchedulerEngine scheduler;
    private final Clock clock;
    private final OperationModeFileWatcher operationModeFileWatcher;
    private final List<ExpirationCallback> expirationCallbacks = new ArrayList<ExpirationCallback>();
    private final List<License.LicenseType> allowedLicenseTypes;
    private final StartTrialClusterTask.Executor startTrialExecutor = new StartTrialClusterTask.Executor();
    private final StartBasicClusterTask.Executor startBasicExecutor = new StartBasicClusterTask.Executor();
    static final int SELF_GENERATED_LICENSE_MAX_NODES = 1000;
    static final int SELF_GENERATED_LICENSE_MAX_RESOURCE_UNITS = 1000;
    public static final String LICENSE_JOB = "licenseJob";
    public static final DateFormatter DATE_FORMATTER = DateFormatter.forPattern((String)"EEEE, MMMM dd, yyyy");
    private static final String ACKNOWLEDGEMENT_HEADER = "This license update requires acknowledgement. To acknowledge the license, please read the following messages and update the license again, this time with the \"acknowledge=true\" parameter:";

    public LicenseService(Settings settings, ThreadPool threadPool, ClusterService clusterService, Clock clock, Environment env, ResourceWatcherService resourceWatcherService, XPackLicenseState licenseState) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.clock = clock;
        this.scheduler = new SchedulerEngine(settings, clock);
        this.licenseState = licenseState;
        this.allowedLicenseTypes = (List)ALLOWED_LICENSE_TYPES_SETTING.get(settings);
        this.operationModeFileWatcher = new OperationModeFileWatcher(resourceWatcherService, XPackPlugin.resolveConfigFile(env, "license_mode"), logger, () -> this.updateLicenseState(this.getLicensesMetadata()));
        this.scheduler.register(this);
        this.populateExpirationCallbacks();
        threadPool.scheduleWithFixedDelay(licenseState::cleanupUsageTracking, TimeValue.timeValueHours((long)1L), "generic");
    }

    private void logExpirationWarning(long expirationMillis, boolean expired) {
        logger.warn("{}", (Object)LicenseService.buildExpirationMessage(expirationMillis, expired));
    }

    static CharSequence buildExpirationMessage(long expirationMillis, boolean expired) {
        String expiredMsg = expired ? "expired" : "will expire";
        String general = LoggerMessageFormat.format(null, (String)"License [{}] on [{}].\n# If you have a new license, please update it. Otherwise, please reach out to\n# your support contact.\n# ", (Object[])new Object[]{expiredMsg, DATE_FORMATTER.formatMillis(expirationMillis)});
        if (expired) {
            general = general.toUpperCase(Locale.ROOT);
        }
        StringBuilder builder = new StringBuilder(general);
        builder.append(System.lineSeparator());
        if (expired) {
            builder.append("# COMMERCIAL PLUGINS OPERATING WITH REDUCED FUNCTIONALITY");
        } else {
            builder.append("# Commercial plugins operate with reduced functionality on license expiration:");
        }
        XPackLicenseState.EXPIRATION_MESSAGES.forEach((feature, messages) -> {
            if (((String[])messages).length > 0) {
                builder.append(System.lineSeparator());
                builder.append("# - ");
                builder.append((String)feature);
                for (String message : messages) {
                    builder.append(System.lineSeparator());
                    builder.append("#  - ");
                    builder.append(message);
                }
            }
        });
        return builder;
    }

    private void populateExpirationCallbacks() {
        this.expirationCallbacks.add(new ExpirationCallback.Pre(LicenseService.days(0), LicenseService.days(25), LicenseService.days(1)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(LicenseService.getExpiryDate(license), false);
            }
        });
        this.expirationCallbacks.add(new ExpirationCallback.Post(LicenseService.days(0), null, TimeValue.timeValueMinutes((long)10L)){

            @Override
            public void on(License license) {
                LicenseService.this.logExpirationWarning(LicenseService.getExpiryDate(license), true);
            }
        });
    }

    public static long getExpiryDate(License license) {
        String licenseUidHash = MessageDigests.toHexString((byte[])MessageDigests.sha256().digest(license.uid().getBytes(StandardCharsets.UTF_8)));
        return LicenseOverrides.overrideDateForLicense(licenseUidHash).map(date -> date.toInstant().toEpochMilli()).orElse(license.expiryDate());
    }

    public static LicenseStatus status(License license) {
        long now = System.currentTimeMillis();
        if (license.issueDate() > now) {
            return LicenseStatus.INVALID;
        }
        if (LicenseService.getExpiryDate(license) < now) {
            return LicenseStatus.EXPIRED;
        }
        return LicenseStatus.ACTIVE;
    }

    public void registerLicense(PutLicenseRequest request, ActionListener<PutLicenseResponse> listener) {
        License.LicenseType licenseType;
        final License newLicense = request.license();
        long now = this.clock.millis();
        if (!LicenseVerifier.verifyLicense(newLicense) || newLicense.issueDate() > now || newLicense.startDate() > now) {
            listener.onResponse((Object)new PutLicenseResponse(true, LicensesStatus.INVALID));
            return;
        }
        try {
            licenseType = License.LicenseType.resolve(newLicense);
        }
        catch (Exception e) {
            listener.onFailure(e);
            return;
        }
        if (licenseType == License.LicenseType.BASIC) {
            listener.onFailure((Exception)new IllegalArgumentException("Registering basic licenses is not allowed."));
        } else if (!this.isAllowedLicenseType(licenseType)) {
            listener.onFailure((Exception)new IllegalArgumentException("Registering [" + licenseType.getTypeName() + "] licenses is not allowed on this cluster"));
        } else if (LicenseService.getExpiryDate(newLicense) < now) {
            listener.onResponse((Object)new PutLicenseResponse(true, LicensesStatus.EXPIRED));
        } else {
            Map<String, String[]> acknowledgeMessages;
            License currentLicense;
            if (!request.acknowledged() && (currentLicense = this.getLicense()) != null && !(acknowledgeMessages = LicenseService.getAckMessages(newLicense, currentLicense)).isEmpty()) {
                listener.onResponse((Object)new PutLicenseResponse(false, LicensesStatus.VALID, ACKNOWLEDGEMENT_HEADER, acknowledgeMessages));
                return;
            }
            if (((Boolean)XPackSettings.SECURITY_ENABLED.get(this.settings)).booleanValue() && ((Boolean)XPackSettings.FIPS_MODE_ENABLED.get(this.settings)).booleanValue() && !XPackLicenseState.isFipsAllowedForOperationMode(newLicense.operationMode())) {
                throw new IllegalStateException("Cannot install a [" + newLicense.operationMode() + "] license unless FIPS mode is disabled");
            }
            this.submitUnbatchedTask("register license [" + newLicense.uid() + "]", (ClusterStateUpdateTask)new AckedClusterStateUpdateTask((AckedRequest)request, listener){

                protected PutLicenseResponse newResponse(boolean acknowledged) {
                    return new PutLicenseResponse(acknowledged, LicensesStatus.VALID);
                }

                public ClusterState execute(ClusterState currentState) throws Exception {
                    XPackPlugin.checkReadyForXPackCustomMetadata(currentState);
                    Version oldestNodeVersion = currentState.nodes().getSmallestNonClientNodeVersion();
                    if (!LicenseService.licenseIsCompatible(newLicense, oldestNodeVersion)) {
                        throw new IllegalStateException("The provided license is not compatible with node version [" + oldestNodeVersion + "]");
                    }
                    Metadata currentMetadata = currentState.metadata();
                    LicensesMetadata licensesMetadata = (LicensesMetadata)currentMetadata.custom("licenses");
                    Version trialVersion = null;
                    if (licensesMetadata != null) {
                        trialVersion = licensesMetadata.getMostRecentTrialVersion();
                    }
                    Metadata.Builder mdBuilder = Metadata.builder((Metadata)currentMetadata);
                    mdBuilder.putCustom("licenses", (Metadata.Custom)new LicensesMetadata(newLicense, trialVersion));
                    return ClusterState.builder((ClusterState)currentState).metadata(mdBuilder).build();
                }
            });
        }
    }

    @SuppressForbidden(reason="legacy usage of unbatched task")
    private void submitUnbatchedTask(String source, ClusterStateUpdateTask task) {
        this.clusterService.submitUnbatchedStateUpdateTask(source, task);
    }

    private static boolean licenseIsCompatible(License license, Version version) {
        int maxVersion = LicenseUtils.getMaxLicenseVersion(version);
        return license.version() <= maxVersion;
    }

    private boolean isAllowedLicenseType(License.LicenseType type) {
        logger.debug("Checking license [{}] against allowed license types: {}", (Object)type, this.allowedLicenseTypes);
        return this.allowedLicenseTypes.contains((Object)type);
    }

    public static Map<String, String[]> getAckMessages(License newLicense, License currentLicense) {
        HashMap<String, String[]> acknowledgeMessages = new HashMap<String, String[]>();
        if (!License.isAutoGeneratedLicense(currentLicense.signature()) && currentLicense.issueDate() > newLicense.issueDate()) {
            acknowledgeMessages.put("license", new String[]{"The new license is older than the currently installed license. Are you sure you want to override the current license?"});
        }
        XPackLicenseState.ACKNOWLEDGMENT_MESSAGES.forEach((feature, ackMessages) -> {
            String[] messages = (String[])ackMessages.apply(currentLicense.operationMode(), newLicense.operationMode());
            if (messages.length > 0) {
                acknowledgeMessages.put((String)feature, messages);
            }
        });
        return acknowledgeMessages;
    }

    private static TimeValue days(int days) {
        return TimeValue.timeValueHours((long)(days * 24));
    }

    @Override
    public void triggered(SchedulerEngine.Event event) {
        LicensesMetadata licensesMetadata = this.getLicensesMetadata();
        if (licensesMetadata != null) {
            License license = licensesMetadata.getLicense();
            if (event.getJobName().equals(LICENSE_JOB)) {
                this.updateLicenseState(license);
            } else if (event.getJobName().startsWith(".license_expiration_job_")) {
                this.expirationCallbacks.stream().filter(expirationCallback -> expirationCallback.getId().equals(event.getJobName())).forEach(expirationCallback -> expirationCallback.on(license));
            }
        }
    }

    public void removeLicense(ActionListener<PostStartBasicResponse> listener) {
        PostStartBasicRequest startBasicRequest = new PostStartBasicRequest().acknowledge(true);
        StartBasicClusterTask task = new StartBasicClusterTask(logger, this.clusterService.getClusterName().value(), this.clock, startBasicRequest, "delete license", listener);
        this.clusterService.submitStateUpdateTask(task.getDescription(), (ClusterStateTaskListener)task, ClusterStateTaskConfig.build((Priority)Priority.NORMAL), (ClusterStateTaskExecutor)this.startBasicExecutor);
    }

    public License getLicense() {
        License license = LicenseService.getLicense(this.clusterService.state().metadata());
        return license == LicensesMetadata.LICENSE_TOMBSTONE ? null : license;
    }

    private LicensesMetadata getLicensesMetadata() {
        return (LicensesMetadata)this.clusterService.state().metadata().custom("licenses");
    }

    void startTrialLicense(PostStartTrialRequest request, ActionListener<PostStartTrialResponse> listener) {
        License.LicenseType requestedType = License.LicenseType.parse(request.getType());
        if (!VALID_TRIAL_TYPES.contains((Object)requestedType)) {
            throw new IllegalArgumentException("Cannot start trial of type [" + requestedType.getTypeName() + "]. Valid trial types are [" + VALID_TRIAL_TYPES.stream().map(License.LicenseType::getTypeName).sorted().collect(Collectors.joining(",")) + "]");
        }
        this.clusterService.submitStateUpdateTask("started trial license", (ClusterStateTaskListener)new StartTrialClusterTask(logger, this.clusterService.getClusterName().value(), this.clock, request, listener), ClusterStateTaskConfig.build((Priority)Priority.NORMAL), (ClusterStateTaskExecutor)this.startTrialExecutor);
    }

    void startBasicLicense(PostStartBasicRequest request, ActionListener<PostStartBasicResponse> listener) {
        StartBasicClusterTask task = new StartBasicClusterTask(logger, this.clusterService.getClusterName().value(), this.clock, request, "start basic license", listener);
        this.clusterService.submitStateUpdateTask(task.getDescription(), (ClusterStateTaskListener)task, ClusterStateTaskConfig.build((Priority)Priority.NORMAL), (ClusterStateTaskExecutor)this.startBasicExecutor);
    }

    private void registerOrUpdateSelfGeneratedLicense() {
        this.submitUnbatchedTask("maybe generate license for cluster", new StartupSelfGeneratedLicenseTask(this.settings, this.clock, this.clusterService));
    }

    protected void doStart() throws ElasticsearchException {
        ClusterState clusterState;
        this.clusterService.addListener((ClusterStateListener)this);
        this.scheduler.start(Collections.emptyList());
        logger.debug("initializing license state");
        if (this.clusterService.lifecycleState() == Lifecycle.State.STARTED && !(clusterState = this.clusterService.state()).blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) && clusterState.nodes().getMasterNode() != null && XPackPlugin.isReadyForXPackCustomMetadata(clusterState)) {
            boolean noLicense;
            LicensesMetadata currentMetadata = (LicensesMetadata)clusterState.metadata().custom("licenses");
            boolean bl = noLicense = currentMetadata == null || currentMetadata.getLicense() == null;
            if (clusterState.getNodes().isLocalNodeElectedMaster() && (noLicense || LicenseUtils.licenseNeedsExtended(currentMetadata.getLicense()))) {
                this.registerOrUpdateSelfGeneratedLicense();
            }
        }
    }

    protected void doStop() throws ElasticsearchException {
        this.clusterService.removeListener((ClusterStateListener)this);
        this.scheduler.stop();
        this.currentLicenseHolder.set(null);
    }

    protected void doClose() throws ElasticsearchException {
    }

    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState previousClusterState = event.previousState();
        ClusterState currentClusterState = event.state();
        if (!currentClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            boolean noLicense;
            boolean noLicenseInCurrentMetadata;
            boolean noLicenseInPrevMetadata;
            if (!XPackPlugin.isReadyForXPackCustomMetadata(currentClusterState)) {
                logger.debug("cannot add license to cluster as the following nodes might not understand the license metadata: {}", new Supplier[]{() -> XPackPlugin.nodesNotReadyForXPackCustomMetadata(currentClusterState)});
                return;
            }
            LicensesMetadata prevLicensesMetadata = (LicensesMetadata)previousClusterState.getMetadata().custom("licenses");
            LicensesMetadata currentLicensesMetadata = (LicensesMetadata)currentClusterState.getMetadata().custom("licenses");
            if (previousClusterState.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK) || prevLicensesMetadata == null) {
                if (currentLicensesMetadata != null) {
                    logger.debug("state recovered: previous license [{}]", (Object)prevLicensesMetadata);
                    logger.debug("state recovered: current license [{}]", (Object)currentLicensesMetadata);
                    this.onUpdate(currentLicensesMetadata);
                } else {
                    logger.trace("state recovered: no current license");
                }
            } else if (!prevLicensesMetadata.equals((Object)currentLicensesMetadata)) {
                logger.debug("previous [{}]", (Object)prevLicensesMetadata);
                logger.debug("current [{}]", (Object)currentLicensesMetadata);
                this.onUpdate(currentLicensesMetadata);
            } else {
                logger.trace("license unchanged [{}]", (Object)currentLicensesMetadata);
            }
            License currentLicense = null;
            boolean bl = noLicenseInPrevMetadata = prevLicensesMetadata == null || prevLicensesMetadata.getLicense() == null;
            if (!noLicenseInPrevMetadata) {
                currentLicense = prevLicensesMetadata.getLicense();
            }
            boolean bl2 = noLicenseInCurrentMetadata = currentLicensesMetadata == null || currentLicensesMetadata.getLicense() == null;
            if (!noLicenseInCurrentMetadata) {
                currentLicense = currentLicensesMetadata.getLicense();
            }
            boolean bl3 = noLicense = noLicenseInPrevMetadata && noLicenseInCurrentMetadata;
            if (currentClusterState.getNodes().isLocalNodeElectedMaster() && (noLicense || LicenseUtils.licenseNeedsExtended(currentLicense) || LicenseUtils.signatureNeedsUpdate(currentLicense, currentClusterState.nodes()))) {
                this.registerOrUpdateSelfGeneratedLicense();
            }
        } else if (logger.isDebugEnabled()) {
            logger.debug("skipped license notifications reason: [{}]", (Object)GatewayService.STATE_NOT_RECOVERED_BLOCK);
        }
    }

    private void updateLicenseState(LicensesMetadata licensesMetadata) {
        if (licensesMetadata != null) {
            this.updateLicenseState(LicenseService.getLicense(licensesMetadata));
        }
    }

    protected static String getExpiryWarning(long licenseExpiryDate, long currentTime) {
        long diff = licenseExpiryDate - currentTime;
        if (LICENSE_EXPIRATION_WARNING_PERIOD.getMillis() > diff) {
            long days = TimeUnit.MILLISECONDS.toDays(diff);
            String expiryMessage = days == 0L && diff > 0L ? "expires today" : (diff > 0L ? String.format(Locale.ROOT, "will expire in [%d] days", days) : String.format(Locale.ROOT, "expired on [%s]", DATE_FORMATTER.formatMillis(licenseExpiryDate)));
            return "Your license " + expiryMessage + ". Contact your administrator or update your license for continued use of features";
        }
        return null;
    }

    protected void updateLicenseState(License license) {
        long time = this.clock.millis();
        if (license == LicensesMetadata.LICENSE_TOMBSTONE) {
            this.licenseState.update(License.OperationMode.MISSING, false, LicenseService.getExpiryWarning(LicenseService.getExpiryDate(license), time));
            return;
        }
        if (license != null) {
            boolean active = LicenseService.getExpiryDate(license) == BASIC_SELF_GENERATED_LICENSE_EXPIRATION_MILLIS ? true : time >= license.issueDate() && time < LicenseService.getExpiryDate(license);
            this.licenseState.update(license.operationMode(), active, LicenseService.getExpiryWarning(LicenseService.getExpiryDate(license), time));
            if (active) {
                logger.debug("license [{}] - valid", (Object)license.uid());
            } else {
                logger.warn("license [{}] - expired", (Object)license.uid());
            }
        }
    }

    private void onUpdate(LicensesMetadata currentLicensesMetadata) {
        License license = LicenseService.getLicense(currentLicensesMetadata);
        if (license != null) {
            License previousLicense = this.currentLicenseHolder.get();
            if (!license.equals(previousLicense)) {
                this.currentLicenseHolder.set(license);
                license.setOperationModeFileWatcher(this.operationModeFileWatcher);
                this.scheduler.add(new SchedulerEngine.Job(LICENSE_JOB, LicenseService.nextLicenseCheck(license)));
                for (ExpirationCallback expirationCallback : this.expirationCallbacks) {
                    this.scheduler.add(new SchedulerEngine.Job(expirationCallback.getId(), (startTime, now) -> expirationCallback.nextScheduledTimeForExpiry(LicenseService.getExpiryDate(license), startTime, now)));
                }
                if (previousLicense != null) {
                    previousLicense.removeOperationModeFileWatcher();
                }
                logger.info("license [{}] mode [{}] - valid", (Object)license.uid(), (Object)license.operationMode().name().toLowerCase(Locale.ROOT));
            }
            this.updateLicenseState(license);
        }
    }

    static SchedulerEngine.Schedule nextLicenseCheck(License license) {
        return (startTime, time) -> {
            if (time < license.issueDate()) {
                return license.issueDate();
            }
            if (time < LicenseService.getExpiryDate(license)) {
                long nextTime;
                for (nextTime = LicenseService.getExpiryDate(license) - LICENSE_EXPIRATION_WARNING_PERIOD.getMillis(); nextTime <= time; nextTime += TimeValue.timeValueDays((long)1L).getMillis()) {
                }
                return nextTime;
            }
            return -1L;
        };
    }

    public static License getLicense(Metadata metadata) {
        LicensesMetadata licensesMetadata = (LicensesMetadata)metadata.custom("licenses");
        return LicenseService.getLicense(licensesMetadata);
    }

    static License getLicense(@Nullable LicensesMetadata metadata) {
        if (metadata != null) {
            License license = metadata.getLicense();
            if (license == LicensesMetadata.LICENSE_TOMBSTONE) {
                return license;
            }
            if (license != null && license.verified()) {
                return license;
            }
        }
        return null;
    }

    private static List<License.LicenseType> getAllowableUploadTypes() {
        return Stream.of(License.LicenseType.values()).filter(t -> t != License.LicenseType.BASIC).toList();
    }

    private static void validateUploadTypesSetting(List<License.LicenseType> value) {
        if (!ALLOWABLE_UPLOAD_TYPES.containsAll(value)) {
            throw new IllegalArgumentException("Invalid value [" + value.stream().map(License.LicenseType::getTypeName).collect(Collectors.joining(",")) + "] for " + ALLOWED_LICENSE_TYPES_SETTING.getKey() + ", allowed values are [" + ALLOWABLE_UPLOAD_TYPES.stream().map(License.LicenseType::getTypeName).collect(Collectors.joining(",")) + "]");
        }
    }
}

