/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.security.authc.jwt;

import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.crypto.MACSigner;
import com.nimbusds.jose.jwk.Curve;
import com.nimbusds.jose.jwk.ECKey;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.KeyOperation;
import com.nimbusds.jose.jwk.KeyUse;
import com.nimbusds.jose.jwk.OctetSequenceKey;
import com.nimbusds.jose.jwk.RSAKey;
import java.nio.charset.StandardCharsets;
import java.security.PublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.SettingsException;
import org.elasticsearch.xpack.core.security.authc.jwt.JwtRealmSettings;
import org.elasticsearch.xpack.security.authc.jwt.JwkSetLoader;

public class JwkValidateUtil {
    private static final Logger LOGGER = LogManager.getLogger(JwkValidateUtil.class);

    static JwkSetLoader.JwksAlgs filterJwksAndAlgorithms(List<JWK> jwks, List<String> algs) throws SettingsException {
        LOGGER.trace("JWKs [" + jwks.size() + "] and Algorithms [" + String.join((CharSequence)",", algs) + "] before filters.");
        Predicate<JWK> keyUsePredicate = j -> j.getKeyUse() == null || KeyUse.SIGNATURE.equals((Object)j.getKeyUse());
        List<JWK> jwksSig = jwks.stream().filter(keyUsePredicate).toList();
        LOGGER.trace("JWKs [" + jwksSig.size() + "] after KeyUse [SIGNATURE||null] filter.");
        Predicate<JWK> keyOpPredicate = j -> j.getKeyOperations() == null || j.getKeyOperations().contains(KeyOperation.VERIFY);
        List<JWK> jwksVerify = jwksSig.stream().filter(keyOpPredicate).toList();
        LOGGER.trace("JWKs [" + jwksVerify.size() + " after KeyOperation [VERIFY||null] filter.");
        List<JWK> jwksFiltered = jwksVerify.stream().filter(j -> algs.stream().anyMatch(a -> JwkValidateUtil.isMatch(j, a))).toList();
        LOGGER.trace("JWKs [" + jwksFiltered.size() + "] after Algorithms [" + String.join((CharSequence)",", algs) + "] filter.");
        List<String> algsFiltered = algs.stream().filter(a -> jwksFiltered.stream().anyMatch(j -> JwkValidateUtil.isMatch(j, a))).toList();
        LOGGER.trace("Algorithms [" + String.join((CharSequence)",", algsFiltered) + "] after remaining JWKs [" + jwksFiltered.size() + "] filter.");
        return new JwkSetLoader.JwksAlgs(jwksFiltered, algsFiltered);
    }

    static boolean isMatch(JWK jwk, String algorithm) {
        try {
            if (JwtRealmSettings.SUPPORTED_SIGNATURE_ALGORITHMS_HMAC.contains(algorithm) && jwk instanceof OctetSequenceKey) {
                int min;
                boolean isMatch;
                OctetSequenceKey jwkHmac = (OctetSequenceKey)jwk;
                int bits = jwkHmac.size();
                boolean bl = isMatch = bits >= (min = MACSigner.getMinRequiredSecretLength((JWSAlgorithm)JWSAlgorithm.parse((String)algorithm)));
                if (!isMatch) {
                    LOGGER.trace("HMAC JWK [" + bits + "] bits too small for algorithm [" + algorithm + "] minimum [" + min + "].");
                }
                return isMatch;
            }
            if (JwtRealmSettings.SUPPORTED_SIGNATURE_ALGORITHMS_RSA.contains(algorithm) && jwk instanceof RSAKey) {
                boolean isMatch;
                RSAKey jwkRsa = (RSAKey)jwk;
                int bits = JwkValidateUtil.computeBitLengthRsa(jwkRsa.toPublicKey());
                int min = 2048;
                boolean bl = isMatch = bits >= 2048;
                if (!isMatch) {
                    LOGGER.trace("RSA JWK [" + bits + "] bits too small for algorithm [" + algorithm + "] minimum [2048].");
                }
                return isMatch;
            }
            if (JwtRealmSettings.SUPPORTED_SIGNATURE_ALGORITHMS_EC.contains(algorithm) && jwk instanceof ECKey) {
                ECKey jwkEc = (ECKey)jwk;
                Curve curve = jwkEc.getCurve();
                Set allowed = Curve.forJWSAlgorithm((JWSAlgorithm)JWSAlgorithm.parse((String)algorithm));
                boolean isMatch = allowed.contains(curve);
                if (!isMatch) {
                    LOGGER.trace("EC JWK [" + curve + "] curve not allowed for algorithm [" + algorithm + "] allowed " + allowed + ".");
                }
                return isMatch;
            }
        }
        catch (Exception e) {
            LOGGER.trace("Unexpected exception", (Throwable)e);
        }
        return false;
    }

    static int computeBitLengthRsa(PublicKey publicKey) throws Exception {
        if (publicKey instanceof RSAPublicKey) {
            RSAPublicKey rsaPublicKey = (RSAPublicKey)publicKey;
            int modulusBigLength = rsaPublicKey.getModulus().bitLength();
            return (modulusBigLength + 8 - 1) / 8 * 8;
        }
        if (publicKey == null) {
            throw new Exception("Expected public key class [RSAPublicKey]. Got [null] instead.");
        }
        throw new Exception("Expected public key class [RSAPublicKey]. Got [" + publicKey.getClass().getSimpleName() + "] instead.");
    }

    static List<JWK> loadJwksFromJwkSetString(String jwkSetConfigKey, CharSequence jwkSetContents) throws SettingsException {
        if (Strings.hasText((CharSequence)jwkSetContents)) {
            try {
                return JWKSet.parse((String)jwkSetContents.toString()).getKeys();
            }
            catch (Exception e) {
                throw new SettingsException("JWKSet parse failed for setting [" + jwkSetConfigKey + "]", (Throwable)e);
            }
        }
        return Collections.emptyList();
    }

    static OctetSequenceKey loadHmacJwkFromJwkString(String jwkSetConfigKey, CharSequence hmacKeyContents) {
        if (Strings.hasText((CharSequence)hmacKeyContents)) {
            try {
                return JwkValidateUtil.buildHmacKeyFromString(hmacKeyContents);
            }
            catch (Exception e) {
                throw new SettingsException("HMAC Key parse failed for setting [" + jwkSetConfigKey + "]", (Throwable)e);
            }
        }
        return null;
    }

    static OctetSequenceKey buildHmacKeyFromString(CharSequence hmacKeyContents) {
        String hmacKeyString = hmacKeyContents.toString();
        byte[] utf8Bytes = hmacKeyString.getBytes(StandardCharsets.UTF_8);
        return new OctetSequenceKey.Builder(utf8Bytes).build();
    }
}

