/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.health.node;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.Version;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.cluster.node.stats.NodeStats;
import org.elasticsearch.action.admin.indices.stats.CommonStatsFlags;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.common.util.concurrent.EsRejectedExecutionException;
import org.elasticsearch.common.util.concurrent.RunOnce;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.health.HealthStatus;
import org.elasticsearch.health.metadata.HealthMetadata;
import org.elasticsearch.health.node.DiskHealthInfo;
import org.elasticsearch.health.node.UpdateHealthInfoCacheAction;
import org.elasticsearch.health.node.action.HealthNodeNotDiscoveredException;
import org.elasticsearch.health.node.selection.HealthNode;
import org.elasticsearch.health.node.selection.HealthNodeTaskExecutor;
import org.elasticsearch.node.NodeService;
import org.elasticsearch.threadpool.Scheduler;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.NodeNotConnectedException;

public class LocalHealthMonitor
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(LocalHealthMonitor.class);
    public static final Setting<TimeValue> POLL_INTERVAL_SETTING = Setting.timeSetting("health.reporting.local.monitor.interval", TimeValue.timeValueSeconds((long)30L), TimeValue.timeValueSeconds((long)10L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final DiskCheck diskCheck;
    private final Client client;
    private volatile TimeValue monitorInterval;
    private volatile boolean enabled;
    private volatile boolean prerequisitesFulfilled;
    private final AtomicReference<DiskHealthInfo> lastReportedDiskHealthInfo = new AtomicReference();
    private final AtomicReference<String> lastSeenHealthNode = new AtomicReference();
    private volatile Monitoring monitoring;

    private LocalHealthMonitor(Settings settings, ClusterService clusterService, NodeService nodeService, ThreadPool threadPool, Client client) {
        this.threadPool = threadPool;
        this.monitorInterval = POLL_INTERVAL_SETTING.get(settings);
        this.enabled = HealthNodeTaskExecutor.ENABLED_SETTING.get(settings);
        this.clusterService = clusterService;
        this.client = client;
        this.diskCheck = new DiskCheck(nodeService);
    }

    public static LocalHealthMonitor create(Settings settings, ClusterService clusterService, NodeService nodeService, ThreadPool threadPool, Client client) {
        LocalHealthMonitor localHealthMonitor = new LocalHealthMonitor(settings, clusterService, nodeService, threadPool, client);
        localHealthMonitor.registerListeners();
        return localHealthMonitor;
    }

    private void registerListeners() {
        ClusterSettings clusterSettings = this.clusterService.getClusterSettings();
        clusterSettings.addSettingsUpdateConsumer(POLL_INTERVAL_SETTING, this::setMonitorInterval);
        clusterSettings.addSettingsUpdateConsumer(HealthNodeTaskExecutor.ENABLED_SETTING, this::setEnabled);
        this.clusterService.addListener(this);
    }

    void setMonitorInterval(TimeValue monitorInterval) {
        this.monitorInterval = monitorInterval;
        this.stopMonitoring();
        this.startMonitoringIfNecessary();
    }

    void setEnabled(boolean enabled) {
        this.enabled = enabled;
        if (enabled) {
            this.startMonitoringIfNecessary();
        } else {
            this.stopMonitoring();
        }
    }

    private void stopMonitoring() {
        Monitoring currentMonitoring = this.monitoring;
        if (currentMonitoring != null) {
            currentMonitoring.cancel();
        }
    }

    private void startMonitoringIfNecessary() {
        if (this.prerequisitesFulfilled && this.enabled) {
            if (!this.isMonitorRunning()) {
                this.monitoring = Monitoring.start(this.monitorInterval, this.threadPool, this.lastReportedDiskHealthInfo, this.lastSeenHealthNode, this.diskCheck, this.clusterService, this.client);
                logger.debug("Local health monitoring started {}", (Object)this.monitoring);
            } else {
                logger.trace("Local health monitoring already started {}, skipping", (Object)this.monitoring);
            }
        }
    }

    private boolean isMonitorRunning() {
        Monitoring scheduled = this.monitoring;
        return scheduled != null && !scheduled.isCancelled();
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        DiscoveryNode currentHealthNode = HealthNode.findHealthNode(event.state());
        DiscoveryNode currentMasterNode = event.state().nodes().getMasterNode();
        boolean healthNodeChanged = this.hasHealthNodeChanged(currentHealthNode, event);
        boolean masterNodeChanged = this.hasMasterNodeChanged(currentMasterNode, event);
        if (healthNodeChanged || masterNodeChanged) {
            this.lastSeenHealthNode.set(currentHealthNode == null ? null : currentHealthNode.getId());
            this.lastReportedDiskHealthInfo.set(null);
            if (logger.isDebugEnabled()) {
                String reason = healthNodeChanged && masterNodeChanged ? "the master node and the health node" : (healthNodeChanged ? "the health node" : "the master node");
                logger.debug("Resetting the health monitoring because {} changed, current health node is {}.", (Object)reason, currentHealthNode == null ? null : Strings.format((String)"[%s][%s]", (Object[])new Object[]{currentHealthNode.getName(), currentHealthNode.getId()}));
            }
        }
        boolean bl = this.prerequisitesFulfilled = event.state().nodesIfRecovered().getMinNodeVersion().onOrAfter(Version.V_8_5_0) && HealthMetadata.getFromClusterState(event.state()) != null && currentHealthNode != null && currentMasterNode != null;
        if (!this.prerequisitesFulfilled || healthNodeChanged || masterNodeChanged) {
            this.stopMonitoring();
        }
        if (this.prerequisitesFulfilled) {
            this.startMonitoringIfNecessary();
        }
    }

    private boolean hasMasterNodeChanged(DiscoveryNode currentMasterNode, ClusterChangedEvent event) {
        DiscoveryNode previousMasterNode = event.previousState().nodes().getMasterNode();
        if (currentMasterNode == null || previousMasterNode == null) {
            return currentMasterNode != previousMasterNode;
        }
        return !previousMasterNode.getEphemeralId().equals(currentMasterNode.getEphemeralId());
    }

    private boolean hasHealthNodeChanged(DiscoveryNode currentHealthNode, ClusterChangedEvent event) {
        DiscoveryNode previousHealthNode = HealthNode.findHealthNode(event.previousState());
        return !Objects.equals(this.lastSeenHealthNode.get(), currentHealthNode == null ? null : currentHealthNode.getId()) || !Objects.equals(previousHealthNode, currentHealthNode);
    }

    @Nullable
    DiskHealthInfo getLastReportedDiskHealthInfo() {
        return this.lastReportedDiskHealthInfo.get();
    }

    static class DiskCheck {
        private final NodeService nodeService;

        DiskCheck(NodeService nodeService) {
            this.nodeService = nodeService;
        }

        DiskHealthInfo getHealth(HealthMetadata healthMetadata, ClusterState clusterState) {
            DiscoveryNode node = clusterState.getNodes().getLocalNode();
            HealthMetadata.Disk diskMetadata = healthMetadata.getDiskMetadata();
            DiskUsage usage = this.getDiskUsage();
            if (usage == null) {
                return new DiskHealthInfo(HealthStatus.UNKNOWN, DiskHealthInfo.Cause.NODE_HAS_NO_DISK_STATS);
            }
            ByteSizeValue totalBytes = ByteSizeValue.ofBytes(usage.getTotalBytes());
            if (node.isDedicatedFrozenNode()) {
                long frozenFloodStageThreshold = diskMetadata.getFreeBytesFrozenFloodStageWatermark(totalBytes).getBytes();
                if (usage.getFreeBytes() < frozenFloodStageThreshold) {
                    logger.debug("Flood stage disk watermark [{}] exceeded on {}", (Object)frozenFloodStageThreshold, (Object)usage);
                    return new DiskHealthInfo(HealthStatus.RED, DiskHealthInfo.Cause.FROZEN_NODE_OVER_FLOOD_STAGE_THRESHOLD);
                }
                return new DiskHealthInfo(HealthStatus.GREEN);
            }
            long floodStageThreshold = diskMetadata.getFreeBytesFloodStageWatermark(totalBytes).getBytes();
            if (usage.getFreeBytes() < floodStageThreshold) {
                logger.debug("Flood stage disk watermark [{}] exceeded on {}", (Object)floodStageThreshold, (Object)usage);
                return new DiskHealthInfo(HealthStatus.RED, DiskHealthInfo.Cause.NODE_OVER_THE_FLOOD_STAGE_THRESHOLD);
            }
            long highThreshold = diskMetadata.getFreeBytesHighWatermark(totalBytes).getBytes();
            if (usage.getFreeBytes() < highThreshold) {
                if (node.canContainData()) {
                    if (!DiskCheck.hasRelocatingShards(clusterState, node)) {
                        logger.debug("High disk watermark [{}] exceeded on {}", (Object)highThreshold, (Object)usage);
                        return new DiskHealthInfo(HealthStatus.YELLOW, DiskHealthInfo.Cause.NODE_OVER_HIGH_THRESHOLD);
                    }
                } else {
                    logger.debug("High disk watermark [{}] exceeded on {}", (Object)highThreshold, (Object)usage);
                    return new DiskHealthInfo(HealthStatus.YELLOW, DiskHealthInfo.Cause.NODE_OVER_HIGH_THRESHOLD);
                }
            }
            return new DiskHealthInfo(HealthStatus.GREEN);
        }

        private DiskUsage getDiskUsage() {
            NodeStats nodeStats = this.nodeService.stats(CommonStatsFlags.NONE, false, false, false, false, true, false, false, false, false, false, false, false, false, false);
            return DiskUsage.findLeastAvailablePath(nodeStats);
        }

        static boolean hasRelocatingShards(ClusterState clusterState, DiscoveryNode node) {
            RoutingNode routingNode = clusterState.getRoutingNodes().node(node.getId());
            if (routingNode == null) {
                return false;
            }
            return routingNode.numberOfShardsWithState(ShardRoutingState.RELOCATING) > 0;
        }
    }

    static class Monitoring
    implements Runnable,
    Scheduler.Cancellable {
        private final TimeValue interval;
        private final String executor;
        private final Scheduler scheduler;
        private final ClusterService clusterService;
        private final DiskCheck diskCheck;
        private final Client client;
        private final AtomicReference<DiskHealthInfo> lastReportedDiskHealthInfo;
        private final AtomicReference<String> lastSeenHealthNode;
        private volatile boolean cancelled = false;
        private volatile Scheduler.ScheduledCancellable scheduledRun;

        private Monitoring(TimeValue interval, Scheduler scheduler, String executor, AtomicReference<DiskHealthInfo> lastReportedDiskHealthInfo, AtomicReference<String> lastSeenHealthNode, DiskCheck diskCheck, ClusterService clusterService, Client client) {
            this.interval = interval;
            this.executor = executor;
            this.scheduler = scheduler;
            this.lastReportedDiskHealthInfo = lastReportedDiskHealthInfo;
            this.lastSeenHealthNode = lastSeenHealthNode;
            this.clusterService = clusterService;
            this.diskCheck = diskCheck;
            this.client = client;
        }

        static Monitoring start(TimeValue interval, Scheduler scheduler, AtomicReference<DiskHealthInfo> lastReportedDiskHealthInfo, AtomicReference<String> lastSeenHealthNode, DiskCheck diskCheck, ClusterService clusterService, Client client) {
            Monitoring monitoring = new Monitoring(interval, scheduler, "management", lastReportedDiskHealthInfo, lastSeenHealthNode, diskCheck, clusterService, client);
            monitoring.scheduledRun = scheduler.schedule(monitoring, TimeValue.ZERO, monitoring.executor);
            return monitoring;
        }

        @Override
        public boolean cancel() {
            if (this.cancelled) {
                return false;
            }
            this.cancelled = true;
            this.scheduledRun.cancel();
            return true;
        }

        @Override
        public boolean isCancelled() {
            return this.cancelled;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (this.cancelled) {
                return;
            }
            boolean nextRunScheduled = false;
            RunOnce scheduleNextRun = new RunOnce(this::scheduleNextRunIfNecessary);
            try {
                ClusterState clusterState = this.clusterService.state();
                HealthMetadata healthMetadata = HealthMetadata.getFromClusterState(clusterState);
                if (healthMetadata != null) {
                    DiskHealthInfo previousHealth = this.lastReportedDiskHealthInfo.get();
                    DiskHealthInfo currentHealth = this.diskCheck.getHealth(healthMetadata, clusterState);
                    if (!currentHealth.equals(previousHealth)) {
                        String nodeId = this.clusterService.localNode().getId();
                        String healthNodeId = this.lastSeenHealthNode.get();
                        ActionListener listener = ActionListener.wrap(response -> {
                            if (Objects.equals(healthNodeId, this.lastSeenHealthNode.get()) && this.lastReportedDiskHealthInfo.compareAndSet(previousHealth, currentHealth)) {
                                logger.debug("Health info [{}] successfully sent, last reported value: {}.", (Object)currentHealth, (Object)this.lastReportedDiskHealthInfo.get());
                            }
                        }, e -> {
                            if (e.getCause() instanceof NodeNotConnectedException || e.getCause() instanceof HealthNodeNotDiscoveredException) {
                                logger.debug("Failed to connect to the health node [{}], will try again.", (Object)e.getCause().getMessage());
                            } else {
                                logger.debug(() -> Strings.format((String)"Failed to send health info [%s] to health node, will try again.", (Object[])new Object[]{currentHealth}), (Throwable)e);
                            }
                        });
                        this.client.execute(UpdateHealthInfoCacheAction.INSTANCE, new UpdateHealthInfoCacheAction.Request(nodeId, currentHealth), ActionListener.runAfter(listener, scheduleNextRun));
                        nextRunScheduled = true;
                    }
                }
            }
            catch (Exception e2) {
                logger.warn(() -> Strings.format((String)"Failed to run scheduled health monitoring on thread pool [%s]", (Object[])new Object[]{this.executor}), (Throwable)e2);
            }
            finally {
                if (!nextRunScheduled) {
                    scheduleNextRun.run();
                }
            }
        }

        private void scheduleNextRunIfNecessary() {
            if (this.cancelled) {
                return;
            }
            try {
                this.scheduledRun = this.scheduler.schedule(this, this.interval, this.executor);
            }
            catch (EsRejectedExecutionException e) {
                logger.debug(() -> Strings.format((String)"Scheduled health monitoring was rejected on thread pool [%s]", (Object[])new Object[]{this.executor}), (Throwable)e);
            }
        }

        public String toString() {
            return "Monitoring{interval=" + this.interval + ", cancelled=" + this.cancelled + "}";
        }
    }
}

