/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.reservedstate.service;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.support.PlainActionFuture;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.ReservedStateMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Randomness;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.env.Environment;
import org.elasticsearch.reservedstate.service.FileSettingsChangedListener;
import org.elasticsearch.reservedstate.service.ReservedClusterStateService;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.XContentParserConfiguration;
import org.elasticsearch.xcontent.XContentType;

public class FileSettingsService
extends AbstractLifecycleComponent
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(FileSettingsService.class);
    public static final String SETTINGS_FILE_NAME = "settings.json";
    public static final String NAMESPACE = "file_settings";
    private static final int REGISTER_RETRY_COUNT = 5;
    private final ClusterService clusterService;
    private final ReservedClusterStateService stateService;
    private final Path operatorSettingsDir;
    private WatchService watchService;
    private Thread watcherThread;
    private FileUpdateState fileUpdateState;
    private WatchKey settingsDirWatchKey;
    private WatchKey configDirWatchKey;
    private volatile boolean active = false;
    public static final String OPERATOR_DIRECTORY = "operator";
    private final List<FileSettingsChangedListener> eventListeners;

    public FileSettingsService(ClusterService clusterService, ReservedClusterStateService stateService, Environment environment) {
        this.clusterService = clusterService;
        this.stateService = stateService;
        this.operatorSettingsDir = environment.configFile().toAbsolutePath().resolve(OPERATOR_DIRECTORY);
        this.eventListeners = new CopyOnWriteArrayList<FileSettingsChangedListener>();
    }

    public Path operatorSettingsDir() {
        return this.operatorSettingsDir;
    }

    public Path operatorSettingsFile() {
        return this.operatorSettingsDir.resolve(SETTINGS_FILE_NAME);
    }

    boolean watchedFileChanged(Path path) throws IOException {
        if (!Files.exists(path, new LinkOption[0])) {
            return false;
        }
        FileUpdateState previousUpdateState = this.fileUpdateState;
        BasicFileAttributes attr = Files.readAttributes(path, BasicFileAttributes.class, new LinkOption[0]);
        this.fileUpdateState = new FileUpdateState(attr.lastModifiedTime().toMillis(), path.toRealPath(new LinkOption[0]).toString(), attr.fileKey());
        return previousUpdateState == null || !previousUpdateState.equals(this.fileUpdateState);
    }

    @Override
    protected void doStart() {
        this.active = Files.exists(this.operatorSettingsDir().getParent(), new LinkOption[0]);
        if (!this.active) {
            return;
        }
        if (DiscoveryNode.isMasterNode(this.clusterService.getSettings())) {
            this.clusterService.addListener(this);
        }
    }

    @Override
    protected void doStop() {
        this.active = false;
        logger.debug("Stopping file settings service");
        this.stopWatcher();
    }

    @Override
    protected void doClose() {
    }

    private boolean currentNodeMaster(ClusterState clusterState) {
        return clusterState.nodes().getLocalNodeId().equals(clusterState.nodes().getMasterNodeId());
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState clusterState = event.state();
        this.startIfMaster(clusterState);
    }

    private void startIfMaster(ClusterState clusterState) {
        if (this.currentNodeMaster(clusterState)) {
            this.startWatcher(clusterState);
        } else {
            this.stopWatcher();
        }
    }

    public void handleSnapshotRestore(ClusterState clusterState, Metadata.Builder mdBuilder) {
        assert (this.currentNodeMaster(clusterState));
        ReservedStateMetadata fileSettingsMetadata = clusterState.metadata().reservedStateMetadata().get(NAMESPACE);
        if (this.watching() && Files.exists(this.operatorSettingsFile(), new LinkOption[0])) {
            if (fileSettingsMetadata != null) {
                ReservedStateMetadata withResetVersion = new ReservedStateMetadata.Builder(fileSettingsMetadata).version(0L).build();
                mdBuilder.put(withResetVersion);
            }
        } else if (fileSettingsMetadata != null) {
            mdBuilder.removeReservedState(fileSettingsMetadata);
        }
    }

    private void refreshExistingFileStateIfNeeded(ClusterState clusterState) {
        ReservedStateMetadata fileSettingsMetadata;
        if (this.watching() && (fileSettingsMetadata = clusterState.metadata().reservedStateMetadata().get(NAMESPACE)) != null && fileSettingsMetadata.version() == 0L && Files.exists(this.operatorSettingsFile(), new LinkOption[0])) {
            try {
                Files.setLastModifiedTime(this.operatorSettingsFile(), FileTime.from(Instant.now()));
            }
            catch (IOException e) {
                logger.warn("encountered I/O error trying to update file settings timestamp", (Throwable)e);
            }
        }
    }

    public boolean watching() {
        return this.watcherThread != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    synchronized void startWatcher(ClusterState clusterState) {
        if (this.watching() || !this.active) {
            this.refreshExistingFileStateIfNeeded(clusterState);
            return;
        }
        logger.info("starting file settings watcher ...");
        try {
            Path settingsDirPath = this.operatorSettingsDir();
            this.watchService = settingsDirPath.getParent().getFileSystem().newWatchService();
            if (Files.exists(settingsDirPath, new LinkOption[0])) {
                this.settingsDirWatchKey = this.enableSettingsWatcher(this.settingsDirWatchKey, settingsDirPath);
            } else {
                logger.debug("operator settings directory [{}] not found, will watch for its creation...", (Object)settingsDirPath);
            }
            this.configDirWatchKey = this.enableSettingsWatcher(this.configDirWatchKey, settingsDirPath.getParent());
        }
        catch (Exception e) {
            if (this.watchService != null) {
                try {
                    this.watchService.close();
                }
                catch (Exception ce) {
                    e.addSuppressed(ce);
                }
                finally {
                    this.watchService = null;
                }
            }
            throw new IllegalStateException("unable to launch a new watch service", e);
        }
        this.watcherThread = new Thread(this::watcherThread, "elasticsearch[file-settings-watcher]");
        this.watcherThread.start();
    }

    private void watcherThread() {
        try {
            WatchKey key;
            logger.info("file settings service up and running [tid={}]", (Object)Thread.currentThread().getId());
            Path path = this.operatorSettingsFile();
            if (Files.exists(path, new LinkOption[0])) {
                logger.debug("found initial operator settings file [{}], applying...", (Object)path);
                this.processSettingsAndNotifyListeners();
            } else {
                for (FileSettingsChangedListener listener : this.eventListeners) {
                    listener.settingsChanged();
                }
            }
            while ((key = this.watchService.take()) != null) {
                Path settingsPath = this.operatorSettingsDir();
                if (Files.exists(settingsPath, new LinkOption[0])) {
                    try {
                        if (logger.isDebugEnabled()) {
                            key.pollEvents().forEach(e -> logger.debug("{}:{}", (Object)e.kind().toString(), (Object)e.context().toString()));
                        } else {
                            key.pollEvents();
                        }
                        key.reset();
                        this.settingsDirWatchKey = this.enableSettingsWatcher(this.settingsDirWatchKey, settingsPath);
                        if (!this.watchedFileChanged(path)) continue;
                        this.processSettingsAndNotifyListeners();
                    }
                    catch (IOException e2) {
                        logger.warn("encountered I/O error while watching file settings", (Throwable)e2);
                    }
                    continue;
                }
                key.pollEvents();
                key.reset();
            }
        }
        catch (InterruptedException | ClosedWatchServiceException expected) {
            logger.info("shutting down watcher thread");
        }
        catch (Exception e3) {
            logger.error("shutting down watcher thread with exception", (Throwable)e3);
        }
    }

    void processSettingsAndNotifyListeners() throws InterruptedException {
        try {
            this.processFileSettings(this.operatorSettingsFile()).get();
            for (FileSettingsChangedListener listener : this.eventListeners) {
                listener.settingsChanged();
            }
        }
        catch (ExecutionException e) {
            logger.error("Error processing operator settings json file", e.getCause());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    synchronized void stopWatcher() {
        if (this.watching()) {
            logger.debug("stopping watcher ...");
            try (WatchService ws = this.watchService;){
                this.watcherThread.interrupt();
                this.watcherThread.join();
                if (this.configDirWatchKey != null) {
                    this.configDirWatchKey.cancel();
                }
                if (this.settingsDirWatchKey == null) return;
                this.settingsDirWatchKey.cancel();
                return;
            }
            catch (IOException e) {
                logger.warn("encountered exception while closing watch service", (Throwable)e);
                return;
            }
            catch (InterruptedException interruptedException) {
                logger.info("interrupted while closing the watch service", (Throwable)interruptedException);
                return;
            }
            finally {
                this.watcherThread = null;
                this.settingsDirWatchKey = null;
                this.configDirWatchKey = null;
                this.watchService = null;
                logger.info("watcher service stopped");
            }
        } else {
            logger.trace("file settings service already stopped");
        }
    }

    long retryDelayMillis(int failedCount) {
        assert (failedCount < 31);
        return 100 * (1 << failedCount) + Randomness.get().nextInt(10);
    }

    WatchKey enableSettingsWatcher(WatchKey previousKey, Path settingsDir) throws IOException, InterruptedException {
        if (previousKey != null) {
            previousKey.cancel();
        }
        int retryCount = 0;
        while (true) {
            try {
                return settingsDir.register(this.watchService, StandardWatchEventKinds.ENTRY_MODIFY, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE);
            }
            catch (IOException e) {
                if (retryCount == 4) {
                    throw e;
                }
                Thread.sleep(this.retryDelayMillis(retryCount));
                ++retryCount;
                continue;
            }
            break;
        }
    }

    PlainActionFuture<Void> processFileSettings(Path path) {
        PlainActionFuture<Void> completion = PlainActionFuture.newFuture();
        logger.info("processing path [{}] for [{}]", (Object)path, (Object)NAMESPACE);
        try (InputStream fis = Files.newInputStream(path, new OpenOption[0]);
             BufferedInputStream bis = new BufferedInputStream(fis);
             XContentParser parser = XContentType.JSON.xContent().createParser(XContentParserConfiguration.EMPTY, (InputStream)bis);){
            this.stateService.process(NAMESPACE, parser, e -> this.completeProcessing((Exception)e, completion));
        }
        catch (Exception e2) {
            completion.onFailure(e2);
        }
        return completion;
    }

    private void completeProcessing(Exception e, PlainActionFuture<Void> completion) {
        if (e != null) {
            completion.onFailure(e);
        } else {
            completion.onResponse(null);
        }
    }

    public void addFileSettingsChangedListener(FileSettingsChangedListener listener) {
        this.eventListeners.add(listener);
    }

    record FileUpdateState(long timestamp, String path, Object fileKey) {
    }
}

